/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.saml.xmlobject.impl;

import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.NotThreadSafe;

import org.opensaml.core.xml.AbstractXMLObject;
import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.schema.XSBooleanValue;

import net.shibboleth.idp.saml.xmlobject.Scope;
import net.shibboleth.shared.annotation.constraint.NotLive;
import net.shibboleth.shared.annotation.constraint.Unmodifiable;

/** Implementation of {@link Scope}. */
@NotThreadSafe
public class ScopeImpl extends AbstractXMLObject implements Scope {

    /** The regexp attribute value. */
    @Nullable private XSBooleanValue regexp;

    /** The string content value. */
    @Nullable private String scopeValue;

    /**
     * Constructor.
     * 
     * @param namespaceURI the namespace the element is in
     * @param elementLocalName the local name of the XML element this Object represents
     * @param namespacePrefix the prefix for the given namespace
     */
    protected ScopeImpl(@Nullable final String namespaceURI, @Nonnull final String elementLocalName,
            @Nullable final String namespacePrefix) {
        super(namespaceURI, elementLocalName, namespacePrefix);
    }

    /** {@inheritDoc} */
    @Nullable public Boolean getRegexp() {
        if (regexp != null) {
            return regexp.getValue();
        }
        
        return Boolean.FALSE;
    }

    /** {@inheritDoc} */
    public void setRegexp(@Nullable final Boolean newRegexp) {
        if (newRegexp != null) {
            regexp = prepareForAssignment(regexp, new XSBooleanValue(newRegexp, false));
        } else {
            regexp = prepareForAssignment(regexp, null);
        }
    }

    /** {@inheritDoc} */
    @Nullable public XSBooleanValue getRegexpXSBoolean() {
        return regexp;
    }

    /** {@inheritDoc} */
    public void setRegexp(@Nullable final XSBooleanValue newRegexp) {
        regexp = prepareForAssignment(regexp, newRegexp);
    }

    /** {@inheritDoc} */
    @Nullable public String getValue() {
        return scopeValue;
    }

    /** {@inheritDoc} */
    public void setValue(@Nullable final String newScopeValue) {
        scopeValue = prepareForAssignment(scopeValue, newScopeValue);
    }
    
    /** {@inheritDoc} */
    @Override @Nullable @Unmodifiable @NotLive public List<XMLObject> getOrderedChildren() {
        return null;
    }
    
}