/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.saml.security.impl;

import java.util.List;

import javax.annotation.Nonnull;

import org.opensaml.core.xml.XMLObject;
import org.opensaml.saml.metadata.resolver.filter.FilterException;
import org.opensaml.saml.metadata.resolver.filter.MetadataNodeProcessor;
import org.opensaml.saml.saml2.metadata.EntitiesDescriptor;
import org.opensaml.saml.saml2.metadata.EntityDescriptor;
import org.opensaml.saml.saml2.metadata.Extensions;
import org.opensaml.security.SecurityException;
import org.opensaml.security.x509.PKIXValidationInformation;
import org.slf4j.Logger;

import net.shibboleth.idp.saml.security.KeyAuthoritySupport;
import net.shibboleth.idp.saml.xmlobject.KeyAuthority;
import net.shibboleth.shared.annotation.constraint.NotLive;
import net.shibboleth.shared.annotation.constraint.Unmodifiable;
import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.primitive.LoggerFactory;

/**
 * An implementation of {@link MetadataNodeProcessor} which supports processing the 
 * Shibboleth {@link KeyAuthority} information within a metadata document.
 */
public class KeyAuthorityNodeProcessor implements MetadataNodeProcessor {
    
    /** Logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(KeyAuthority.class);

    /** {@inheritDoc} */
    public void process(@Nonnull final XMLObject metadataNode) throws FilterException {
        if (metadataNode instanceof EntitiesDescriptor ed) {
            handleEntitiesDescriptor(ed);
        } else if (metadataNode instanceof EntityDescriptor ed) {
            handleEntityDescriptor(ed);
        }
    }

    /**
     * Handle an {@link EntitiesDescriptor}.
     * 
     * @param entitiesDescriptor the entities descriptor being processed
     * 
     * @throws FilterException if there is a fatal error during processing
     */
    protected void handleEntitiesDescriptor(@Nonnull final EntitiesDescriptor entitiesDescriptor)
            throws FilterException {
        log.debug("Processing EntitiesDescriptor with id '{}', name '{}'", 
                entitiesDescriptor.getID(), entitiesDescriptor.getName());
        
        final List<XMLObject> keyAuthorities = getKeyAuthorities(entitiesDescriptor);
        if (keyAuthorities.isEmpty()) {
            return;
        }
        
        log.debug("Saw at least one KeyAuthority for EntitiesDescriptor with id '{}', name '{}'", 
                entitiesDescriptor.getID(), entitiesDescriptor.getName());
        
        for (final XMLObject keyAuthority : keyAuthorities) {
            try {
                final PKIXValidationInformation pkixInfo = KeyAuthoritySupport.
                        extractPKIXValidationInfo((KeyAuthority) keyAuthority);
                if (pkixInfo != null) {
                    keyAuthority.getObjectMetadata().put(pkixInfo);
                }
            } catch (final SecurityException e) {
                //TODO should throw here or just log error and continue?  
                throw new FilterException("Error extracting PKIX validation info from KeyAuthority", e);
            }
        }
        
    }

    /**
     * Handle an {@link EntityDescriptor}.
     * 
     * @param entityDescriptor the entity descriptor being processed
     * 
     * @throws FilterException if there is a fatal error during processing
     */
    protected void handleEntityDescriptor(@Nonnull final EntityDescriptor entityDescriptor) throws FilterException {
        XMLObject currentParent = entityDescriptor.getParent();
        while (currentParent != null) {
            if (currentParent instanceof EntitiesDescriptor) {
                for (final XMLObject keyAuthority : getKeyAuthorities((EntitiesDescriptor) currentParent)) {
                    entityDescriptor.getObjectMetadata().putAll(keyAuthority.getObjectMetadata()
                            .get(PKIXValidationInformation.class));
                }
            }
            currentParent = currentParent.getParent();
        }
    }
    
    /**
     * Get the list of KeyAuthority's from an EntitiesDescriptor's Extensions.
     * 
     * @param entitiesDescriptor the entities descriptor to process.
     * @return list of XMLObjects
     */
    @Nonnull @Unmodifiable @NotLive protected List<XMLObject> getKeyAuthorities(
            @Nonnull final EntitiesDescriptor entitiesDescriptor) {
        final Extensions extensions = entitiesDescriptor.getExtensions();
        if (extensions == null) {
            return CollectionSupport.emptyList();
        }
        
        return CollectionSupport.copyToList(extensions.getUnknownXMLObjects(KeyAuthority.DEFAULT_ELEMENT_NAME));
    }
    
}