/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.saml.metadata.impl;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.saml.metadata.resolver.MetadataResolver;
import org.opensaml.saml.metadata.resolver.filter.MetadataFilter;
import org.opensaml.saml.saml2.metadata.EntityDescriptor;
import org.slf4j.Logger;

import net.shibboleth.shared.annotation.ParameterName;
import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.component.AbstractIdentifiableInitializableComponent;
import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.resolver.CriteriaSet;
import net.shibboleth.shared.resolver.ResolverException;
import net.shibboleth.shared.service.ReloadableService;
import net.shibboleth.shared.service.ServiceException;
import net.shibboleth.shared.service.ServiceableComponent;

/**
 * This class uses the service interface to implement {@link MetadataResolver}.
 * 
 */
public class ReloadableMetadataResolver extends AbstractIdentifiableInitializableComponent implements
        MetadataResolver {

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(ReloadableMetadataResolver.class);

    /** The service which managed the reloading. */
    @Nonnull private final ReloadableService<MetadataResolver> service;

    /**
     * Constructor.
     * 
     * @param resolverService the service which will manage the loading.
     */
    public ReloadableMetadataResolver(
            @Nonnull @ParameterName(name="resolverService") final ReloadableService<MetadataResolver> resolverService) {
        service = Constraint.isNotNull(resolverService, "MetadataResolver Service cannot be null");
    }

    /** {@inheritDoc} */
    @Nonnull public Iterable<EntityDescriptor> resolve(@Nullable final CriteriaSet criteria) throws ResolverException {
        checkComponentActive();
        try (final ServiceableComponent<MetadataResolver> component = service.getServiceableComponent()) {
            return component.getComponent().resolve(criteria);
        } catch (final ResolverException e) {
            log.error("ReloadableMetadataResolver '{}': Error during resolution", getId(), e);
        } catch (final ServiceException e) {
            log.error("ReloadableMetadataResolver '{}': Error accessing underlying metadata source: "
                    + "Invalid configuration.", getId(), e);
        }

        return CollectionSupport.emptySet();
    }

    /** {@inheritDoc} */
    @Nullable public EntityDescriptor resolveSingle(@Nullable final CriteriaSet criteria) throws ResolverException {
        checkComponentActive();
        try (final ServiceableComponent<MetadataResolver> component = service.getServiceableComponent()) {
            return component.getComponent().resolveSingle(criteria);
        } catch (final ResolverException e) {
            log.error("ReloadableMetadataResolver '{}': Error during resolution", getId(), e);
        } catch (final ServiceException e) {
            log.error("ReloadableMetadataResolver '{}': Error accessing underlying metadata source: "
                    + "Invalid configuration.", getId(), e);
        }
        return null;
    }

    /** {@inheritDoc} */
    public boolean isRequireValidMetadata() {
        checkComponentActive();
        try (final ServiceableComponent<MetadataResolver> component = service.getServiceableComponent()) {
            return component.getComponent().isRequireValidMetadata();
        } catch (final ServiceException e) {
            log.error("ReloadableMetadataResolver '{}': Error accessing underlying metadata source: "
                    + "Invalid configuration.", getId(), e);
            throw e;
        }
    }

    /** {@inheritDoc} */
    public void setRequireValidMetadata(final boolean requireValidMetadata) {
        throw new IllegalAccessError("Cannot set RequireValidMetadata");
    }

    /** {@inheritDoc} */
    @Nullable public MetadataFilter getMetadataFilter() {
        checkComponentActive();
        try (final ServiceableComponent<MetadataResolver> component = service.getServiceableComponent()) {
            return component.getComponent().getMetadataFilter();
        } catch (final ServiceException e) {
            log.error("ReloadableMetadataResolver '{}': Error accessing underlying metadata source: "
                    + "Invalid configuration.", getId(), e);
            throw e;
        }
    }

    /** {@inheritDoc} */
    public void setMetadataFilter(@Nullable final MetadataFilter newFilter) {
        throw new UnsupportedOperationException("Cannot set Metadata filter");
    }

    /** {@inheritDoc} */
    @Nullable public String getType() {
        return null;
    }
    
}