/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.shibboleth.idp.saml.metadata.impl;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import org.opensaml.saml.metadata.resolver.BatchMetadataResolver;
import org.opensaml.saml.metadata.resolver.ChainingMetadataResolver;
import org.opensaml.saml.metadata.resolver.MetadataResolver;
import org.opensaml.saml.metadata.resolver.RefreshableMetadataResolver;
import org.opensaml.saml.metadata.resolver.RemoteMetadataResolver;
import org.opensaml.saml.metadata.resolver.filter.MetadataFilter;
import org.opensaml.saml.metadata.resolver.filter.MetadataFilterChain;
import org.slf4j.Logger;

import com.codahale.metrics.Gauge;
import com.codahale.metrics.MetricFilter;
import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.MetricSet;
import com.google.common.base.Predicates;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableMap.Builder;

import net.shibboleth.shared.annotation.ParameterName;
import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.annotation.constraint.NotLive;
import net.shibboleth.shared.annotation.constraint.Unmodifiable;
import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.component.ComponentInitializationException;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.resolver.ResolverException;
import net.shibboleth.shared.service.ReloadableServiceGaugeSet;
import net.shibboleth.shared.service.ServiceException;
import net.shibboleth.shared.service.ServiceableComponent;

/**
 * Additional gauges for metadata resolvers.
 */
public class MetadataResolverServiceGaugeSet extends ReloadableServiceGaugeSet<MetadataResolver>
    implements MetricSet, MetricFilter {
    
    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(MetadataResolverServiceGaugeSet.class);

// Checkstyle: MethodLength|AnonInnerLength OFF
    /**
     * Constructor.
     * 
     * @param metricName name to include in metric names produced by this set
     */
    public MetadataResolverServiceGaugeSet(
            @Nonnull @NotEmpty @ParameterName(name="metricName") final String metricName) {
        super(metricName);

        getMetricMap().put(
                MetricRegistry.name(metricName, "update"),
                new Gauge<Map<String,Instant>>() {
                    public Map<String,Instant> getValue() {
                        return valueGetter(new BiConsumer<Builder<String,Instant>, MetadataResolver>() {
                            public void accept(final Builder<String,Instant> mapBuilder,
                                    final MetadataResolver resolver) {
                                if (resolver instanceof RefreshableMetadataResolver
                                        && ((RefreshableMetadataResolver) resolver).getLastUpdate() != null) {
                                    mapBuilder.put(resolver.getId(),
                                            ((RefreshableMetadataResolver) resolver).getLastUpdate());
                                }
                            };
                        });
                    }
                });
        
        getMetricMap().put(
                MetricRegistry.name(metricName, "refresh"),
                new Gauge<Map<String,Instant>>() {
                    public Map<String,Instant> getValue() {
                        return valueGetter(new BiConsumer<Builder<String,Instant>, MetadataResolver>() {
                            public void accept(final Builder<String,Instant> mapBuilder,
                                    final MetadataResolver resolver) {
                                if (resolver instanceof RefreshableMetadataResolver
                                        && ((RefreshableMetadataResolver) resolver).getLastRefresh() != null) {
                                    mapBuilder.put(resolver.getId(),
                                            ((RefreshableMetadataResolver) resolver).getLastRefresh());
                                }
                            };
                        });
                    }
                });
                
        getMetricMap().put(
                MetricRegistry.name(metricName, "successfulRefresh"),
                new Gauge<Map<String,Instant>>() {
                    public Map<String,Instant> getValue() {
                        return valueGetter(new BiConsumer<Builder<String,Instant>, MetadataResolver>() {
                            public void accept(final Builder<String,Instant> mapBuilder,
                                    final MetadataResolver resolver) {
                                if (resolver instanceof RefreshableMetadataResolver
                                        && ((RefreshableMetadataResolver) resolver)
                                            .getLastSuccessfulRefresh()  != null) {
                                    mapBuilder.put(resolver.getId(),
                                            ((RefreshableMetadataResolver) resolver).getLastSuccessfulRefresh());
                                }
                            };
                        });
                    }
                });
        
        getMetricMap().put(
                MetricRegistry.name(metricName, "error"),
                new Gauge<Map<String,String>>() {
                    public Map<String,String> getValue() {
                        return valueGetter(new BiConsumer<Builder<String,String>, MetadataResolver>() {
                            public void accept(final Builder<String,String> mapBuilder,
                                    final MetadataResolver resolver) {
                                if (resolver instanceof RefreshableMetadataResolver
                                        && ((RefreshableMetadataResolver) resolver)
                                            .getLastFailureCause() != null) {
                                    mapBuilder.put(resolver.getId(),
                                            extractErrorMessage(
                                                    ((RefreshableMetadataResolver) resolver).getLastFailureCause()));
                                }
                            };
                        });
                    }
                });

        getMetricMap().put(
                MetricRegistry.name(metricName, "rootValidUntil"),
                new Gauge<Map<String,Instant>>() {
                    public Map<String,Instant> getValue() {
                        return valueGetter(new BiConsumer<Builder<String,Instant>, MetadataResolver>() {
                            public void accept(final Builder<String,Instant> mapBuilder,
                                    final MetadataResolver resolver) {
                                if (resolver instanceof BatchMetadataResolver
                                        && ((BatchMetadataResolver) resolver).getRootValidUntil() != null) {
                                    mapBuilder.put(resolver.getId(),
                                            ((BatchMetadataResolver) resolver).getRootValidUntil());
                                }
                            };
                        });
                    }
                });

        getMetricMap().put(
                MetricRegistry.name(metricName, "type"),
                new Gauge<Map<String,String>>() {
                    public Map<String,String> getValue() {
                        return valueGetter(new BiConsumer<Builder<String,String>, MetadataResolver>() {
                            public void accept(final Builder<String,String> mapBuilder,
                                    final MetadataResolver resolver) {
                                if (resolver.getType() != null) {
                                    mapBuilder.put(resolver.getId(), resolver.getType());
                                }
                            };
                        });
                    }
                });

        getMetricMap().put(
                MetricRegistry.name(metricName, "uri"),
                new Gauge<Map<String,String>>() {
                    public Map<String,String> getValue() {
                        return valueGetter(new BiConsumer<Builder<String,String>, MetadataResolver>() {
                            public void accept(final Builder<String,String> mapBuilder,
                                    final MetadataResolver resolver) {
                                if (resolver instanceof RemoteMetadataResolver remote) {
                                    final String uri = remote.getMetadataURI();
                                    if (uri != null) {
                                        mapBuilder.put(resolver.getId(), remote.getMetadataURI());
                                    }
                                }
                            };
                        });
                    }
                });

        getMetricMap().put(
                MetricRegistry.name(metricName, "filters"),
                new Gauge<Map<String,Collection<String>>>() {
                    public Map<String,Collection<String>> getValue() {
                        return valueGetter(new BiConsumer<Builder<String,Collection<String>>, MetadataResolver>() {
                            public void accept(final Builder<String,Collection<String>> mapBuilder,
                                    final MetadataResolver resolver) {
                                final MetadataFilter filter = resolver.getMetadataFilter();
                                if (filter instanceof MetadataFilterChain chaining) {
                                    mapBuilder.put(resolver.getId(),
                                            chaining.getFilters().stream()
                                                .map(MetadataFilter::getType)
                                                .filter(Predicates.notNull())
                                                .collect(Collectors.toUnmodifiableList()));
                                } else if (filter != null) {
                                    final String type = filter.getType();
                                    if (type != null) {
                                        mapBuilder.put(resolver.getId(), CollectionSupport.singletonList(type));
                                    }
                                }
                            };
                        });
                    }
                });
    }
// Checkstyle: MethodLength|AnonInnerLength ON

    /**
     * Extract the error message to report out.
     *
     * @param t the throwable to process
     *
     * @return the error message string to report out
     */
    @Nonnull private String extractErrorMessage(final Throwable t) {
        Throwable source = null;

        // These are often wrapping the real error, so use the cause as the source if available
        if (ResolverException.class.isInstance(t) && t.getCause() != null) {
            source = t.getCause();
        } else {
            source = t;
        }

        if (source.getMessage() != null) {
            return source.getClass().getName() + ": " + source.getMessage();
        }

        Throwable cause = source.getCause();
        while (cause != null) {
            if (cause.getMessage() != null) {
                return cause.getClass().getName() + ": " + cause.getMessage();
            }
            cause = cause.getCause();
        }
        return source.getClass().getName() + ": <Detailed error message not specified>";
    }

    /**
     * Helper Function for map construction.
     * 
     * <p>
     * This does all the service handling and just calls the specific {@link BiConsumer} to
     * add each appropriate the value to the map.
     * </p>
     * 
     * @param <T> the type of value being reported out
     * @param consume the thing which does checking and adding the building
     * @return an appropriate map
     */
    @Nonnull private <T> Map<String,T> valueGetter(
            @Nonnull final BiConsumer<Builder<String,T>, MetadataResolver> consume) {
        final Builder<String,T> mapBuilder = ImmutableMap.builder();
        try (final ServiceableComponent<?> component = getService().getServiceableComponent()) {
            // Check type - just in case
            if (!(component.getComponent() instanceof MetadataResolver)) {
                log.warn("{} : Injected Service was not for an Metadata Resolver : ({}) ",
                        getLogPrefix(), component.getComponent().getClass());
            } else {
                for (final MetadataResolver resolver : getMetadataResolvers(
                        (MetadataResolver) component.getComponent())) {
                    consume.accept(mapBuilder, resolver);
                }
            }
        } catch (final ServiceException e) {
            // Nothing to do.
        }
        return mapBuilder.build();
    }

    
    /** {@inheritDoc} */
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();

        try (final ServiceableComponent<?> component = getService().getServiceableComponent()) {
            if (component.getComponent() instanceof MetadataResolver) {
                return;
            }
            log.error("{} : Injected service was not for a MetadataResolver ({}) ",
                    getLogPrefix(), component.getClass());
            throw new ComponentInitializationException("Injected service was not for a MetadataResolver");
        } catch (final ServiceException e) {
            log.debug("{} : Injected service has not initialized sucessfully yet. Skipping type test",
                    getLogPrefix(), e);
        }
    }

    /** Get all the resolvers rooted in the provider tree (including the root).
     * @param parent - root of the chaining resolver tree.
     * @return - the list.
     */
    @Nonnull @Unmodifiable @NotLive private List<MetadataResolver> getAllChildren(
            @Nonnull final ChainingMetadataResolver parent) {
        final ArrayList<MetadataResolver> result = new ArrayList<>(parent.getResolvers().size());
        
        for (final MetadataResolver child: parent.getResolvers()) {
            if (child instanceof ChainingMetadataResolver) {
                result.addAll(getAllChildren((ChainingMetadataResolver) child));
            } else {
                result.add(child);
            }
        }
        
        return CollectionSupport.copyToList(result);
    }

    /**
     * Return the resolvers to report on.
     * 
     * @param rootResolver root component
     * 
     * @return resolvers to report on
     */
    @Nonnull @Unmodifiable @NotLive private Iterable<MetadataResolver> getMetadataResolvers(
            @Nonnull final MetadataResolver rootResolver) {
        
        if (rootResolver instanceof ChainingMetadataResolver) {
            return getAllChildren((ChainingMetadataResolver) rootResolver);
        }
        return CollectionSupport.singletonList(rootResolver);
    }

}