/*
 * SonarQube Scanner for Jenkins
 * Copyright (C) 2007-2025 SonarSource Sàrl
 * mailto:info AT sonarsource DOT com
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package hudson.plugins.sonar.model;

import edu.umd.cs.findbugs.annotations.CheckForNull;
import hudson.EnvVars;
import hudson.Util;
import hudson.model.AbstractBuild;
import hudson.model.BuildListener;
import hudson.model.Cause;
import hudson.model.Result;
import hudson.plugins.sonar.Messages;
import hudson.triggers.SCMTrigger;
import hudson.util.VariableResolver;
import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import org.kohsuke.stapler.DataBoundConstructor;

/**
 * @author Evgeny Mandrikov
 * @since 1.2
 */
public class TriggersConfig implements Serializable {

  private static final long serialVersionUID = 1L;

  private boolean skipScmCause;

  private boolean skipUpstreamCause;

  /**
   * @since 1.7
   */
  private String envVar;

  public TriggersConfig() {
    skipScmCause = false;
    skipUpstreamCause = false;
    envVar = null;
  }

  @DataBoundConstructor
  public TriggersConfig(boolean skipScmCause, boolean skipUpstreamCause, @CheckForNull String envVar) {
    this.skipScmCause = skipScmCause;
    this.skipUpstreamCause = skipUpstreamCause;
    this.envVar = envVar;
  }

  public boolean isSkipScmCause() {
    return skipScmCause;
  }

  public void setSkipScmCause(boolean b) {
    this.skipScmCause = b;
  }

  public boolean isSkipUpstreamCause() {
    return skipUpstreamCause;
  }

  public void setSkipUpstreamCause(boolean b) {
    this.skipUpstreamCause = b;
  }

  public String getEnvVar() {
    return Util.fixEmptyAndTrim(envVar);
  }

  public void setEnvVar(String envVar) {
    this.envVar = envVar;
  }

  public String isSkipSonar(AbstractBuild<?, ?> build, BuildListener listener) throws IOException, InterruptedException {
    Result result = build.getResult();

    if (result != null && result.isWorseThan(Result.UNSTABLE)) {
      // skip analysis if build failed
      // unstable means that build completed, but there were some test failures, which is not critical for analysis
      return Messages.SonarPublisher_BadBuildStatus(result.toString());
    }

    // skip analysis by environment variable or build parameter
    if (getEnvVar() != null) {
      // check against build parameters
      String value = build.getBuildVariableResolver().resolve(getEnvVar());
      if ("true".equalsIgnoreCase(value)) {
        return Messages.Skipping_Sonar_analysis();
      }
      // check against environment variable
      EnvVars envVars = build.getEnvironment(listener);
      value = new VariableResolver.ByMap<>(envVars).resolve(getEnvVar());
      if ("true".equalsIgnoreCase(value)) {
        return Messages.Skipping_Sonar_analysis();
      }
    }

    List<Cause> causes = new ArrayList<>(build.getCauses());

    // skip analysis, when all causes from blacklist
    if (isSkipScmCause() || isSkipUpstreamCause()) {
      Iterator<Cause> iter = causes.iterator();
      while (iter.hasNext()) {
        Cause cause = iter.next();
        if ((isSkipScmCause() && SCMTrigger.SCMTriggerCause.class.isInstance(cause)) || (isSkipUpstreamCause() && Cause.UpstreamCause.class.isInstance(cause))) {
          iter.remove();
        }
      }
    }
    return causes.isEmpty() ? Messages.Skipping_Sonar_analysis() : null;
  }

  /**
   * For internal use only.
   */
  public static class SonarCause extends Cause {
    @Override
    public String getShortDescription() {
      return null;
    }
  }
}
