package org.jenkinsci.plugins.vines;

import java.net.ConnectException;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;
import javax.net.ssl.SSLHandshakeException;

import hudson.Util;
import java.net.URI;
import java.net.URISyntaxException;
import com.fasterxml.jackson.databind.JsonNode;
import hudson.EnvVars;
import hudson.Extension;
import hudson.FilePath;
import hudson.Launcher;
import hudson.model.Item;
import hudson.model.Run;
import hudson.model.TaskListener;
import hudson.security.ACL;
import hudson.tasks.ArtifactArchiver;
import hudson.tasks.BuildStepDescriptor;
import hudson.tasks.Builder;
import hudson.util.FormValidation;
import jenkins.model.Jenkins;
import jenkins.tasks.SimpleBuildStep;
import org.jenkinsci.Symbol;
import org.jenkinsci.plugins.plaincredentials.StringCredentials;
import org.kohsuke.stapler.AncestorInPath;
import org.kohsuke.stapler.DataBoundConstructor;
import org.kohsuke.stapler.QueryParameter;
import org.kohsuke.stapler.verb.POST;
import com.cloudbees.plugins.credentials.common.StandardListBoxModel;
import com.cloudbees.plugins.credentials.domains.DomainRequirement;
import hudson.AbortException;

import hudson.ProxyConfiguration;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;

import java.io.IOException;
import java.io.OutputStream;
import java.time.Duration;
import java.util.Collections;

public class VinesScanBuilder extends Builder implements SimpleBuildStep {

    private final String credentialsId;
    private final String targetUrl;
    private final int pollingIntervalSec;
    private final int timeoutMinutes;

    private final double cvssCutoff;
    private final int maxCritical;
    private final int maxHigh;
    private final int maxMedium;
    private final boolean newOnly;

    @DataBoundConstructor
    public VinesScanBuilder(String credentialsId, String targetUrl,
                            int pollingIntervalSec, int timeoutMinutes,
                            double cvssCutoff, int maxCritical, int maxHigh, int maxMedium, boolean newOnly) {
        this.credentialsId = credentialsId;
        this.targetUrl = targetUrl;
        this.pollingIntervalSec = pollingIntervalSec <= 0 ? 6 : pollingIntervalSec;
        this.timeoutMinutes = timeoutMinutes <= 0 ? 60 : timeoutMinutes;
        this.cvssCutoff = cvssCutoff;
        this.maxCritical = Math.max(0, maxCritical);
        this.maxHigh = Math.max(0, maxHigh);
        this.maxMedium = Math.max(0, maxMedium);
        this.newOnly = newOnly; // note: KPI provides totals; “newOnly” doesn’t apply to KPI
    }

    public String getCredentialsId() { return credentialsId; }
    public String getTargetUrl() { return targetUrl; }
    public int getPollingIntervalSec() { return pollingIntervalSec; }
    public int getTimeoutMinutes() { return timeoutMinutes; }
    public double getCvssCutoff() { return cvssCutoff; }
    public int getMaxCritical() { return maxCritical; }
    public int getMaxHigh() { return maxHigh; }
    public int getMaxMedium() { return maxMedium; }
    public boolean isNewOnly() { return newOnly; }

    @Override
    public void perform(Run<?, ?> run, FilePath workspace, EnvVars env, Launcher launcher, TaskListener listener)
            throws InterruptedException, IOException {

        // 1) Syntax validation (URI)
        String urlErr = validateUrl(targetUrl);
        if (urlErr != null) throw new AbortException("[Vines] Invalid target URL: " + urlErr);

        // 2) Reachability preflight (HEAD via Jenkins proxy; fallback tiny GET)
        try {
            URI u = new URI(targetUrl);
            String reachErr = preflightReachable(u);
            if (reachErr != null) throw new AbortException("[Vines] Target not reachable: " + reachErr);
        } catch (URISyntaxException e) {
            throw new AbortException("[Vines] Invalid URI");
        }

        // 3) Start scan
        final VinesClient client = new VinesClient(run, listener, credentialsId);
        listener.getLogger().println("[Vines] Starting scan…");
        String scanId;
        try {
            scanId = client.startScan(targetUrl);
        } catch (Exception e) {
            String m = e.getMessage();
            throw new AbortException("[Vines] Failed to start scan: " + (m == null ? e.getClass().getSimpleName() : m));
        }
        if (scanId == null || scanId.isEmpty()) throw new AbortException("[Vines] Missing scan id");
        listener.getLogger().println("[Vines] Scan started: " + scanId);

        // 4) Poll status until finished
        long deadline = System.currentTimeMillis() + Duration.ofMinutes(timeoutMinutes).toMillis();
        JsonNode finalStatus = null;
        while (System.currentTimeMillis() < deadline) {
            JsonNode status;
            try { status = client.getStatus(scanId); }
            catch (Exception ex) { listener.getLogger().println("[Vines] Status fetch failed: " + ex.getMessage()); status = null; }
            if (status != null) {
                String state = status.path("status").asText();
                listener.getLogger().println("[Vines] State: " + state);
                if ("finished".equalsIgnoreCase(state) || "completed".equalsIgnoreCase(state)) { finalStatus = status; break; }
                if ("error".equalsIgnoreCase(state) || "failed".equalsIgnoreCase(state))  throw new AbortException("[Vines] Scan error state");
            }
            Thread.sleep(pollingIntervalSec * 1000L);
        }
        if (finalStatus == null) throw new AbortException("[Vines] Timeout waiting for scan");

        // 5) Fetch KPI (authoritative counts + cvss.max). KPI is mandatory for gating here.
        JsonNode kpi;
        try {
            kpi = client.getKpi(scanId);
        } catch (IOException | InterruptedException e) {
            if (e instanceof InterruptedException) Thread.currentThread().interrupt();
            String m = e.getMessage();
            throw new AbortException("[Vines] KPI fetch failed: " + (m == null ? e.getClass().getSimpleName() : m));
        }
        if (kpi == null) throw new AbortException("[Vines] KPI not available");

        // Parse KPI exactly as provided:
        JsonNode sum = dig(kpi, "risk_scores", "summary"); // authoritative
        JsonNode totals = kpi.path("totals");
        int critical = (sum.has("critical") ? sum.path("critical").asInt(0) : totals.path("critical").asInt(0));
        int high     = (sum.has("high")     ? sum.path("high").asInt(0)     : totals.path("high").asInt(0));
        int medium   = (sum.has("medium")   ? sum.path("medium").asInt(0)   : totals.path("medium").asInt(0));
        double maxCvss = dig(kpi, "cvss", "max").asDouble(0.0);

        listener.getLogger().printf("[Vines] Findings (kpi): critical=%d, high=%d, medium=%d; max_cvss=%.2f%n",
                critical, high, medium, maxCvss);

        // 6) Gates
        boolean fail = false;
        if (maxCritical >= 0 && critical > maxCritical) fail = true;
        if (maxHigh     >= 0 && high     > maxHigh)     fail = true;
        if (maxMedium   >= 0 && medium   > maxMedium)   fail = true;
        if (cvssCutoff  >  0 && maxCvss >= cvssCutoff)  fail = true;
        if (fail) throw new AbortException("[Vines] Quality gates failed");

        // 7) Archive report (best-effort)
        try {
            byte[] pdf = client.fetchPdf(scanId);
            if (pdf != null && pdf.length > 0) {
                FilePath out = new FilePath(workspace, "vines-report.pdf");
                try (OutputStream os = out.write()) { os.write(pdf); }
                new ArtifactArchiver("vines-report.pdf").perform(run, workspace, launcher, listener);
                listener.getLogger().println("[Vines] Report archived: vines-report.pdf");
            } else {
                listener.getLogger().println("[Vines] PDF report not available.");
            }
        } catch (Exception e) {
            listener.getLogger().println("[Vines] PDF download failed: " + e.getMessage());
        }
    }

    // ---------- helpers ----------

    public static JsonNode dig(JsonNode root, String... keys) {
        if (root == null) return null;
        JsonNode cur = root;
        for (String k : keys) {
            if (cur == null) return null;
            cur = cur.path(k);
        }
        return cur;
    }

    // ---- shared URL syntax validation (no network) ----
    private static String validateUrl(String value) {
        String v = Util.fixEmptyAndTrim(value);
        if (v == null) return "Target URL is required";
        try {
            URI u = new URI(v);
            String scheme = u.getScheme();
            if (scheme == null || !(scheme.equalsIgnoreCase("http") || scheme.equalsIgnoreCase("https"))) {
                return "Must start with http:// or https://";
            }
            if (u.getHost() == null || u.getHost().isBlank()) {
                return "Host is missing";
            }
            return null;
        } catch (URISyntaxException e) {
            return "Invalid URL: " + e.getMessage();
        }
    }

    // ---- lightweight reachability check (HEAD with proxy; falls back to tiny GET) ----
    private static String preflightReachable(URI u) {
        try {
            HttpClient client = ProxyConfiguration.newHttpClientBuilder()
                    .connectTimeout(Duration.ofSeconds(5))
                    .build();

            HttpRequest head = ProxyConfiguration.newHttpRequestBuilder(u)
                    .method("HEAD", HttpRequest.BodyPublishers.noBody())
                    .timeout(Duration.ofSeconds(8))
                    .header("User-Agent", "VinesAI-Jenkins-Preflight/1.0")
                    .build();

            HttpResponse<Void> r = client.send(head, HttpResponse.BodyHandlers.discarding());
            return acceptStatus(r.statusCode());
        } catch (IOException | InterruptedException headErr) {
            if (headErr instanceof InterruptedException) Thread.currentThread().interrupt();
            if (isTerminalNetworkError(headErr)) {
                return describeNetworkError(headErr, u);
            }
            try {
                HttpClient client = ProxyConfiguration.newHttpClientBuilder()
                        .connectTimeout(Duration.ofSeconds(5))
                        .build();

                HttpRequest get = ProxyConfiguration.newHttpRequestBuilder(u)
                        .GET()
                        .timeout(Duration.ofSeconds(8))
                        .header("User-Agent", "VinesAI-Jenkins-Preflight/1.0")
                        .header("Range", "bytes=0-0")
                        .build();

                HttpResponse<Void> r = client.send(get, HttpResponse.BodyHandlers.discarding());
                return acceptStatus(r.statusCode());
            } catch (IOException | InterruptedException getErr) {
                if (getErr instanceof InterruptedException) Thread.currentThread().interrupt();
                return describeNetworkError(getErr, u);
            }
        }
    }

    private static String acceptStatus(int sc) {
        return (sc >= 100 && sc < 600) ? null : "HTTP status " + sc;
    }

    private static boolean isTerminalNetworkError(Exception e) {
        return e instanceof UnknownHostException
            || e instanceof ConnectException
            || e instanceof SocketTimeoutException
            || e instanceof SSLHandshakeException;
    }

    private static String describeNetworkError(Exception e, URI u) {
        String host = u.getHost();
        int port = (u.getPort() > 0) ? u.getPort()
                : ("https".equalsIgnoreCase(u.getScheme()) ? 443 : 80);

        if (e instanceof UnknownHostException) {
            return "Unknown host: " + host + " (DNS lookup failed)";
        }
        if (e instanceof ConnectException) {
            String msg = e.getMessage();
            return "Connection failed to " + host + ":" + port + (msg != null && !msg.isBlank() ? " (" + msg + ")" : "");
        }
        if (e instanceof SocketTimeoutException) {
            return "Timed out connecting to " + host + ":" + port;
        }
        if (e instanceof SSLHandshakeException) {
            String msg = e.getMessage();
            return "TLS handshake failed with " + host + (msg != null && !msg.isBlank() ? ": " + msg : "");
        }
        String msg = e.getMessage();
        return (msg == null || msg.isBlank()) ? e.getClass().getSimpleName() : msg;
    }

    @Extension
    @Symbol("vinesScan")
    public static final class DescriptorImpl extends BuildStepDescriptor<Builder> {

        @Override public boolean isApplicable(Class c) { return true; }
        @Override public String getDisplayName() { return "Vines AI DAST Scan"; }

        @POST
        @SuppressWarnings("lgtm[jenkins/no-permission-check]")
        public FormValidation doCheckTargetUrl(@QueryParameter String value) {
            String err = validateUrl(value); // syntax-only, no network I/O
            return err == null ? FormValidation.ok() : FormValidation.error(err);
        }



        /** Fill the credentials dropdown with Secret Text credentials (System / Folder / Job context). */
        @POST
        public StandardListBoxModel doFillCredentialsIdItems(
                @AncestorInPath Item item,
                @QueryParameter String credentialsId) {

            // Permission checks (unchanged)
            if (item == null) {
                if (!Jenkins.get().hasPermission(Jenkins.ADMINISTER)) {
                    StandardListBoxModel empty = new StandardListBoxModel();
                    empty.includeEmptyValue();
                    return empty;
                }
            } else {
                if (!item.hasPermission(Item.EXTENDED_READ)
                        && !item.hasPermission(com.cloudbees.plugins.credentials.CredentialsProvider.USE_ITEM)) {
                    StandardListBoxModel empty = new StandardListBoxModel();
                    empty.includeEmptyValue();
                    return empty;
                }
            }

            StandardListBoxModel m = new StandardListBoxModel();
            m.includeEmptyValue();
            m.includeAs(ACL.SYSTEM, item, StringCredentials.class, Collections.<DomainRequirement>emptyList());
            m.includeCurrentValue(credentialsId);
            return m;
        }

    }
}
