package org.jenkinsci.plugins.vines;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import hudson.ProxyConfiguration;
import hudson.model.Run;
import hudson.model.TaskListener;
import hudson.util.Secret;
import org.jenkinsci.plugins.plaincredentials.StringCredentials;
import com.cloudbees.plugins.credentials.CredentialsProvider;
import com.cloudbees.plugins.credentials.domains.DomainRequirement;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.Collections;

public class VinesClient {
    private static final String BASE_URL = "https://vines.rosebird.org";

    private final Run<?,?> run;
    private final TaskListener listener;
    private Secret token;
    private final HttpClient http;
    private final ObjectMapper om = new ObjectMapper();

    public VinesClient(Run<?,?> run, TaskListener listener, String credentialsId) {
        this.run = run;
        this.listener = listener;
        this.token = resolveToken(credentialsId);
        this.http = ProxyConfiguration.newHttpClientBuilder()
                .connectTimeout(Duration.ofSeconds(30))
                .build();

        if (this.token == null || this.token.getPlainText().isBlank()) {
            String env = System.getenv("VINES_TOKEN");
            if (env != null && !env.isBlank()) {
                listener.getLogger().println("[Vines] Using VINES_TOKEN from environment.");
                this.token = Secret.fromString(env);
            } else {
                throw new IllegalStateException("[Vines] API token not found (Credentials or VINES_TOKEN required).");
            }
        }
    }

    private Secret resolveToken(String credentialsId) {
        if (credentialsId == null || credentialsId.isBlank()) return null;
        StringCredentials c = CredentialsProvider.findCredentialById(
                credentialsId, StringCredentials.class, run, Collections.<DomainRequirement>emptyList()
        );
        if (c != null && c.getSecret() != null) return c.getSecret();
        return null;
    }

    public String startScan(String targetUrl) throws IOException, InterruptedException {
        String url = BASE_URL + "/api/v1/scans";
        String bodyJson = "{\"target\":\"" + escape(targetUrl) + "\"}";
        HttpRequest req = ProxyConfiguration.newHttpRequestBuilder(URI.create(url))
                .header("Authorization", "Bearer " + token.getPlainText())
                .header("Content-Type", "application/json")
                .timeout(Duration.ofMinutes(2))
                .POST(HttpRequest.BodyPublishers.ofString(bodyJson, StandardCharsets.UTF_8))
                .build();
        HttpResponse<byte[]> resp = http.send(req, HttpResponse.BodyHandlers.ofByteArray());
        if (resp.statusCode() / 100 != 2) throw new IOException("HTTP " + resp.statusCode());
        JsonNode node = om.readTree(resp.body());
        String id = node.path("scan_id").asText(null);
        if (id == null || id.isBlank()) id = node.path("id").asText(null);
        if (id == null || id.isBlank()) {
            JsonNode alt = node.path("data").path("scan_id");
            if (!alt.isMissingNode()) id = alt.asText();
        }
        return id;
    }

    public JsonNode getStatus(String scanId) throws IOException, InterruptedException {
        String url = BASE_URL + "/api/v1/scan/" + scanId + "/status";
        HttpRequest req = ProxyConfiguration.newHttpRequestBuilder(URI.create(url))
                .header("Authorization", "Bearer " + token.getPlainText())
                .timeout(Duration.ofSeconds(30))
                .GET().build();
        HttpResponse<byte[]> resp = http.send(req, HttpResponse.BodyHandlers.ofByteArray());
        if (resp.statusCode() / 100 != 2) throw new IOException("HTTP " + resp.statusCode());
        return om.readTree(resp.body());
    }

    /** Definitive KPI endpoint with counts and max CVSS. */
    public JsonNode getKpi(String scanId) throws IOException, InterruptedException {
        String url = BASE_URL + "/api/v1/scan/" + scanId + "/kpi";
        HttpRequest req = ProxyConfiguration.newHttpRequestBuilder(URI.create(url))
                .header("Authorization", "Bearer " + token.getPlainText())
                .timeout(Duration.ofSeconds(30))
                .GET().build();
        HttpResponse<byte[]> resp = http.send(req, HttpResponse.BodyHandlers.ofByteArray());
        if (resp.statusCode() / 100 != 2) throw new IOException("HTTP " + resp.statusCode());
        return om.readTree(resp.body());
    }

    /** Legacy fallbacks (optional). */
    public JsonNode getSummary(String scanId) throws IOException, InterruptedException {
        String[] templates = new String[] {
                BASE_URL + "/api/v1/scan/%s/summary",
                BASE_URL + "/api/v1/scan/%s",
                BASE_URL + "/api/v1/scan/%s/status?details=1"
        };
        for (String t : templates) {
            String url = String.format(t, scanId);
            HttpRequest req = ProxyConfiguration.newHttpRequestBuilder(URI.create(url))
                    .header("Authorization", "Bearer " + token.getPlainText())
                    .timeout(Duration.ofSeconds(30))
                    .GET().build();
            HttpResponse<byte[]> resp = http.send(req, HttpResponse.BodyHandlers.ofByteArray());
            if (resp.statusCode() / 100 == 2) {
                try { return om.readTree(resp.body()); } catch (Exception ignore) { }
            }
        }
        return null;
    }

    public byte[] fetchPdf(String scanId) throws IOException, InterruptedException {
        String url = BASE_URL + "/api/v1/report/" + scanId + "/pdf";
        HttpRequest req = ProxyConfiguration.newHttpRequestBuilder(URI.create(url))
                .header("Authorization", "Bearer " + token.getPlainText())
                .timeout(Duration.ofMinutes(2))
                .GET().build();
        HttpResponse<byte[]> resp = http.send(req, HttpResponse.BodyHandlers.ofByteArray());
        if (resp.statusCode() / 100 != 2) throw new IOException("HTTP " + resp.statusCode());
        return resp.body();
    }

    private static String escape(String s) {
        return s.replace("\\", "\\\\").replace("\"", "\\\"");
    }
}
