/*
 * Decompiled with CFR 0.152.
 */
package io.jenkins.plugins.swarmcloud.ratelimit;

import edu.umd.cs.findbugs.annotations.NonNull;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Level;
import java.util.logging.Logger;

public class ProvisionRateLimiter {
    private static final Logger LOGGER = Logger.getLogger(ProvisionRateLimiter.class.getName());
    private static final ConcurrentHashMap<String, RateLimitState> CLOUD_STATES = new ConcurrentHashMap();
    public static final int DEFAULT_MAX_PROVISIONS_PER_MINUTE = 10;
    public static final long DEFAULT_MIN_INTERVAL_MS = 1000L;
    public static final long FAILURE_COOLDOWN_MS = TimeUnit.SECONDS.toMillis(30L);

    public static boolean canProvision(@NonNull String cloudName) {
        return ProvisionRateLimiter.canProvision(cloudName, 10, 1000L);
    }

    public static boolean canProvision(@NonNull String cloudName, int maxPerMinute, long minIntervalMs) {
        RateLimitState state = CLOUD_STATES.computeIfAbsent(cloudName, k -> new RateLimitState());
        return state.canProvision(maxPerMinute, minIntervalMs);
    }

    public static void recordProvision(@NonNull String cloudName) {
        RateLimitState state = CLOUD_STATES.computeIfAbsent(cloudName, k -> new RateLimitState());
        state.recordProvision();
        LOGGER.log(Level.FINE, "Recorded provision for cloud: {0}, count this minute: {1}", new Object[]{cloudName, state.getProvisionCountInWindow()});
    }

    public static void recordFailure(@NonNull String cloudName) {
        RateLimitState state = CLOUD_STATES.computeIfAbsent(cloudName, k -> new RateLimitState());
        state.recordFailure();
        LOGGER.log(Level.WARNING, "Recorded provision failure for cloud: {0}, consecutive failures: {1}", new Object[]{cloudName, state.getConsecutiveFailures()});
    }

    public static void resetFailures(@NonNull String cloudName) {
        RateLimitState state = CLOUD_STATES.get(cloudName);
        if (state != null) {
            state.resetFailures();
        }
    }

    public static RateLimitInfo getInfo(@NonNull String cloudName) {
        RateLimitState state = CLOUD_STATES.get(cloudName);
        if (state == null) {
            return new RateLimitInfo(0, 0, 0L, true);
        }
        return new RateLimitInfo(state.getProvisionCountInWindow(), state.getConsecutiveFailures(), state.getLastProvisionTime(), state.canProvision(10, 1000L));
    }

    public static long getWaitTime(@NonNull String cloudName) {
        return ProvisionRateLimiter.getWaitTime(cloudName, 10, 1000L);
    }

    public static long getWaitTime(@NonNull String cloudName, int maxPerMinute, long minIntervalMs) {
        RateLimitState state = CLOUD_STATES.get(cloudName);
        if (state == null) {
            return 0L;
        }
        return state.getWaitTime(maxPerMinute, minIntervalMs);
    }

    public static void clearAll() {
        CLOUD_STATES.clear();
    }

    private static class RateLimitState {
        private final AtomicLong lastProvisionTime = new AtomicLong(0L);
        private final AtomicLong windowStart = new AtomicLong(System.currentTimeMillis());
        private final AtomicInteger windowCount = new AtomicInteger(0);
        private final AtomicInteger consecutiveFailures = new AtomicInteger(0);
        private final AtomicLong lastFailureTime = new AtomicLong(0L);

        private RateLimitState() {
        }

        boolean canProvision(int maxPerMinute, long minIntervalMs) {
            long cooldown;
            long now = System.currentTimeMillis();
            long lastFail = this.lastFailureTime.get();
            int failures = this.consecutiveFailures.get();
            if (failures > 0 && now - lastFail < (cooldown = FAILURE_COOLDOWN_MS * (long)Math.min(failures, 5))) {
                LOGGER.log(Level.FINE, "In failure cooldown, remaining: {0}ms", cooldown - (now - lastFail));
                return false;
            }
            long lastProv = this.lastProvisionTime.get();
            if (lastProv > 0L && now - lastProv < minIntervalMs) {
                return false;
            }
            long windowAge = now - this.windowStart.get();
            if (windowAge >= TimeUnit.MINUTES.toMillis(1L)) {
                this.windowStart.set(now);
                this.windowCount.set(0);
            }
            return this.windowCount.get() < maxPerMinute;
        }

        void recordProvision() {
            long now = System.currentTimeMillis();
            this.lastProvisionTime.set(now);
            if (now - this.windowStart.get() >= TimeUnit.MINUTES.toMillis(1L)) {
                this.windowStart.set(now);
                this.windowCount.set(0);
            }
            this.windowCount.incrementAndGet();
        }

        void recordFailure() {
            this.consecutiveFailures.incrementAndGet();
            this.lastFailureTime.set(System.currentTimeMillis());
        }

        void resetFailures() {
            this.consecutiveFailures.set(0);
            this.lastFailureTime.set(0L);
        }

        int getProvisionCountInWindow() {
            long now = System.currentTimeMillis();
            if (now - this.windowStart.get() >= TimeUnit.MINUTES.toMillis(1L)) {
                return 0;
            }
            return this.windowCount.get();
        }

        int getConsecutiveFailures() {
            return this.consecutiveFailures.get();
        }

        long getLastProvisionTime() {
            return this.lastProvisionTime.get();
        }

        long getWaitTime(int maxPerMinute, long minIntervalMs) {
            long windowRemaining;
            long intervalRemaining;
            long lastProv;
            long cooldown;
            long cooldownRemaining;
            long now = System.currentTimeMillis();
            long wait = 0L;
            long lastFail = this.lastFailureTime.get();
            int failures = this.consecutiveFailures.get();
            if (failures > 0 && (cooldownRemaining = (cooldown = FAILURE_COOLDOWN_MS * (long)Math.min(failures, 5)) - (now - lastFail)) > 0L) {
                wait = Math.max(wait, cooldownRemaining);
            }
            if ((lastProv = this.lastProvisionTime.get()) > 0L && (intervalRemaining = minIntervalMs - (now - lastProv)) > 0L) {
                wait = Math.max(wait, intervalRemaining);
            }
            if (this.windowCount.get() >= maxPerMinute && (windowRemaining = TimeUnit.MINUTES.toMillis(1L) - (now - this.windowStart.get())) > 0L) {
                wait = Math.max(wait, windowRemaining);
            }
            return wait;
        }
    }

    public static class RateLimitInfo {
        private final int provisionCount;
        private final int failureCount;
        private final long lastProvision;
        private final boolean canProvision;

        public RateLimitInfo(int provisionCount, int failureCount, long lastProvision, boolean canProvision) {
            this.provisionCount = provisionCount;
            this.failureCount = failureCount;
            this.lastProvision = lastProvision;
            this.canProvision = canProvision;
        }

        public int getProvisionCount() {
            return this.provisionCount;
        }

        public int getFailureCount() {
            return this.failureCount;
        }

        public long getLastProvision() {
            return this.lastProvision;
        }

        public boolean canProvision() {
            return this.canProvision;
        }
    }
}

