package io.jenkins.plugins.openmfa.util;

import hudson.model.User;
import io.jenkins.plugins.openmfa.MFAGlobalConfiguration;
import io.jenkins.plugins.openmfa.MFAUserProperty;
import io.jenkins.plugins.openmfa.base.MFAException;
import io.jenkins.plugins.openmfa.constant.TOTPConstants;
import java.util.Optional;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import lombok.AccessLevel;
import lombok.NoArgsConstructor;
import org.apache.commons.codec.binary.Base32;

/**
 * Utility class for TOTP (Time-based One-Time Password) generation and
 * verification.
 */
@NoArgsConstructor(access = AccessLevel.PRIVATE)
public class TOTPUtil {

  /**
   * Generates a TOTP code for the given secret at the current time.
   *
   * @param secret
   *          Base32-encoded secret key
   * @return 6-digit TOTP code
   */
  public static String generateTOTP(String secret) {
    return generateTOTP(
      secret,
      System.currentTimeMillis()
        / TOTPConstants.MILLIS_TO_SECONDS
        / TOTPConstants.TIME_STEP_SECONDS
    );
  }

  /**
   * Generates a TOTP code for the given secret and time counter.
   *
   * @param secret
   *          Base32-encoded secret key
   * @param timeCounter
   *          time counter (usually current time / 30)
   * @return 6-digit TOTP code
   */
  public static String generateTOTP(String secret, long timeCounter) {
    Base32 base32 = new Base32();
    byte[] bytes = base32.decode(secret);
    String hexKey = bytesToHex(bytes);
    String hexTime = Long.toHexString(timeCounter);

    return generateTOTP(
      hexKey, hexTime, String.valueOf(TOTPConstants.TOTP_CODE_DIGITS)
    );
  }

  /**
   * Generates the provisioning URI for QR code generation.
   *
   * @param username
   *          Jenkins username
   * @param secret
   *          Base32-encoded secret key
   * @param issuer
   *          Issuer name (e.g., "Jenkins")
   * @return otpauth:// URI
   */
  public static String getProvisioningUri(
    String username, String secret, String issuer) {
    return String.format(
      TOTPConstants.TOTP_URI_FORMAT, issuer, username, secret.replace("=", ""), issuer
    );
  }

  /**
   * Checks if MFA is enabled for the current user.
   *
   * @return true if MFA is enabled, false otherwise
   */
  public static boolean isMFAEnabled() {
    return JenkinsUtil.getCurrentUser()
      .map(TOTPUtil::isMFAEnabled)
      .orElse(false);
  }

  /**
   * Checks if MFA is enabled for the given user.
   *
   * @param user
   *          the user to check
   * @return true if MFA is enabled, false otherwise
   */
  public static boolean isMFAEnabled(User user) {
    MFAUserProperty property = MFAUserProperty.forUser(user);
    return property != null && property.isEnabled();
  }

  /**
   * Checks if MFA is required for the current user.
   *
   * @return true if MFA is required, false otherwise
   */
  public static boolean isMFARequired() {
    Optional<User> user = JenkinsUtil.getCurrentUser();
    if (user.isPresent()) {
      MFAUserProperty property = MFAUserProperty.forUser(user.get());
      return MFAGlobalConfiguration.get().isRequireMFA()
        || (property != null && property.isEnabled());
    }
    return false;
  }

  /**
   * Verifies a TOTP code against the secret, allowing for time drift.
   *
   * @param secret
   *          Base32-encoded secret key
   * @param code
   *          6-digit code to verify
   * @return true if the code is valid
   */
  public static boolean verifyCode(String secret, String code) {
    if (code == null || code.length() != TOTPConstants.TOTP_CODE_DIGITS) {
      return false;
    }

    try {
      long currentTimeCounter =
        System.currentTimeMillis()
          / TOTPConstants.MILLIS_TO_SECONDS
          / TOTPConstants.TIME_STEP_SECONDS;

      // Check current time window and ±1 window (90 seconds total)
      for (int i =
        -TOTPConstants.TIME_WINDOW_TOLERANCE; i <= TOTPConstants.TIME_WINDOW_TOLERANCE; i++) {
        String generatedCode = generateTOTP(secret, currentTimeCounter + i);
        if (generatedCode.equals(code)) {
          return true;
        }
      }
    } catch (Exception e) {
      return false;
    }

    return false;
  }

  private static String bytesToHex(byte[] bytes) {
    StringBuilder sb = new StringBuilder();
    for (byte b : bytes) {
      sb.append(String.format(TOTPConstants.HEX_FORMAT, b));
    }
    return sb.toString();
  }

  private static String generateTOTP(String key, String time, String returnDigits) {
    try {
      StringBuilder paddedTime = new StringBuilder();
      while (time.length() < TOTPConstants.HEX_TIME_PADDING_LENGTH) {
        paddedTime.append(TOTPConstants.PADDING_ZERO);
      }
      paddedTime.append(time);

      byte[] msg = hexStringToByteArray(paddedTime.toString());
      byte[] k = hexStringToByteArray(key);

      byte[] hash = hmacSha(TOTPConstants.HMAC_ALGORITHM, k, msg);

      int offset = hash[hash.length - 1] & TOTPConstants.OFFSET_MASK;

      int binary =
        ((hash[offset]
          & TOTPConstants.BINARY_FIRST_BYTE_MASK) << TOTPConstants.SHIFT_24_BITS)
          | ((hash[offset + 1]
            & TOTPConstants.BINARY_OTHER_BYTE_MASK) << TOTPConstants.SHIFT_16_BITS)
          | ((hash[offset + 2]
            & TOTPConstants.BINARY_OTHER_BYTE_MASK) << TOTPConstants.SHIFT_8_BITS)
          | (hash[offset + 3] & TOTPConstants.BINARY_OTHER_BYTE_MASK);

      int otp =
        binary
          % ((int) Math
            .pow(TOTPConstants.DECIMAL_BASE, Integer.parseInt(returnDigits)));

      String result = Integer.toString(otp);
      while (result.length() < Integer.parseInt(returnDigits)) {
        result = TOTPConstants.PADDING_ZERO + result;
      }
      return result;
    } catch (Exception e) {
      throw new MFAException("Error generating TOTP", e);
    }
  }

  private static byte[] hexStringToByteArray(String s) {
    int len = s.length();
    byte[] data = new byte[len / TOTPConstants.HEX_CHARS_PER_BYTE];
    for (int i = 0; i < len; i += TOTPConstants.HEX_CHARS_PER_BYTE) {
      data[i / TOTPConstants.HEX_CHARS_PER_BYTE] =
        (byte) ((Character.digit(s.charAt(i), TOTPConstants.HEX_RADIX) << 4)
          + Character.digit(s.charAt(i + 1), TOTPConstants.HEX_RADIX));
    }
    return data;
  }

  private static byte[] hmacSha(String crypto, byte[] keyBytes, byte[] text) {
    try {
      Mac hmac = Mac.getInstance(crypto);
      SecretKeySpec macKey =
        new SecretKeySpec(keyBytes, TOTPConstants.MAC_KEY_ALGORITHM);
      hmac.init(macKey);
      return hmac.doFinal(text);
    } catch (Exception e) {
      throw new MFAException("Error generating HMAC", e);
    }
  }
}
