package io.jenkins.plugins.neuvector;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import hudson.AbortException;
import hudson.FilePath;
import hudson.Launcher;
import hudson.model.Run;
import hudson.util.ArgumentListBuilder;
import hudson.util.Secret;

import java.io.File;
import java.io.IOException;
import java.io.PrintStream;

import java.io.UnsupportedEncodingException;
import java.net.SocketException;
import java.net.URI;
import java.net.URISyntaxException;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.util.*;
import java.util.concurrent.TimeUnit;

import io.jenkins.plugins.neuvector.model.SeverityRating;
import io.jenkins.plugins.neuvector.model.report.ScanResult;
import io.jenkins.plugins.neuvector.model.report.Vulnerability;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.conn.ConnectTimeoutException;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.ssl.SSLContextBuilder;
import org.apache.http.conn.ssl.TrustSelfSignedStrategy;
import org.apache.http.entity.StringEntity;
import org.apache.http.HttpEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;

import javax.annotation.Nonnull;
import javax.net.ssl.*;

public class NeuVectorWorker {

    private static final int HTTP_CLIENT_CONFIG_TIMEOUT_MINUTES = 30;
    private static final String SCAN_REPORT = "scan_result.json";
    private final Log logger;
    private final Config config;
    private String token;
    private String artifactname;
    private FilePath workspace;

    public NeuVectorWorker(Log logger, Config config){
        this.logger = logger;
        this.config = config;
    }
    public NeuVectorWorker(Log logger, Config config, @Nonnull FilePath workspace, String artifactName) {
        this.logger = logger;
        this.config = config;
        this.workspace = workspace;
        this.artifactname = artifactName;
    }

    @SuppressFBWarnings("DMI_HARDCODED_ABSOLUTE_FILENAME")
    public void scan(Run<?, ?> run, Launcher launcher) throws IOException, InterruptedException, VulnerabilityCriterionException, NoSuchAlgorithmException, KeyStoreException, KeyManagementException, SSLHandshakeException, SSLPeerUnverifiedException {
        String scanJson;

        if (config.isStandaloneScanner()){
            scanJson = scanByStandaloneScanner(run, launcher);
        }else{
            scanJson = scanByAPI();
        }

        if(scanJson.length() > 0) {
            ScanResult scanResult = processScanReport(scanJson, config);
            printJsonReport(scanJson);
            printHTMLReport(scanResult);
            printTxtReport(scanResult);
            makeIfFailDecision(scanResult.getHighSeverityNumber(),scanResult.getMediumSeverityNumber(),scanResult.isBlackListVulExisted(),scanResult.getExistedBlackListVulSet());
        }
    }

    public void printJsonReport(String scanJson) throws IOException, InterruptedException{
        JSONObject reportJson = JSONObject.fromObject(scanJson);
        FilePath workspaceFP = new FilePath(workspace, artifactname + ".json");
        try (PrintStream printStream = new PrintStream(workspaceFP.write(), false, "UTF-8")) {
            Log logger = new Log(printStream);
            logger.println(reportJson.toString(4));
        }
    }

    public void printHTMLReport(ScanResult scanResult) throws IOException, InterruptedException {
        FilePath workspaceFP = new FilePath(workspace, artifactname + ".html");
        try (PrintStream printStream = new PrintStream(workspaceFP.write(), false, "UTF-8")) {
            Log logger = new Log(printStream);
            logger.println("");
            logger.println("<!DOCTYPE html>\n" +
                    "<html lang=\"en\">\n" +
                    "<head>" +
                    "<link rel=\"stylesheet\" href=\"styles.css\">" +
                    "</head>\n" +
                    "<body>");
            logger.println("<h1>Scan Report </h1>\n");

            // output the scan meta data
            logger.println("<h3>Summary</h3>");
            logger.println("<table>\n" +
                    "    <tr>\n" +
                    "        <th>Registry URL</th>\n" +
                    "        <th>Repository</th>\n" +
                    "        <th>Tag</th>\n" +
                    "        <th>High severity VULs</th>\n" +
                    "        <th>High severity threshold</th>\n" +
                    "        <th>Medium severity VULs</th>\n" +
                    "        <th>Medium severity threshold</th>\n" +
                    "        <th>VULs to fail the build</th>\n" +
                    "        <th>Total VULs</th>\n" +
                    "    </tr>");

            logger.println("<tr>");
            if (scanResult.isLocalScan()) {
                logger.println("<td>Local image</td>");
            } else {
                logger.println("<td><a target=\"_parent\" href=\"" + scanResult.getRepository() + "\">" + scanResult.getRegistry() + "</a></td>");
            }

            logger.println("<td> " + scanResult.getRepository() + "</td>");
            logger.println("<td> " + scanResult.getTag() + "</td>");
            logger.println("<td> " + scanResult.getHighVulnerabilitySet().size() + "</td>");
            if(scanResult.getHighSeverityThreshold() != 0) {
                logger.println("<td> " + scanResult.getHighSeverityThreshold() + "</td>");
            }else{
                logger.println("<td> No Limit </td>");
            }

            logger.println("<td> " + scanResult.getMediumVulnerabilitySet().size() + "</td>");
            if(scanResult.getMediumSeverityThreshold() != 0) {
                logger.println("<td> " + scanResult.getMediumSeverityThreshold() + "</td>");
            }else{
                logger.println("<td> No Limit </td>");
            }
            if (scanResult.isBlackListVulExisted()) {
                logger.println("<td> " + String.join(", ", scanResult.getExistedBlackListVulSet()) + "</td>");
            } else {
                logger.println("<td></td>");
            }
            logger.println("<td> " + scanResult.getTotalVulnerabilityNumber() + "</td>");
            logger.println("</tr></table>");

            // the detail info of Vulnerabilities
            logger.println("<h3> Vulnerabilities </h3>");

            if (scanResult.getTotalVulnerabilityNumber() == 0) {
                logger.println("<p>Scanned. No vulnerabilities found.</p>");
            } else {
                logger.println("<table>\n" +
                        "    <tr>\n" +
                        "        <th>Name</th>\n" +
                        "        <th>Severity</th>\n" +
                        "        <th>Score</th>\n" +
                        "        <th>Package</th>\n" +
                        "        <th>Filename</th>\n" +
                        "        <th>Fixed_version</th>\n" +
                        "        <th>Vectors</th>\n" +
                        "        <th>Description</th>\n" +
                        "        <th>Feed_rating</th>\n" +
                        "    </tr>");

                for (Vulnerability vulnerability : scanResult.getHighVulnerabilitySet()) {
                    logger.println("<tr>");
                    logger.println("<td><a target=\"_blank\" href=\"" + vulnerability.getLink() + "\">" + vulnerability.getName().toUpperCase() + "</td>");
                    logger.println("<td> High </td>");
                    logger.println("<td> " + vulnerability.getScore() + "</td>");
                    logger.println("<td>" + vulnerability.getPackage_name() + ":" + vulnerability.getPackage_version() + "</td>");
                    logger.println("<td>" + vulnerability.getFile_name() + "</td>");
                    logger.println("<td>" + vulnerability.getFixed_version() + "</td>");
                    logger.println("<td>" + vulnerability.getVectors() + "</td>");
                    logger.println("<td>" + vulnerability.getDescription() + "</td>");
                    logger.println("<td>" + vulnerability.getFeed_rating() + "</td>");
                    logger.println("</tr>");
                }

                for (Vulnerability vulnerability : scanResult.getMediumVulnerabilitySet()) {
                    logger.println("<tr>");
                    logger.println("<td><a target=\"_blank\" href=\"" + vulnerability.getLink() + "\">" + vulnerability.getName().toUpperCase() + "</td>");
                    logger.println("<td> Medium </td>");
                    logger.println("<td> " + vulnerability.getScore() + "</td>");
                    logger.println("<td>" + vulnerability.getPackage_name() + ":" + vulnerability.getPackage_version() + "</td>");
                    logger.println("<td>" + vulnerability.getFile_name() + "</td>");
                    logger.println("<td>" + vulnerability.getFixed_version() + "</td>");
                    logger.println("<td>" + vulnerability.getVectors() + "</td>");
                    logger.println("<td>" + vulnerability.getDescription() + "</td>");
                    logger.println("<td>" + vulnerability.getFeed_rating() + "</td>");
                    logger.println("</tr>");
                }

                logger.println("</table>\n");
            }

            // output the layer scan result
            if(scanResult.isScanLayerConfigured()){
                if(scanResult.isScanLayerSupported()){
                    logger.println("<h3> Layer Vulnerability History </h3>");
                    Set<String> keys = scanResult.getLayeredVulsMap().keySet();
                    for(String key : keys){
                        Set<Vulnerability> vulSet = scanResult.getLayeredVulsMap().get(key);
                        logger.println("<p>Layer digest " + key + " contains " + vulSet.size() + " vulnerabilities.</p>");
                        if(vulSet.size() > 0 ) {
                            logger.println("<table>\n" +
                                    "    <tr>\n" +
                                    "        <th>Name</th>\n" +
                                    "        <th>Score</th>\n" +
                                    "        <th>Package</th>\n" +
                                    "        <th>Filename</th>\n" +
                                    "        <th>Fixed_version</th>\n" +
                                    "        <th>Link</th>\n" +
                                    "        <th>Feed_rating</th>\n" +
                                    "    </tr>");

                            for (Vulnerability vulnerability : vulSet) {
                                logger.println("<tr>\n");
                                logger.println("<td><a target=\"_parent\" href=\"" + vulnerability.getLink() + "\">" + vulnerability.getName() + "</td>");
                                logger.println("<td> " + vulnerability.getScore() + "</td>");
                                logger.println("<td> " + vulnerability.getPackage_name() + ":" + vulnerability.getPackage_version() + "</td>");
                                logger.println("<td> " + vulnerability.getFile_name() + "</td>");
                                logger.println("<td> " + vulnerability.getFixed_version() + "</td>");
                                logger.println("<td> " + vulnerability.getLink() + "</td>");
                                logger.println("<td> " + vulnerability.getFeed_rating() + "</td>");
                                logger.println("</tr>");
                            }
                            logger.println("</table>");
                        }
                    }

                }else{
                    logger.println("<p> Your Controller Does Not Support Layer Vulnerability Scan </p>");
                }
            }

            // output the found exempted vulnerabilities
            if(scanResult.isWhiteListVulExisted()){
                logger.println("<h3>Exempted Vulnerabilities</h3");
                if(scanResult.getExistedWhiteListVulSet().size() == 1){
                    logger.println("<p> " + scanResult.getExistedWhiteListVulSet().iterator().next().toUpperCase() + "</p>");
                }else{
                    logger.println("<p> " + String.join(",", scanResult.getExistedWhiteListVulSet()) + "</p>");
                }
            }

            logger.println("</body>\n" +
                        "</html>");


        }
    }

    public void printTxtReport(ScanResult scanResult) throws IOException, InterruptedException{
        FilePath workspaceFP = new FilePath(workspace, artifactname);
        try (PrintStream printStream = new PrintStream(workspaceFP.write(), false, "UTF-8")) {
            Log logger = new Log(printStream);
            logger.println("");
            logger.println("************************ Scan Report ************************");
            if (!scanResult.isLocalScan()) {
                logger.println("Registry URL: " + scanResult.getRegistry());
            }
            logger.println("Repository: " + scanResult.getRepository());
            logger.println("Tag: " + scanResult.getTag());
            logger.println("High severity vulnerabilities: " + scanResult.getHighSeverityNumber());
            logger.println("Medium severity vulnerabilities: " + scanResult.getMediumSeverityNumber());
            logger.println("Total vulnerabilities: " + scanResult.getTotalVulnerabilityNumber());
            logger.println("********************** Vulnerabilities **********************");
            logger.println("");

            if (scanResult.getTotalVulnerabilityNumber() == 0) {
                logger.println("Scanned. No vulnerabilities found.");
            } else {
                for (Vulnerability vulnerability : scanResult.getHighVulnerabilitySet()) {
                    logger.println("Name: " + vulnerability.getName().toUpperCase());
                    logger.println("Score: " + vulnerability.getScore());
                    logger.println("Severity: High");
                    logger.println("Vectors: " + vulnerability.getVectors());
                    logger.println("Description: " + vulnerability.getDescription());
                    logger.println("Package_name: " + vulnerability.getPackage_name());
                    logger.println("Package_version: " + vulnerability.getPackage_version());
                    logger.println("Filename: " + vulnerability.getFile_name());
                    logger.println("Fixed_version: " + vulnerability.getFixed_version());
                    logger.println("Link: " + vulnerability.getLink());
                    logger.println("");
                }
                for (Vulnerability vulnerability : scanResult.getMediumVulnerabilitySet()) {
                    logger.println("Name: " + vulnerability.getName().toUpperCase());
                    logger.println("Score: " + vulnerability.getScore());
                    logger.println("Severity: Medium");
                    logger.println("Vectors: " + vulnerability.getVectors());
                    logger.println("Description: " + vulnerability.getDescription());
                    logger.println("Package_name: " + vulnerability.getPackage_name());
                    logger.println("Package_version: " + vulnerability.getPackage_version());
                    logger.println("Filename: " + vulnerability.getFile_name());
                    logger.println("Fixed_version: " + vulnerability.getFixed_version());
                    logger.println("Link: " + vulnerability.getLink());
                    logger.println("");
                }

                // output the layer scan result
                if(scanResult.isScanLayerConfigured()){
                    logger.println("");
                    if(scanResult.isScanLayerSupported()){
                        logger.println("**************** Layer Vulnerability History ****************");
                        logger.println("");
                        Set<String> keys = scanResult.getLayeredVulsMap().keySet();
                        for(String key : keys){
                            Set<Vulnerability> vulSet = scanResult.getLayeredVulsMap().get(key);
                            logger.println("Layer digest " + key + " contains " + vulSet.size() + " vulnerabilities.");
                            logger.println("");
                            for(Vulnerability vulnerability: vulSet){
                                logger.println("Name: " + vulnerability.getName().toLowerCase()
                                        + ", Score: " + vulnerability.getScore()
                                        + ", Package_name: " + vulnerability.getPackage_name()
                                        + ", Package_version: " + vulnerability.getPackage_version()
                                        + ", Filename: " + vulnerability.getFile_name()
                                        + ", Fixed_version: " + vulnerability.getFixed_version()
                                        + ", Link: " + vulnerability.getLink());
                            }
                        }
                        logger.println("");
                    }else{
                        logger.println("*** Your Controller Does Not Support Layer Vulnerability Scan ***");
                        logger.println("");
                    }
                }

                // output the found exempted vulnerabilities
                if(scanResult.isWhiteListVulExisted()){
                    logger.println("********************** Exempt Vulnerability **********************");
                    if(scanResult.getExistedWhiteListVulSet().size() == 1){
                        logger.println("The vulnerability " + scanResult.getExistedWhiteListVulSet().iterator().next().toUpperCase() + " is exempt.");
                    }else{
                        logger.println("The vulnerabilities " + String.join(",", scanResult.getExistedWhiteListVulSet()) + " are exempt.");
                    }
                }
            }
        }
    }

    private String scanByStandaloneScanner(Run<?, ?> run, Launcher launcher) throws IOException, InterruptedException{
        int exitCode = 0;
        String scanResultPath;
        ArgumentListBuilder args = new ArgumentListBuilder();

        File outFile = new File(run.getRootDir(), "out");
        Launcher.ProcStarter ps = launcher.launch();
        try (PrintStream print_stream = new PrintStream(outFile, "UTF-8")){
            ps.stderr(print_stream);
            ps.stdout(print_stream);
            ps.quiet(true);

            // docker login
            if(! config.getScannerRegistryUser().isEmpty()){
                args.add("docker", "login");
                args.add("-u",config.getScannerRegistryUser(),"-p");
                args.addMasked(config.getScannerRegistryPassword().getPlainText());
                args.add(config.getScannerRegistryURL());
                ps.cmds(args);
                logger.println("Logging in " + config.getScannerRegistryURL() + " with " + config.getScannerRegistryUser() + " ... ");
                exitCode = ps.join();
            }

            if(exitCode != 0) {
                logger.println("docker failed to login " + config.getScannerRegistryURL() + " Please check the global configuration.");
            }

            // docker pull NeuVector Scanner
            args.clear();
            args.add("docker", "pull");
            args.add(config.getScannerRegistryURL() + "/" + config.getScannerImage());
            ps.cmds(args);
            logger.println("Pulling NeuVector Scanner from " + config.getScannerRegistryURL() + "/" + config.getScannerImage() + " ...");
            exitCode = ps.join(); // RUN !

            if(exitCode != 0) {
                logger.println("docker failed to pull " + config.getScannerRegistryURL() + "/" + config.getScannerImage() + " Please check the global configuration.");
            }

            // docker run scan
            args.clear();
            String nvScannerName = generateScannerName();
            args.add("docker", "run");
            args.add("--name", nvScannerName);

            /*
               When the Jenkins runs as an application, the scan result file is generated
               in {Jenkins worksapce}/scan_result.json.
            */
            scanResultPath = this.workspace.getRemote();

            // When user setup controller api url, the scan result will sent to the controller.
            if (config.isSendReportToController() && config.getControllerApiUrl()!= null) {
                args.add("-e", "CLUSTER_JOIN_ADDR=" + config.getControllerApiUrl().getHost(), "-e", "CLUSTER_JOIN_PORT=" + config.getControllerApiUrl().getPort(), "-e", "SCANNER_CTRL_API_USERNAME=" + config.getUser(), "-e", "SCANNER_CTRL_API_PASSWORD=" + config.getPassword());
            }

            String regUrl = "";
            if(!config.isLocal()){
                regUrl = config.getRegistry().getRegUrl();
                args.add("-e","SCANNER_REGISTRY=" + regUrl, "-e", "SCANNER_REGISTRY_USERNAME=" + config.getRegistry().getRegUsername() ,"-e", "SCANNER_REGISTRY_PASSWORD=" + config.getRegistry().getRegPassword());
                regUrl = regUrl + "/";
            }
            args.add("-e", "SCANNER_REPOSITORY="+config.getRepository(), "-e", "SCANNER_TAG="+config.getTag(), "-e", "SCANNER_SCAN_LAYERS="+config.getScanLayers(), "-e", "SCANNER_ON_DEMAND=true", "-v", "/var/run/docker.sock:/var/run/docker.sock",config.getScannerRegistryURL()+"/"+config.getScannerImage());

            ps.cmds(args);

            logger.println("Scanning " + regUrl + config.getRepository() + ":" + config.getTag());
            exitCode = ps.join(); // RUN !

            if( exitCode != 0 ){
                String causeMessage = "Failed to run the scan. Check the log in " + outFile.getAbsolutePath();
                logger.println(causeMessage);
                throw new AbortException(causeMessage);
            }

            // docker cp scan_result.json
            args.clear();
            args.add("docker", "cp", nvScannerName + ":/var/neuvector/" + SCAN_REPORT, scanResultPath);
            ps.cmds(args);
            exitCode = ps.join();

            if(exitCode != 0) {
                logger.println("docker failed to do the copy from " + nvScannerName + ":/var/neuvector/" + SCAN_REPORT + " to " + scanResultPath);
            }else {
                logger.println("Copied NeuVector Scan result from " + nvScannerName + ":/var/neuvector/" + SCAN_REPORT + " to " + scanResultPath);
            }

            // docker rm NeuVector Scanner
            args.clear();
            args.add("docker", "rm", nvScannerName);
            ps.cmds(args);
            exitCode = ps.join();

            if(exitCode != 0) {
                logger.println("Failed to remove NeuVector Scanner container " + nvScannerName);
            }else {
                logger.println("Removed the NeuVector Scanner container " + nvScannerName);
            }

            // docker log out
            args.clear();
            args.add("docker","logout");
            ps.cmds(args);

            if(ps.join() == 0){
                logger.println("Logged out Docker registry " + config.getScannerRegistryURL());
            }else{
                logger.println("Failed to log out Docker registry "  + config.getScannerRegistryURL());
            }

        }

        String scanReport = scanResultPath + "/" + SCAN_REPORT;

        StringBuilder contentBuilder = new StringBuilder();

        FilePath scanReportFile = new FilePath(launcher.getChannel(), scanReport);

        boolean isScanFileExist = false;
        int scanTimeout = config.getScanTimeout().intValue() * 60 ;
        int timer = 0;

        //By default, 10 mins timeout to create the scan result file
        while ( !isScanFileExist && (timer < scanTimeout)){
            if (scanReportFile.exists()) {
                isScanFileExist = true;
            }

            TimeUnit.SECONDS.sleep(10);
            timer = timer + 10;
        }

        if (isScanFileExist){
            logger.println("Scan result file: " + scanReportFile.toURI().getPath());
            contentBuilder.append(scanReportFile.readToString());
        }else{
            logger.println("Failed to find the scan result file in " + scanReportFile.toURI().getPath());
            throw new AbortException("Failed to get the scan result.");
        }

        return contentBuilder.toString();
    }

    // if disableAPIKeyVerification, NeuVector will use jwt token to communicate with controller, otherwise, it will use API key
    private String scanByAPI() throws IOException, NoSuchAlgorithmException, KeyStoreException, KeyManagementException, SSLHandshakeException, SSLPeerUnverifiedException {
        try (CloseableHttpClient httpclient = makeHttpClient()) {
            if (config.getDisableAPIKeyVerification()) {
                getToken(httpclient);
            }
            String scanJson = requestScan(httpclient);
            if (config.getDisableAPIKeyVerification()) {
                logout(httpclient);
            }
            return scanJson;
        }
    }

    private String getSeverity(Double score, Double highSeverityThreshold, Double mediumSeverityThreshold) {
        String severity = "";
        if(score != null) {
            if (score >= highSeverityThreshold){
                severity = SeverityRating.High.name();
            }else if(score >= mediumSeverityThreshold){
                severity = SeverityRating.Medium.name();
            }
        }

        return severity;
    }

    private CloseableHttpClient makeHttpClient() throws NoSuchAlgorithmException, KeyStoreException, KeyManagementException, SSLHandshakeException, SSLPeerUnverifiedException {
        SSLContextBuilder builder = new SSLContextBuilder();
        SSLConnectionSocketFactory sslsf;

        if (config.getDisableTLSCertVerification()) {
            builder.loadTrustMaterial(null, new TrustSelfSignedStrategy());
            sslsf = new SSLConnectionSocketFactory(builder.build(), NoopHostnameVerifier.INSTANCE);
        } else {
            builder.loadTrustMaterial(null, new BouncyCastleTrustStrategy(config.getServerCertificate()));
            sslsf = new SSLConnectionSocketFactory(builder.build(), new WhitelistedHostnameVerifier(config.getControllerApiUrl().getHost()));
        }

        RequestConfig config = RequestConfig.custom()
            .setConnectTimeout(HTTP_CLIENT_CONFIG_TIMEOUT_MINUTES * 60 * 1000)
            .setSocketTimeout(HTTP_CLIENT_CONFIG_TIMEOUT_MINUTES * 60 * 1000)
            .setConnectionRequestTimeout(HTTP_CLIENT_CONFIG_TIMEOUT_MINUTES * 60 * 1000)
            .build();
        return HttpClients.custom().setSSLSocketFactory(sslsf).setDefaultRequestConfig(config).build();
    }

    public void testConnection() throws IOException, NoSuchAlgorithmException, KeyStoreException, KeyManagementException, SSLHandshakeException, SSLPeerUnverifiedException {
        try (CloseableHttpClient httpclient = makeHttpClient()) {
            getToken(httpclient);
            logout(httpclient);
        }
    }

    private void getToken(CloseableHttpClient httpclient) throws AbortException {
        String uriPathForGetToken = "/v1/auth";
        URI uriForGetToken = buildUri(config.getControllerApiUrl(), uriPathForGetToken);

        HttpPost httpPostForGetToken = new HttpPost(uriForGetToken);
        httpPostForGetToken.addHeader("Content-Type", "application/json");

        JSONObject passwordJson = new JSONObject();
        passwordJson.put("username", config.getUser());
        passwordJson.put("password", config.getPassword());
        JSONObject httpBodyJson = new JSONObject();
        httpBodyJson.put("password", passwordJson);

        try {
            httpPostForGetToken.setEntity(new StringEntity(httpBodyJson.toString()));
        } catch (UnsupportedEncodingException e) {
            String causeMessage = "Unsupported encoding from NeuVector Username and/or Password in global configuration.";
            if (logger != null) {
                logger.println(causeMessage);
            }
            throw new AbortException(causeMessage);
        }

        try (CloseableHttpResponse httpResponseFromGetToken = httpclient.execute(httpPostForGetToken)) {
            int statusCode = httpResponseFromGetToken.getStatusLine().getStatusCode();
            HttpEntity httpEntityFromGetToken = httpResponseFromGetToken.getEntity();
            String serverMessageFromGetToken = EntityUtils.toString(httpEntityFromGetToken);
            EntityUtils.consume(httpEntityFromGetToken);

            if (statusCode == 200) {
                token = JSONObject.fromObject(serverMessageFromGetToken).getJSONObject("token").getString("token");
            } else if (statusCode == 401 || statusCode == 404 || statusCode == 405) {
                String causeMessage = "Invalid credential of NeuVector controller";
                if (logger != null) {
                    logger.println(causeMessage);
                }
                throw new AbortException(causeMessage);
            } else {
                String causeMessage = "Failed to get token. Http status code: " + statusCode + ". Message: " + serverMessageFromGetToken;
                if (logger != null) {
                    logger.println(causeMessage);
                }
                throw new AbortException(causeMessage);
            }
        }  catch (SSLHandshakeException e) {
            String causeMessage = e.getMessage();

            if (logger != null) {
                logger.println(causeMessage);
            }
            throw new AbortException(causeMessage);
        } catch (SSLPeerUnverifiedException e) {
            String causeMessage = "Hostname verification error: " + e.getMessage();
            if (logger != null) {
                logger.println(causeMessage);
            }
            throw new AbortException(causeMessage);
        }  catch (SocketException e) {
            String causeMessage = "SocketException: " + e.getMessage();
            if (logger != null) {
                logger.println(causeMessage);
            }
            throw new AbortException(causeMessage);
        } catch (ConnectTimeoutException e) {
            String causeMessage = "Connection timed out: " + e.getMessage();
            if (logger != null) {
                logger.println(causeMessage);
            }
            throw new AbortException(causeMessage);
        } catch (IllegalArgumentException e) {
            String causeMessage = "IllegalArgumentException: " + e.getMessage();
            if (logger != null) {
                logger.println(causeMessage);
            }
            throw new AbortException(causeMessage);
        } catch (ClientProtocolException e) {
            String causeMessage = "Invalid settings of the NeuVector Vulnerability Scanner on the global configuration page.";
            if (logger != null) {
                logger.println(causeMessage);
            }
            throw new AbortException(causeMessage);
        } catch (IOException e) {
            String causeMessage = "NeuVector controller connection error with token: " + e.getMessage();
            if (logger != null) {
                logger.println(causeMessage);
            }
            throw new AbortException(causeMessage);
        }
    }

    private void logout(CloseableHttpClient httpclient) throws IOException {
        String uriPathForLogout = "/v1/auth";
        URI uriForLogout = buildUri(config.getControllerApiUrl(), uriPathForLogout);
        HttpDelete httpDeleteForLogout = new HttpDelete(uriForLogout);
        httpDeleteForLogout.addHeader("Content-Type", "application/json");
        httpDeleteForLogout.addHeader("X-Auth-Token", token);
        CloseableHttpResponse httpResponseFromLogout = httpclient.execute(httpDeleteForLogout);
        httpResponseFromLogout.close();
    }

    public void testApiKey() throws IOException, NoSuchAlgorithmException, KeyStoreException, KeyManagementException, SSLHandshakeException, SSLPeerUnverifiedException {
        try (CloseableHttpClient httpclient = makeHttpClient()) {
            validateApiKeyWithRepoScan(httpclient);
        }
    }

    private void validateApiKeyWithRepoScan(CloseableHttpClient httpclient) throws AbortException {
        String uriPathForValidation = "/v1/scan/repository";
        URI uriForValidation = buildUri(config.getControllerApiUrl(), uriPathForValidation);

        HttpPost httpPostForValidation = new HttpPost(uriForValidation);
        httpPostForValidation.addHeader("Content-Type", "application/json");
        httpPostForValidation.addHeader("X-Auth-Apikey", config.getServerApiKey().getPlainText());

        // Send an empty payload
        try {
            httpPostForValidation.setEntity(new StringEntity("{}"));
        } catch (UnsupportedEncodingException e) {
            String causeMessage = "Unsupported encoding for validation payload.";
            if (logger != null) {
                logger.println(causeMessage);
            }
            throw new AbortException(causeMessage);
        }

        try (CloseableHttpResponse httpResponseForValidation = httpclient.execute(httpPostForValidation)) {
            int statusCode = httpResponseForValidation.getStatusLine().getStatusCode();
            HttpEntity httpEntityForValidation = httpResponseForValidation.getEntity();
            String serverMessageForValidation = EntityUtils.toString(httpEntityForValidation);
            EntityUtils.consume(httpEntityForValidation);

            if (statusCode == 400) {
                // API key is accepted; this is the expected behavior with an empty payload
                if (logger != null) {
                    logger.println("API key is valid but empty payload resulted in expected error: 400");
                }
            } else if (statusCode == 403) {
                // API key is not accepted
                String causeMessage = "Invalid API key: Access forbidden (403).";
                if (logger != null) {
                    logger.println(causeMessage);
                }
                throw new AbortException(causeMessage);
            } else {
                // Unexpected status codes
                String causeMessage = "Unexpected response. Http status code: " + statusCode + ". Message: " + serverMessageForValidation;
                if (logger != null) {
                    logger.println(causeMessage);
                }
                throw new AbortException(causeMessage);
            }
        } catch (SSLHandshakeException e) {
            String causeMessage = e.getMessage();
            if (logger != null) {
                logger.println(causeMessage);
            }
            throw new AbortException(causeMessage);
        } catch (SSLPeerUnverifiedException e) {
            String causeMessage = "Hostname verification error: " + e.getMessage();
            if (logger != null) {
                logger.println(causeMessage);
            }
            throw new AbortException(causeMessage);
        } catch (SocketException e) {
            String causeMessage = "SocketException: " + e.getMessage();
            if (logger != null) {
                logger.println(causeMessage);
            }
            throw new AbortException(causeMessage);
        } catch (ConnectTimeoutException e) {
            String causeMessage = "Connection timed out: " + e.getMessage();
            if (logger != null) {
                logger.println(causeMessage);
            }
            throw new AbortException(causeMessage);
        } catch (IllegalArgumentException e) {
            String causeMessage = "IllegalArgumentException: " + e.getMessage();
            if (logger != null) {
                logger.println(causeMessage);
            }
            throw new AbortException(causeMessage);
        } catch (ClientProtocolException e) {
            String causeMessage = "Invalid settings of the NeuVector Vulnerability Scanner on the global configuration page.";
            if (logger != null) {
                logger.println(causeMessage);
            }
            throw new AbortException(causeMessage);
        } catch (IOException e) {
            String causeMessage = "NeuVector controller connection error during API key validation: " + e.getMessage();
            if (logger != null) {
                logger.println(causeMessage);
            }
            throw new AbortException(causeMessage);
        }
    }

    private String requestScan(CloseableHttpClient httpclient) throws IOException {
        long startTimeMillis = System.currentTimeMillis();
        int timeoutSeconds = config.getTimeout() * 60;

        String uriPathForScan = "/v1/scan/repository";
        URI uriForScan = buildUri(config.getControllerApiUrl(), uriPathForScan);

        HttpPost httpPostForScan = new HttpPost(uriForScan);
        httpPostForScan.addHeader("Content-Type", "application/json");
        if (config.getDisableAPIKeyVerification()) {
            httpPostForScan.addHeader("X-Auth-Token", token);
        } else {
            httpPostForScan.addHeader("X-Auth-Apikey", config.getServerApiKey().getPlainText());
        }

        JSONObject httpBodyJson = new JSONObject();
        JSONObject requestJson = new JSONObject();
        requestJson.put("repository", config.getRepository());
        if (config.getTag() != null) {
            requestJson.put("tag", config.getTag());
        }

        if (config.getScanMeta() != null) {
            JSONObject metaJson = new JSONObject();
            metaJson.put("source", config.getScanMeta().getSource());
            metaJson.put("user", config.getScanMeta().getUser());
            metaJson.put("job", config.getScanMeta().getJob());
            metaJson.put("workspace", config.getScanMeta().getWorkspace());
            requestJson.put("metadata", metaJson);
        } else {
            requestJson.put("metadata",null);
        }

        if (config.isLocal()) {
            requestJson.put("registry", "");
            requestJson.put("username", "");
            requestJson.put("password", "");
        } else {
            requestJson.put("registry", config.getRegistry().getRegUrl());
            requestJson.put("username", config.getRegistry().getRegUsername());
            requestJson.put("password", Secret.toString(config.getRegistry().getRegPassword()).trim());
        }
        requestJson.put("scan_layers", config.getScanLayers());

        httpBodyJson.put("request", requestJson);

        try{
            httpPostForScan.setEntity(new StringEntity(httpBodyJson.toString()));
        } catch (UnsupportedEncodingException e) {
            String causeMessage = "Unsupported encoding from registry, repository or tag.";
            logger.println(causeMessage);
            throw new AbortException(causeMessage);
        }

        CloseableHttpResponse httpResponseFromScan = null;
        try {
            httpResponseFromScan = httpclient.execute(httpPostForScan);
            while (httpResponseFromScan.getStatusLine().getStatusCode() == 304) {
                if (timeoutSeconds > 0) {
                    long elapsedSeconds = (System.currentTimeMillis() - startTimeMillis) / 1000;
                    if (elapsedSeconds > timeoutSeconds) {
                        String causeMessage = "Time out.";
                        logger.println(causeMessage);
                        throw new AbortException(causeMessage);
                    }
                }

                httpResponseFromScan = httpclient.execute(httpPostForScan);
                logger.println("Scanning in progress...");
            }

            int statusCode = httpResponseFromScan.getStatusLine().getStatusCode();
            HttpEntity httpEntityFromScan = httpResponseFromScan.getEntity();
            String serverMessageFromScan = "N/A";

            if (httpEntityFromScan != null) {
                serverMessageFromScan = EntityUtils.toString(httpEntityFromScan);
                EntityUtils.consume(httpEntityFromScan);
            }

            if (statusCode != 200 || httpEntityFromScan == null) {
                String causeMessage = "Scan failed. Http status code: " + statusCode + ". Message: " + serverMessageFromScan;
                logger.println(causeMessage);
                throw new AbortException(causeMessage);
            }
            return serverMessageFromScan;
        } catch (IOException e) {
            String causeMessage = "NeuVector controller connection error: " + e.getMessage();
            logger.println(causeMessage);
            throw new AbortException(causeMessage);
        } finally {
            if (httpResponseFromScan != null) {
                httpResponseFromScan.close();
            }
        }
    }

    private URI buildUri(ControllerEndpointUrl apiUrl, String path) throws AbortException {
        URI uri;
        try {
            uri = new URIBuilder().setScheme(apiUrl.getProtocol()).setHost(apiUrl.getHost()).setPort(apiUrl.getPort()).setPath(path).build();
        } catch (URISyntaxException e) {
            String causeMessage = "URI syntax error from NeuVector Controller IP and/or API port in global configuration.";
            logger.println(causeMessage);
            throw new AbortException(causeMessage);
        }
        return uri;
    }

    private ScanResult processScanReport(String serverMessageFromScan, Config config) throws AbortException {

        ScanResult scanResult = new ScanResult();

        //initial scanResult with the registry, repository and tag of the scanned image
        if(config.isLocal()){
            scanResult.setLocalScan(true);
        }else{
            scanResult.setLocalScan(false);
            scanResult.setRegistry(config.getRegistry().getRegUrl());
        }

        scanResult.setRepository(config.getRepository());
        scanResult.setTag(config.getTag());


        int highSeverityThreshold = 0;
        if(Objects.nonNull(config.getNumberOfHighSeverityToFail()) && !config.getNumberOfHighSeverityToFail().isEmpty()){
            highSeverityThreshold = Integer.parseInt(config.getNumberOfHighSeverityToFail());
        }
        int mediumSeverityThreshold = 0;
        if(Objects.nonNull(config.getNumberOfMediumSeverityToFail()) && !config.getNumberOfMediumSeverityToFail().isEmpty()){
            mediumSeverityThreshold = Integer.parseInt(config.getNumberOfMediumSeverityToFail());
        }

        // initial the threshold for high and medium severity vulnerabilities
        scanResult.setHighSeverityThreshold(highSeverityThreshold);
        scanResult.setMediumSeverityThreshold(mediumSeverityThreshold);

        Set<String> vulBlackListSet = new HashSet<>();
        Set<String> vulWhiteListSet = new HashSet<>();

        //Vuls blacklist
        if (config.getNameOfVulnerabilityToFailOne() != null && !config.getNameOfVulnerabilityToFailOne().isEmpty()) {
            vulBlackListSet.add(config.getNameOfVulnerabilityToFailOne().toLowerCase());
        }
        if (config.getNameOfVulnerabilityToFailTwo() != null && !config.getNameOfVulnerabilityToFailTwo().isEmpty()) {
            vulBlackListSet.add(config.getNameOfVulnerabilityToFailTwo().toLowerCase());
        }
        if (config.getNameOfVulnerabilityToFailThree() != null && !config.getNameOfVulnerabilityToFailThree().isEmpty()) {
            vulBlackListSet.add(config.getNameOfVulnerabilityToFailThree().toLowerCase());
        }
        if (config.getNameOfVulnerabilityToFailFour() != null && !config.getNameOfVulnerabilityToFailFour().isEmpty()) {
            vulBlackListSet.add(config.getNameOfVulnerabilityToFailFour().toLowerCase());
        }

        //Vuls white list
        if (config.getNameOfVulnerabilityToExemptOne() != null && !config.getNameOfVulnerabilityToExemptOne().isEmpty()) {
            vulWhiteListSet.add(config.getNameOfVulnerabilityToExemptOne().toLowerCase());
        }
        if (config.getNameOfVulnerabilityToExemptTwo() != null && !config.getNameOfVulnerabilityToExemptTwo().isEmpty()) {
            vulWhiteListSet.add(config.getNameOfVulnerabilityToExemptTwo().toLowerCase());
        }
        if (config.getNameOfVulnerabilityToExemptThree() != null && !config.getNameOfVulnerabilityToExemptThree().isEmpty()) {
            vulWhiteListSet.add(config.getNameOfVulnerabilityToExemptThree().toLowerCase());
        }
        if (config.getNameOfVulnerabilityToExemptFour() != null && !config.getNameOfVulnerabilityToExemptFour().isEmpty()) {
            vulWhiteListSet.add(config.getNameOfVulnerabilityToExemptFour().toLowerCase());
        }

        // initial scanResult with blackListSet and whiteListSet
        scanResult.setBlackListVulSet(vulBlackListSet);
        scanResult.setWhiteListVulSet(vulWhiteListSet);

        JSONObject reportJson = JSONObject.fromObject(serverMessageFromScan).getJSONObject("report");

        // to exit if the report json object is empty
        if ( reportJson.isNullObject() ){
            String causeMessage = "Scan failed. Error Message: " + JSONObject.fromObject(serverMessageFromScan).get("error_message").toString();
            logger.println(causeMessage);
            throw new AbortException(causeMessage);
        }

        int totalVulnerabilityNumber = 0;
        int totalHighSeverity = 0;
        int totalMediumSeverity = 0;

        boolean isSeverityScaleCustomized = config.getCustomizedRatingScale();
        Double highSeverityScaleThreshold = config.getHighSeverityThreshold();
        Double mediumSeverityScaleThreshold = config.getMediumSeverityThreshold();

        boolean hasBlackListVuls = false;
        boolean hasWhiteListVuls = false;
        Set<String> existedBlackListVulSet = new HashSet<>();
        Set<String> existedWhiteListVulSet = new HashSet<>();
        Set<Vulnerability> highVulnerabilitySet = new HashSet<>();
        Set<Vulnerability> mediumVulnerabilitySet = new HashSet<>();

        JSONArray vulnerabilityArray = reportJson.getJSONArray("vulnerabilities");
        if (vulnerabilityArray.size() > 0) {
            for (int i = 0; i < vulnerabilityArray.size(); i++) {
                JSONObject vulnerabilityObject = vulnerabilityArray.getJSONObject(i);
                String name = vulnerabilityObject.getString("name").toLowerCase();

                if (!vulBlackListSet.isEmpty() && vulBlackListSet.contains(name)) {
                    hasBlackListVuls = true;
                    existedBlackListVulSet.add(name.toUpperCase());
                }

                if ( vulWhiteListSet.isEmpty() || !vulWhiteListSet.contains(name) ) {
                    totalVulnerabilityNumber = totalVulnerabilityNumber + 1;
                    String severity;
                    if(isSeverityScaleCustomized){
                        String s_score = vulnerabilityObject.getString("score");
                        severity = getSeverity(Double.parseDouble(s_score), highSeverityScaleThreshold, mediumSeverityScaleThreshold);
                    }else{
                        severity = vulnerabilityObject.getString("severity");
                    }

                    Vulnerability vulnerability = new Vulnerability();

                    vulnerability.setName(name);
                    vulnerability.setLink(vulnerabilityObject.getString("link"));
                    vulnerability.setScore(Float.valueOf(vulnerabilityObject.getString("score")));
                    vulnerability.setPackage_name(vulnerabilityObject.getString("package_name"));
                    vulnerability.setPackage_version(vulnerabilityObject.getString("package_version"));
                    vulnerability.setFixed_version(vulnerabilityObject.getString("fixed_version"));
                    vulnerability.setVectors(vulnerabilityObject.getString("vectors"));
                    vulnerability.setDescription(vulnerabilityObject.getString("description"));
                    vulnerability.setFeed_rating(vulnerabilityObject.getString("feed_rating"));
                    if (severity.equalsIgnoreCase("High")) {
                        totalHighSeverity = totalHighSeverity + 1;
                        vulnerability.setSeverity("High");
                        highVulnerabilitySet.add(vulnerability);
                    } else if (severity.equalsIgnoreCase("Medium")) {
                        totalMediumSeverity = totalMediumSeverity + 1;
                        vulnerability.setSeverity("Medium");
                        mediumVulnerabilitySet.add(vulnerability);
                    }

                } else{
                    hasWhiteListVuls = true;
                    existedWhiteListVulSet.add(name.toUpperCase());
                }
            }
            // initial the scanResult with found blackListSet, whiteListSet,
            // total Vul number, total high severity number, total medium severity number
            // all high severity vul set and all medium severity vul set
            scanResult.setBlackListVulExisted(hasBlackListVuls);
            scanResult.setWhiteListVulExisted(hasWhiteListVuls);
            scanResult.setExistedBlackListVulSet(existedBlackListVulSet);
            scanResult.setExistedWhiteListVulSet(existedWhiteListVulSet);
            scanResult.setTotalVulnerabilityNumber(totalVulnerabilityNumber);
            scanResult.setHighSeverityNumber(totalHighSeverity);
            scanResult.setMediumSeverityNumber(totalMediumSeverity);
            scanResult.setHighVulnerabilitySet(highVulnerabilitySet);
            scanResult.setMediumVulnerabilitySet(mediumVulnerabilitySet);

            // scan layers
            if(config.getScanLayers()){
                scanResult.setScanLayerConfigured(true);
                if (reportJson.has("layers")){
                    scanResult.setScanLayerSupported(true);
                    JSONArray layerArray = reportJson.getJSONArray("layers");
                    LinkedHashMap<String, Set<Vulnerability>> layeredVulnerabilityMap = new LinkedHashMap<String, Set<Vulnerability>>();
                    for (int i = 0; i < layerArray.size(); i++) {
                        JSONObject layerObject = layerArray.getJSONObject(i);
                        int subStringLen = 12;
                        if( layerObject.getString("digest").length() < 12 ){
                            subStringLen = layerObject.getString("digest").length();
                        }
                        String layerDigest = layerObject.getString("digest").substring(0, subStringLen);
                        JSONArray layerVulnerabilityArray = layerObject.getJSONArray("vulnerabilities");
                        Set<Vulnerability> layeredVulnerabilitySet = new HashSet<>();
                        for (int j = 0; j < layerVulnerabilityArray.size(); j++) {
                            JSONObject layerVulnerabilityObject = layerVulnerabilityArray.getJSONObject(j);
                            String vulnerabilityName = layerVulnerabilityObject.getString("name").toLowerCase();
                            if(! (hasWhiteListVuls && vulWhiteListSet.contains(vulnerabilityName))){
                                Vulnerability vulnerability = new Vulnerability();
                                vulnerability.setName(vulnerabilityName);
                                vulnerability.setScore(Float.valueOf(layerVulnerabilityObject.getString("score")));
                                vulnerability.setPackage_name(layerVulnerabilityObject.getString("package_name"));
                                vulnerability.setPackage_version(layerVulnerabilityObject.getString("package_version"));
                                vulnerability.setFixed_version(layerVulnerabilityObject.getString("fixed_version"));
                                vulnerability.setLink(layerVulnerabilityObject.getString("link"));
                                vulnerability.setFeed_rating(layerVulnerabilityObject.getString("feed_rating"));
                                layeredVulnerabilitySet.add(vulnerability);
                            }
                        }
                        layeredVulnerabilityMap.put(layerDigest, layeredVulnerabilitySet);
                    }
                    scanResult.setLayeredVulsMap(layeredVulnerabilityMap);
                }else{
                    scanResult.setScanLayerSupported(false);
                }
            }else{
                scanResult.setScanLayerConfigured(false);
            }

        } else {
            scanResult.setTotalVulnerabilityNumber(totalVulnerabilityNumber);
            scanResult.setScanSummary("Scanned. No vulnerabilities found.");
        }

        return scanResult;
    }

    private void makeIfFailDecision(int currentHighSeverity, int currentMediumSeverity, boolean foundName,
                                    Set<String> namesToPresent) throws VulnerabilityCriterionException {
        boolean numberExceed = false;
        StringBuilder statementBuilder = new StringBuilder();

        if (config.getNumberOfHighSeverityToFail() != null && !config.getNumberOfHighSeverityToFail().isEmpty()) {
            int configNumberOfHigh = Integer.parseInt(config.getNumberOfHighSeverityToFail());
            if (configNumberOfHigh != 0 && configNumberOfHigh <= currentHighSeverity) {
                numberExceed = true;
                statementBuilder.append(currentHighSeverity).append(" High severity vulnerabilities");
            }
        }

        if (config.getNumberOfMediumSeverityToFail() != null && !config.getNumberOfMediumSeverityToFail().isEmpty()) {
            int configNumberOfMedium = Integer.parseInt(config.getNumberOfMediumSeverityToFail());
            if (configNumberOfMedium != 0 && configNumberOfMedium <= currentMediumSeverity) {
                if (numberExceed) {
                    statementBuilder.append(", ");
                }
                numberExceed = true;
                statementBuilder.append(currentMediumSeverity).append(" Medium severity vulnerabilities");
            }
        }

        if (foundName) {
            if (numberExceed) {
                statementBuilder.append(", and ");
            }
            numberExceed = true;
            statementBuilder.append("vulnerabilities: ").append(namesToPresent.toString());
        }

        logger.println("");
        logger.println("****************** NeuVector scan summary *******************");

        StringBuilder messageBuilder = new StringBuilder();
        if(!config.isLocal()){
            messageBuilder.append("Registry URL: ").append(config.getRegistry().getRegUrl()).append(", ");
        }

        messageBuilder.append("Repository: ").append(config.getRepository()).append(", ")
                .append("Tag: ").append(config.getTag()).append(", ")
                .append("Total vulnerabilities: ").append(currentHighSeverity + currentMediumSeverity).append(", ")
                .append("High severity vulnerabilities: ").append(currentHighSeverity).append(", ")
                .append("Medium severity vulnerabilities: ").append(currentMediumSeverity).toString();
        logger.println(messageBuilder.toString());

        if (numberExceed || foundName) {
            statementBuilder.append(" are present.");
            String causeMessage = "Build failed. It doesn't meet the required criteria because " + statementBuilder;
            logger.println(causeMessage);
            throw new VulnerabilityCriterionException(causeMessage);
        }
    }

    private String generateScannerName(){
        String saltChars = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
        StringBuilder salt = new StringBuilder();
        Random rnd = new Random();
        while (salt.length() < 6) { // length of the random string.
            int index = (int) (rnd.nextFloat() * saltChars.length());
            salt.append(saltChars.charAt(index));
        }
        return salt.toString();
    }
}
