package io.jenkins.plugins.neuvector;

import com.google.common.base.Strings;
import hudson.Extension;
import hudson.model.AbstractDescribableImpl;
import hudson.model.Descriptor;
import hudson.util.Secret;
import java.io.Serializable;

import hudson.util.FormValidation;
import org.kohsuke.stapler.DataBoundConstructor;
import org.kohsuke.stapler.QueryParameter;
import org.kohsuke.stapler.verb.POST;
import java.util.Iterator;
import java.util.List;

import java.net.URI;
import java.net.URISyntaxException;
import jenkins.model.Jenkins;

public class ControllerEndpointUrl extends AbstractDescribableImpl<ControllerEndpointUrl> implements Serializable {

    private static final long serialVersionUID = 1L;
    private String nickname;
    private String controllerEndpointUrl;
    private String protocol;
    private String host;
    private Integer port;
    private String subpath;

    private String controllerUser;
    private Secret controllerPassword;
    private Integer timeout;
    private Boolean disableTLSCertVerification;
    private Boolean disableAPIKeyVerification;
    private String serverCertificate;
    private Secret serverApiKey;

    @DataBoundConstructor
    public ControllerEndpointUrl(String nickname, String controllerEndpointUrl, String controllerUser, Secret controllerPassword, Boolean disableTLSCertVerification, Boolean disableAPIKeyVerification, String serverCertificate, Integer timeout, Secret serverApiKey) throws URISyntaxException {
        this.nickname = nickname.trim();
        URI uri = new URI(controllerEndpointUrl);
        this.controllerEndpointUrl = controllerEndpointUrl;
        this.protocol = uri.getScheme();
        this.host = uri.getHost();
        this.port = (uri.getPort() == -1) ? 443 : uri.getPort();
        this.subpath = uri.getPath();     

        this.controllerUser = controllerUser.trim();
        this.controllerPassword = controllerPassword;
        this.timeout = (timeout == null) ? Integer.valueOf(0) : timeout;
        this.disableTLSCertVerification = disableTLSCertVerification;
        this.disableAPIKeyVerification = disableAPIKeyVerification;
        this.serverCertificate = serverCertificate;
        this.serverApiKey = serverApiKey;
    }

    public String getControllerEndpointUrl() {
        return controllerEndpointUrl;
    }
    
    public void setControllerEndpointUrl(String controllerEndpointUrl) {
        this.controllerEndpointUrl = controllerEndpointUrl;
    }
    
    public String getNickname() {
        return nickname;
    }
    
    public void setNickname(String nickname) {
        this.nickname = nickname;
    }
    
    public String getProtocol() {
        return protocol;
    }
    
    public void setProtocol(String protocol) {
        this.protocol = protocol;
    }
    
    public String getHost() {
        return host;
    }
    
    public void setHost(String host) {
        this.host = host;
    }
    
    public Integer getPort() {
        return port;
    }
    
    public void setPort(Integer port) {
        this.port = port;
    }
    
    public String getSubpath() {
        return subpath;
    }
    
    public void setSubpath(String subpath) {
        this.subpath = subpath;
    }
    
    public String getControllerUser() {
        return controllerUser;
    }
    
    public void setControllerUser(String controllerUser) {
        this.controllerUser = controllerUser;
    }
    
    public Secret getControllerPassword() {
        return controllerPassword;
    }
    
    public void setControllerPassword(Secret controllerPassword) {
        this.controllerPassword = controllerPassword;
    }
    
    public Boolean getDisableTLSCertVerification() {
        return disableTLSCertVerification;
    }
    
    public void setDisableTLSCertVerification(Boolean disableTLSCertVerification) {
        this.disableTLSCertVerification = disableTLSCertVerification;
    }

    public Boolean getDisableAPIKeyVerification() {
        return disableAPIKeyVerification;
    }

    public void setDisableAPIKeyVerification(Boolean disableAPIKeyVerification) {
        this.disableAPIKeyVerification = disableAPIKeyVerification;
    }

    public String getServerCertificate() {
        return serverCertificate;
    }
    
    public void setServerCertificate(String serverCertificate) {
        this.serverCertificate = serverCertificate;
    }
    
    public Secret getServerApiKey() {
        return serverApiKey;
    }
    
    public void setServerApiKey(Secret serverApiKey) {
        this.serverApiKey = serverApiKey;
    }

    public Integer getTimeout() {
        return timeout;
    }
    
    public void setTimeout(Integer timeout) {
        this.timeout = timeout;
    }

    @Override
    public String toString() {
        StringBuilder apiUrlBuilder = new StringBuilder();
        apiUrlBuilder.append(protocol).append("://").append(host).append(":").append(port).append(subpath);
        return apiUrlBuilder.toString();
    }

    @Extension
    public static class DescriptorImpl extends Descriptor<ControllerEndpointUrl> {
        private final NeuVectorBuilder.DescriptorImpl globalConfig = Jenkins.get().getDescriptorByType(NeuVectorBuilder.DescriptorImpl.class);
        @Override
        public String getDisplayName() {
            return "Add ControllerEndpointUrl";
        }

        public DescriptorImpl() {
            super.load();
        }

        public FormValidation doCheckNickname(@QueryParameter String nickname) {
            if (Strings.isNullOrEmpty(nickname) || nickname.trim().isEmpty()) {
                return FormValidation.error(Messages.Registry_DescriptorImpl_errors_nickname());
            }
            return FormValidation.ok();
        }

        public FormValidation doCheckControllerEndpointUrl(@QueryParameter String value) {
            if (value != null && !value.trim().isEmpty() && ! value.trim().matches("^https://([a-zA-Z0-9.-]+)(:([0-9]+))?(/[a-zA-Z0-9./-]+)?$")) {
                return FormValidation.error(Messages.NeuVectorBuilder_DescriptorImpl_errors_controllerApiUrl());
            }
            return FormValidation.ok();
        }

        public FormValidation doCheckTimeout(@QueryParameter String value) {
            try {
                if (Integer.parseInt(value) < 0) {
                    return FormValidation.error(Messages.NeuVectorBuilder_DescriptorImpl_errors_timeout());
                }
                return FormValidation.ok();
            } catch (NumberFormatException e) {
                return FormValidation.error(Messages.NeuVectorBuilder_DescriptorImpl_errors_timeout());
            }
        }
        
        public FormValidation doDeleteRegistry(
            @QueryParameter String nickname,
            @QueryParameter String controllerEndpointUrl,
            @QueryParameter String controllerUser) {
            // Ensure user has permission
            Jenkins.get().checkPermission(Jenkins.ADMINISTER);
            List<ControllerEndpointUrl> controllerEndpointUrls = globalConfig.getControllerEndpointUrls();

            Iterator<ControllerEndpointUrl> iterator = controllerEndpointUrls.iterator();
            while (iterator.hasNext()) {
                ControllerEndpointUrl endpoint = iterator.next();
                if (endpoint.getNickname().equals(nickname)) {
                    iterator.remove();
                    // Optionally, save the updated list to persist changes
                    globalConfig.save();
                }
            }
            return FormValidation.ok("Deleted successfully");
        }

        @POST
        public FormValidation doTestConnection(@QueryParameter("nickname") String nickname,
                                               @QueryParameter("controllerEndpointUrl") String controllerEndpointUrl,
                                               @QueryParameter("controllerUser") String controllerUser,
                                               @QueryParameter("controllerPassword") Secret controllerPassword,
                                               @QueryParameter("disableTLSCertVerification") Boolean disableTLSCertVerification,
                                               @QueryParameter("disableAPIKeyVerification") Boolean disableAPIKeyVerification,
                                               @QueryParameter("serverCertificate") String serverCertificate,
                                               @QueryParameter("serverApiKey") Secret serverApiKey,
                                               @QueryParameter("uuid") String uuid) throws URISyntaxException {
            Jenkins.get().checkPermission(Jenkins.ADMINISTER);
            Config config = new Config.ConfigBuilder(nickname, controllerEndpointUrl, controllerUser, controllerPassword, disableTLSCertVerification, disableAPIKeyVerification, serverCertificate, 0, serverApiKey).build();
            NeuVectorWorker worker = new NeuVectorWorker(null, config);

            try {
                // If disableAPIKeyVerification, NeuVector will use jwt token to communicate with controller, otherwise, it will use API key
                if (disableAPIKeyVerification) {
                    worker.testConnection();
                } else {
                    worker.testApiKey();
                }
                return FormValidation.ok("Connection Success");
            } catch (Exception e) {
                return FormValidation.error("Failed to connect " + controllerEndpointUrl + ". " + e.getMessage());
            }
        }
    }
}