/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.jira.cloud.jenkins.buildinfo.client;

import com.atlassian.jira.cloud.jenkins.buildinfo.client.BuildUpdateFailedException;
import com.atlassian.jira.cloud.jenkins.buildinfo.client.model.BuildApiResponse;
import com.atlassian.jira.cloud.jenkins.buildinfo.client.model.Builds;
import com.atlassian.jira.cloud.jenkins.buildinfo.client.model.JiraBuildInfo;
import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.IOException;
import java.io.NotSerializableException;
import java.util.Objects;
import java.util.Optional;
import javax.inject.Inject;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import okhttp3.ResponseBody;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BuildsApi {
    private static final Logger log = LoggerFactory.getLogger(BuildsApi.class);
    private static final String BUILDS_API_URL = "https://api.atlassian.com/jira/builds/0.1/cloud/%s/bulk";
    private static final MediaType JSON = MediaType.get((String)"application/json; charset=utf-8");
    private final OkHttpClient httpClient;
    private final ObjectMapper objectMapper;
    private String buildsApiEndpoint;

    @Inject
    public BuildsApi(OkHttpClient httpClient, ObjectMapper objectMapper) {
        this.httpClient = Objects.requireNonNull(httpClient);
        this.objectMapper = Objects.requireNonNull(objectMapper);
        this.buildsApiEndpoint = BUILDS_API_URL;
    }

    public Optional<BuildApiResponse> postBuildUpdate(String cloudId, String accessToken, String jiraSiteUrl, JiraBuildInfo jiraBuildInfo) {
        try {
            String buildsPayload = this.objectMapper.writeValueAsString((Object)new Builds(jiraBuildInfo));
            Request request = this.getRequest(cloudId, accessToken, buildsPayload);
            Response response = this.httpClient.newCall(request).execute();
            this.checkForErrorResponse(jiraSiteUrl, response);
            return Optional.ofNullable(this.handleResponseBody(jiraSiteUrl, response));
        }
        catch (NotSerializableException e) {
            this.handleError(String.format("Empty body when submitting builds for %s", jiraSiteUrl));
        }
        catch (JsonParseException | JsonMappingException e) {
            this.handleError(String.format("Invalid JSON when submitting builds for %s", jiraSiteUrl));
        }
        catch (JsonProcessingException e) {
            this.handleError("Unable to build the request payload for Builds API: " + e.getMessage());
        }
        catch (IOException e) {
            this.handleError(String.format("Server exception when submitting builds for %s: %s", jiraSiteUrl, e.getMessage()));
        }
        return Optional.empty();
    }

    private void checkForErrorResponse(String jiraSiteUrl, Response response) {
        if (!response.isSuccessful()) {
            String message = String.format("Error response code %d when submitting builds for %s", response.code(), jiraSiteUrl);
            ResponseBody responseBody = response.body();
            if (responseBody != null) {
                log.error(String.format("Error response body when submitting builds for %s: %s", jiraSiteUrl, responseBody));
            }
            this.handleError(message);
        }
    }

    private BuildApiResponse handleResponseBody(String jiraSiteUrl, Response response) throws IOException {
        BuildApiResponse buildApiResponse;
        if (response.body() == null) {
            String message = String.format("Empty response body when submitting builds for %s", jiraSiteUrl);
            this.handleError(message);
        }
        if (!(buildApiResponse = (BuildApiResponse)this.objectMapper.readValue(response.body().bytes(), BuildApiResponse.class)).getRejectedBuilds().isEmpty()) {
            log.warn(String.format("Rejected builds when submitting builds for %s: %s", jiraSiteUrl, buildApiResponse.getRejectedBuilds()));
        }
        if (!buildApiResponse.getAcceptedBuilds().isEmpty()) {
            log.info(String.format("Accepted builds when submitting builds for %s: %s", jiraSiteUrl, buildApiResponse.getAcceptedBuilds()));
        }
        if (!buildApiResponse.getUnknownIssueKeys().isEmpty()) {
            log.warn(String.format("Unknown issue keys when submitting builds for %s: %s", jiraSiteUrl, buildApiResponse.getUnknownIssueKeys()));
        }
        return buildApiResponse;
    }

    void setBuildsApiEndpoint(String buildsApiEndpoint) {
        this.buildsApiEndpoint = buildsApiEndpoint;
    }

    private Request getRequest(String cloudId, String accessToken, String buildsPayload) {
        RequestBody body = RequestBody.create((MediaType)JSON, (String)buildsPayload);
        return new Request.Builder().url(String.format(this.buildsApiEndpoint, cloudId)).addHeader("Authorization", "Bearer " + accessToken).post(body).build();
    }

    private void handleError(String message) {
        log.error(message);
        throw new BuildUpdateFailedException(message);
    }
}

