/*
Copyright (C) 2016-2024 Sysdig

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
package com.sysdig.jenkins.plugins.sysdig.application.vm.report;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.sysdig.jenkins.plugins.sysdig.domain.vm.scanresult.Package;
import com.sysdig.jenkins.plugins.sysdig.domain.vm.scanresult.ScanResult;
import com.sysdig.jenkins.plugins.sysdig.domain.vm.scanresult.Vulnerability;
import edu.umd.cs.findbugs.annotations.NonNull;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;

public class VulnerabilityReportProcessor {

    public static JsonObject generateVulnerabilityReport(ScanResult result) {
        var securityJson = new JsonObject();
        JsonArray columnsJson = new JsonArray();

        for (String column : Arrays.asList(
                "Tag",
                "CVE ID",
                "Severity",
                "Vulnerability Package",
                "Fix Available",
                "URL",
                "Package Type",
                "Package Path",
                "Disclosure Date",
                "Solution Date")) {
            JsonObject columnJson = new JsonObject();
            columnJson.addProperty("title", column);
            columnsJson.add(columnJson);
        }

        securityJson.add("columns", columnsJson);

        JsonArray dataJson = new JsonArray();
        dataJson.addAll(getVulnerabilitiesArray(
                result.metadata().pullString(),
                result.packages().stream()
                        .sorted(Comparator.comparing(Package::name))
                        .toList()));
        securityJson.add("data", dataJson);
        return securityJson;
    }

    protected static JsonArray getVulnerabilitiesArray(@NonNull String tag, @NonNull List<Package> vulList) {
        var dataJson = new JsonArray();
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");

        for (Package packageJson : vulList) {
            for (Vulnerability vulnerability : packageJson.vulnerabilities()) {
                var vulnArray = new JsonArray();
                vulnArray.add(tag);
                vulnArray.add(vulnerability.cve());
                vulnArray.add(vulnerability.severity().toString());
                vulnArray.add(packageJson.name());
                vulnArray.add(vulnerability.fixVersion().orElse("None"));
                vulnArray.add("");
                vulnArray.add(packageJson.type().toString());
                vulnArray.add(packageJson.path());
                vulnArray.add(dateFormat.format(vulnerability.disclosureDate()));
                vulnArray.add(
                        vulnerability.solutionDate().map(dateFormat::format).orElse("None"));
                dataJson.add(vulnArray);
            }
        }

        return dataJson;
    }
}
