/*
 * Decompiled with CFR 0.152.
 */
package io.jenkins.plugins.artifactory_artifacts;

import com.cloudbees.plugins.credentials.Credentials;
import com.cloudbees.plugins.credentials.CredentialsProvider;
import com.cloudbees.plugins.credentials.common.UsernamePasswordCredentials;
import edu.umd.cs.findbugs.annotations.NonNull;
import io.jenkins.plugins.artifactory_artifacts.Utils;
import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.stream.Collectors;
import org.jfrog.artifactory.client.Artifactory;
import org.jfrog.artifactory.client.ArtifactoryClientBuilder;
import org.jfrog.artifactory.client.DownloadableArtifact;
import org.jfrog.artifactory.client.ItemHandle;
import org.jfrog.artifactory.client.UploadableArtifact;
import org.jfrog.artifactory.client.model.AqlItemType;
import org.jfrog.artifactory.client.model.File;
import org.jfrog.filespecs.FileSpec;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class ArtifactoryClient
implements AutoCloseable {
    public static final Logger LOGGER = LoggerFactory.getLogger(ArtifactoryClient.class);
    private final ArtifactoryConfig config;
    private final Artifactory artifactory;

    public ArtifactoryClient(@NonNull String serverUrl, @NonNull String repository, @NonNull UsernamePasswordCredentials credentials) {
        this.config = new ArtifactoryConfig(serverUrl, repository, credentials);
        this.artifactory = this.buildArtifactory();
    }

    public ArtifactoryClient(@NonNull ArtifactoryConfig config) {
        this(config.getServerUrl(), config.getRepository(), config.getCredentials());
    }

    public void uploadArtifact(Path file, String targetPath) throws IOException {
        UploadableArtifact artifact = this.artifactory.repository(this.config.repository).upload(Utils.urlEncodeParts(targetPath), file.toFile());
        artifact.withSize(Files.size(file));
        artifact.withListener((bytesRead, totalBytes) -> LOGGER.trace(String.format("Uploaded %d/%d", bytesRead, totalBytes)));
        artifact.doUpload();
        LOGGER.trace(String.format("Uploaded %s to %s", file, targetPath));
    }

    public void deleteArtifact(String targetPath) {
        this.artifactory.repository(this.config.repository).delete(Utils.urlEncodeParts(targetPath));
    }

    public void move(String sourcePath, String targetPath) {
        ItemHandle sourceItem = this.artifactory.repository(this.config.repository).folder(Utils.urlEncodeParts(sourcePath));
        sourceItem.move(this.config.repository, Utils.urlEncodeParts(targetPath));
    }

    public void copy(String sourcePath, String targetPath) {
        ItemHandle sourceItem = this.artifactory.repository(this.config.repository).folder(Utils.urlEncodeParts(sourcePath));
        sourceItem.copy(this.config.repository, targetPath);
    }

    public InputStream downloadArtifact(String targetPath) throws IOException {
        DownloadableArtifact artifact = this.artifactory.repository(this.config.repository).download(Utils.urlEncodeParts(targetPath));
        return artifact.doDownload();
    }

    public boolean isFolder(String targetPath) throws IOException {
        try {
            return this.artifactory.repository(this.config.repository).isFolder(Utils.urlEncodeParts(targetPath));
        }
        catch (Exception e) {
            LOGGER.debug(String.format("Failed to check if %s is a folder", targetPath));
            return false;
        }
    }

    public List<FileInfo> list(String targetPath) throws IOException {
        if (!this.isFolder(targetPath)) {
            LOGGER.debug(String.format("Target path %s is not a folder. Cannot list files", targetPath));
            return List.of();
        }
        FileSpec fileSpec = FileSpec.fromString((String)String.format("{\"files\": [{\"pattern\": \"%s/%s*\"}]}", this.config.repository, targetPath));
        return this.artifactory.searches().artifactsByFileSpec(fileSpec).stream().map(item -> new FileInfo(String.format("%s/%s", item.getPath(), item.getName()), item.getModified().getTime(), item.getSize(), item.getType())).collect(Collectors.toList());
    }

    public boolean isFile(String targetPath) throws IOException {
        if (this.isFolder(targetPath)) {
            return false;
        }
        try {
            File file = (File)this.artifactory.repository(this.config.repository).file(Utils.urlEncodeParts(targetPath)).info();
            return !file.isFolder();
        }
        catch (Exception e) {
            LOGGER.debug(String.format("Failed to check if %s is a file", targetPath));
            return false;
        }
    }

    public long lastUpdated(String targetPath) throws IOException {
        LOGGER.trace(String.format("Getting last updated time for %s", targetPath));
        return this.artifactory.repository(this.config.repository).file(targetPath).info().getLastModified().getTime();
    }

    public long size(String targetPath) throws IOException {
        if (this.isFolder(targetPath)) {
            return 0L;
        }
        LOGGER.trace(String.format("Getting size for %s", targetPath));
        File file = (File)this.artifactory.repository(this.config.repository).file(Utils.urlEncodeParts(targetPath)).info();
        return file.getSize();
    }

    public ArtifactoryConfig buildArtifactoryConfig() {
        return new ArtifactoryConfig(this.config.serverUrl, this.config.repository, this.config.credentials);
    }

    private Artifactory buildArtifactory() {
        return ArtifactoryClientBuilder.create().setUrl(this.config.serverUrl).setUsername(this.config.credentials.getUsername()).setPassword(this.config.credentials.getPassword().getPlainText()).addInterceptorLast((request, httpContext) -> LOGGER.debug(String.format("Sending Artifactory request to %s", request.getRequestLine()))).build();
    }

    @Override
    public void close() throws Exception {
        this.artifactory.close();
    }

    public static final class ArtifactoryConfig
    implements Serializable {
        private static final long serialVersionUID = 1L;
        private final String serverUrl;
        private final String repository;
        private final UsernamePasswordCredentials credentials;
        private final int maxUploadRetries;
        private final int retryDelaySeconds;

        public ArtifactoryConfig(String serverUrl, String repository, UsernamePasswordCredentials credentials) {
            this(serverUrl, repository, credentials, 0, 5);
        }

        public ArtifactoryConfig(String serverUrl, String repository, UsernamePasswordCredentials credentials, int maxUploadRetries, int retryDelaySeconds) {
            this.serverUrl = serverUrl;
            this.repository = repository;
            this.credentials = (UsernamePasswordCredentials)CredentialsProvider.snapshot(UsernamePasswordCredentials.class, (Credentials)credentials);
            this.maxUploadRetries = maxUploadRetries;
            this.retryDelaySeconds = retryDelaySeconds;
        }

        public String getServerUrl() {
            return this.serverUrl;
        }

        public String getRepository() {
            return this.repository;
        }

        public UsernamePasswordCredentials getCredentials() {
            return this.credentials;
        }

        public int getMaxUploadRetries() {
            return this.maxUploadRetries;
        }

        public int getRetryDelaySeconds() {
            return this.retryDelaySeconds;
        }
    }

    public static final class FileInfo
    implements Serializable {
        private static final long serialVersionUID = 1L;
        private final String path;
        private final long lastUpdated;
        private final long size;
        private final AqlItemType type;

        public FileInfo(String path, long lastUpdated, long size, AqlItemType type) {
            this.path = path;
            this.lastUpdated = lastUpdated;
            this.size = size;
            this.type = type;
        }

        public String getPath() {
            return this.path;
        }

        public long getLastUpdated() {
            return this.lastUpdated;
        }

        public long getSize() {
            return this.size;
        }

        public boolean isDirectory() {
            return this.type.equals((Object)AqlItemType.FOLDER);
        }

        public boolean isFile() {
            return this.type.equals((Object)AqlItemType.FILE);
        }
    }
}

