package org.jenkinsci.gradle.plugins.testing

import com.squareup.javapoet.JavaFile
import com.squareup.javapoet.MethodSpec
import com.squareup.javapoet.ParameterizedTypeName
import com.squareup.javapoet.TypeSpec
import junit.framework.Test
import junit.framework.TestCase
import org.gradle.workers.WorkAction
import org.jvnet.hudson.test.PluginAutomaticTestBuilder
import javax.lang.model.element.Modifier

abstract class GenerateTest : WorkAction<GenerateTestParameters> {
    private companion object {
        private const val DEFAULT_PKG_NAME = "org.jenkinsci.plugins.generated"
        private const val INDENT = "    "
        private val mapStringString = ParameterizedTypeName.get(Map::class.java, String::class.java, String::class.java)
    }

    override fun execute() {
        val testParameters = parameters.testParameters.get()
        val method = MethodSpec.methodBuilder("suite")
                .addModifiers(Modifier.PUBLIC, Modifier.STATIC)
                .returns(Test::class.java)
                .addException(Exception::class.java)
                .addStatement("${'$'}T parameters = new ${'$'}T<>()", mapStringString, HashMap::class.java)
                .apply {
                    for ((key, value) in testParameters.entries) {
                        addStatement("parameters.put(${'$'}S, ${'$'}S)", key, value)
                    }
                }
                .addStatement("return ${'$'}T.build(parameters)", PluginAutomaticTestBuilder::class.java)
                .build()
        val requestedName = parameters.testName.get()
        val test = TypeSpec.classBuilder(requestedName.substringAfterLast('.'))
                .addJavadoc("Entry point to auto-generated tests. Generated by the " +
                        "generateJenkinsTests task in org.jenkins-ci.jpi plugin.")
                .addModifiers(Modifier.PUBLIC)
                .superclass(TestCase::class.java)
                .addMethod(method)
                .build()

        val pkg = requestedName.substringBeforeLast('.', "").let {
            if (it.isNotEmpty()) it else "$DEFAULT_PKG_NAME.${parameters.pluginId.get().replace('-', '_')}"
        }
        val file = JavaFile.builder(pkg, test)
                .skipJavaLangImports(true)
                .indent(INDENT)
                .build()

        file.writeTo(parameters.outputDir.asFile.get())
    }
}
