/*
 * Jenkins Sonargraph Integration Plugin
 * Copyright (C) 2015-2025 hello2morrow GmbH
 * mailto: support AT hello2morrow DOT com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.hello2morrow.sonargraph.integration.jenkins.controller;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.kohsuke.stapler.StaplerRequest2;
import org.kohsuke.stapler.StaplerResponse2;

import com.hello2morrow.sonargraph.integration.jenkins.persistence.MetricIds;

import hudson.model.Job;
import hudson.model.ProminentProjectAction;
import jenkins.model.JenkinsLocationConfiguration;

/**
 * This object is the responsible of managing the action that will take the user to see the graphs generated by sonargraph.
 * 
 * @author esteban
 *
 */
public class SonargraphIntegrationAction implements ProminentProjectAction
{
    /** Job calling this action. */
    private final Job<?, ?> job;

    private final MetricIds metaData;
    private final List<String> selectedMetrics;

    public SonargraphIntegrationAction(final Job<?, ?> job, final List<String> selectedMetrics, final MetricIds metricMetaData)
    {
        this.job = job;
        this.metaData = metricMetaData;
        this.selectedMetrics = selectedMetrics;
    }

    public Collection<String> getChartsForMetrics()
    {
        final List<String> chartsForMetrics = new ArrayList<>();

        for (final String metric : selectedMetrics)
        {
            chartsForMetrics.add(metric);
        }

        return chartsForMetrics;
    }

    /**
     * Method that generates the chart and adds it to the response object to allow jenkins to display it. It is called in
     * SonargraphChartAction/index.jelly in the src attribute of an img tag.
     */
    public void doGetPlot(final StaplerRequest2 req, final StaplerResponse2 rsp)
    {
        Plotter.doGetPlot(job, metaData, req, rsp);
    }

    public Job<?, ?> getJob()
    {
        return job;
    }

    /**
     * Icon that will appear next to the link defined by this action.
     */
    @Override
    public String getIconFileName()
    {
        return ConfigParameters.SONARGRAPH_ICON.getValue();
    }

    /**
     * Name of the link for this action
     */
    @Override
    public String getDisplayName()
    {
        return ConfigParameters.ACTION_DISPLAY_INTEGRATION.getValue();
    }

    /**
     * Last segment of the url that will lead to this action. e.g https://localhost:8080/jobName/sonargraph
     */
    @Override
    public String getUrlName()
    {
        return ConfigParameters.ACTION_URL_INTEGRATION.getValue();
    }

    /**
     * Report URL is used as html href for "Show most recent Sonargraph Report" in SonargraphIntegrationAction/index.jelly. Depends on global Jenkins URL
     * being set correctly in Jenkins. Otherwise a relative path is used, which may work or not.
     * 
     * @return the absolute or relative url/path for the most recent sonargraph report.
     */
    public String getReportURL()
    {
        return getURL(ConfigParameters.ACTION_URL_REPORT);
    }
    
    /**
     * Diff URL is used as html href for "Show most recent Sonargraph Diff" in SonargraphIntegrationAction/index.jelly. Depends on global Jenkins URL
     * being set correctly in Jenkins. Otherwise a relative path is used, which may work or not.
     * 
     * @return the absolute or relative url/path for the most recent sonargraph diff.
     */
    public String getDiffURL()
    {
        return getURL(ConfigParameters.ACTION_URL_DIFF);
    }
    
    private String getURL(ConfigParameters lastPart)
    {
        final JenkinsLocationConfiguration globalConfig = JenkinsLocationConfiguration.get();
        final String url = globalConfig.getUrl();
        final String projectName = job.getFullName();

        if (url != null)
        {
            return url + ConfigParameters.JOB_FOLDER.getValue() + projectName + "/" + lastPart.getValue();
        }
        else
        {
            return "../" + lastPart.getValue();
        } 
    }
}