/*
 * The MIT License
 * 
 * Copyright (c) 2015 Stefan Brausch, Jochen A. Fuerbacher
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package hudson.plugins.build_timeout.operations;

import java.io.IOException;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.kohsuke.stapler.DataBoundConstructor;

import hudson.Extension;
import hudson.model.AbstractBuild;
import hudson.model.AbstractProject;
import hudson.model.BuildListener;
import hudson.plugins.build_timeout.BuildTimeOutOperation;
import hudson.plugins.build_timeout.BuildTimeOutOperationDescriptor;

import com.chikli.hudson.plugin.naginator.FixedDelay;

/**
 * Abort the build.
 */
public class AbortAndRestartOperation extends BuildTimeOutOperation {
    
    private final String maxRestarts;
    
    private static final Logger log = Logger.getLogger(AbortAndRestartOperation.class.getName());
    
    /**
     * @return max restarts.
     */
    public String getMaxRestarts() {
        return maxRestarts;
    }
        
    @DataBoundConstructor
    public AbortAndRestartOperation(String maxRestarts){
        this.maxRestarts = maxRestarts;
    }
    
    private static boolean isPresent() {
        try {
            Class.forName("com.chikli.hudson.plugin.naginator.NaginatorScheduleAction");
            return true;
        } catch (ClassNotFoundException ex) {
            log.log(Level.FINEST, "Naginator not available. ", ex);
            return false;
        }
    }
    
    private void rescheduleBuild(AbstractBuild<?, ?> build, BuildListener listener) {
        FixedDelay sd = new FixedDelay(0); //Reschedule now!
        
        String maxRestartsStr = null;
        try {
            maxRestartsStr = build.getEnvironment(listener).expand(this.maxRestarts);
        } catch (IOException | InterruptedException e) {
            listener.error("Failed to expand environment variables.");
            e.printStackTrace(listener.getLogger());
            return;
        }

        int maxRestarts = 0;
        try {
            maxRestarts = Integer.parseInt(maxRestartsStr);
        } catch (NumberFormatException e) {
            listener.error("Invalid Maximum restarts: {0}", maxRestartsStr);
            e.printStackTrace(listener.getLogger());
            return;
        }
        build.addAction(new com.chikli.hudson.plugin.naginator.NaginatorScheduleAction(maxRestarts, sd, false));
        listener.getLogger().println(Messages.AbortAndRestartOperation_ScheduledRestart(maxRestarts));
    }
    
    /**
     * @param build
     * @param listener
     * @param effectiveTimeout
     * @return
     * @see hudson.plugins.build_timeout.BuildTimeOutOperation#perform(hudson.model.AbstractBuild, hudson.model.BuildListener, long)
     */
    @Override
    public boolean perform(AbstractBuild<?, ?> build, BuildListener listener, long effectiveTimeout) {
        if (isPresent()) {
            rescheduleBuild(build, listener);
        } else {
            listener.error(Messages.AbortAndRestartOperation_InstallNaginator());
        }
        
        return new AbortOperation().perform(build, listener, effectiveTimeout);
    }
   
    @Extension(optional = true)
    public static class DescriptorImpl extends BuildTimeOutOperationDescriptor {
        @Override
        public String getDisplayName() {
            return Messages.AbortAndRestartOperation_DisplayName();
        }
        
        @Override
        public boolean isApplicable(Class<? extends AbstractProject<?,?>> jobType) {
            return AbortAndRestartOperation.isPresent();
        }
    }
}
