package org.jenkinsci.plugins.azurekeyvaultplugin;

import com.azure.security.keyvault.secrets.SecretClient;
import com.cloudbees.plugins.credentials.Credentials;
import com.cloudbees.plugins.credentials.CredentialsScope;
import com.cloudbees.plugins.credentials.SystemCredentialsProvider;
import com.cloudbees.plugins.credentials.common.IdCredentials;
import com.microsoft.azure.util.AzureBaseCredentials;
import com.microsoft.azure.util.AzureCredentials;
import com.microsoft.azure.util.AzureImdsCredentials;
import com.microsoft.jenkins.keyvault.SecretClientCache;
import hudson.Extension;
import hudson.ExtensionList;
import hudson.model.Item;
import hudson.util.FormValidation;
import hudson.util.ListBoxModel;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Optional;
import java.util.logging.Level;
import java.util.logging.Logger;
import jenkins.model.GlobalConfiguration;
import jenkins.model.Jenkins;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.jenkinsci.Symbol;
import org.kohsuke.stapler.AncestorInPath;
import org.kohsuke.stapler.DataBoundSetter;
import org.kohsuke.stapler.QueryParameter;
import org.kohsuke.stapler.verb.POST;

@Extension
@Symbol("azureKeyVault")
public class AzureKeyVaultGlobalConfiguration extends GlobalConfiguration {

    public static final String GENERATED_ID = "azure-keyvault-autogenerated";
    private static final Logger LOG = Logger.getLogger(AzureKeyVaultGlobalConfiguration.class.getName());
    private static final String GENERATED_DESCRIPTION = "Auto generated credential from environment";
    private String keyVaultURL;
    private String credentialID;

    public AzureKeyVaultGlobalConfiguration() {
        load();
    }

    public String getKeyVaultURL() {
        resolveKeyVaultUrlFromEnvironment()
                .ifPresent(url -> {
                    this.keyVaultURL = url;
                    save();
                });

        return keyVaultURL;
    }

    @DataBoundSetter
    public void setKeyVaultURL(String keyVaultURL) {
        this.keyVaultURL = keyVaultURL;
        save();
        refresh();
    }

    private void refresh() {
        AzureCredentialsProvider azureCredentialsProvider = ExtensionList.lookupSingleton(AzureCredentialsProvider.class);
        azureCredentialsProvider.refreshCredentials();
    }

    public String getCredentialID() {
        resolveCredentialIdFromEnvironment()
                .ifPresent(id -> {
                    this.credentialID = id;
                    save();
                });

        return this.credentialID;
    }

    private Optional<String> resolveKeyVaultUrlFromEnvironment() {
        Optional<String> url = getPropertyByEnvOrSystemProperty("AZURE_KEYVAULT_URL", "jenkins.azure-keyvault.url");

        if (url.isPresent() && url.get().equals(this.keyVaultURL)) {
            // don't overwrite the url if it matches what we currently have so as we don't save to disk all the time
            return Optional.empty();
        }

        return url;
    }

    private Optional<String> resolveCredentialIdFromEnvironment() {

        // directly lookup the credential so that we don't get a stackoverflow due to credential provider
        Optional<Credentials> optionalCredentials = SystemCredentialsProvider.getInstance()
                .getCredentials()
                .stream()
                .filter(credentials -> (credentials instanceof AzureCredentials || credentials instanceof AzureImdsCredentials) && ((IdCredentials) credentials).getId().equals(GENERATED_ID))
                .findAny();

        boolean isUami = getPropertyByEnvOrSystemProperty("AZURE_KEYVAULT_UAMI_ENABLED", "jenkins.azure-keyvault.uami.enabled")
                .map(u -> "true".equals(u))
                .orElse(false);

        CredentialsScope scope = getPropertyByEnvOrSystemProperty("AZURE_KEYVAULT_SP_SCOPE", "jenkins.azure-keyvault.sp.scope")
                .map(s -> "SYSTEM".equalsIgnoreCase(s) ? CredentialsScope.SYSTEM : CredentialsScope.GLOBAL)
                .orElse(CredentialsScope.GLOBAL);

        AzureBaseCredentials credentials;
        if (isUami) {
            if (optionalCredentials.filter(c -> c instanceof AzureImdsCredentials && scope.equals(c.getScope())).isPresent()) {
                // don't overwrite the credential if it matches what we currently have so as we don't save to disk all the time
                return Optional.empty();
            }

            credentials = new AzureImdsCredentials(
                    scope, GENERATED_ID, GENERATED_DESCRIPTION
            );
            storeCredential(credentials);
            return Optional.of(credentials.getId());
        }

        String clientId = getPropertyByEnvOrSystemProperty("AZURE_KEYVAULT_SP_CLIENT_ID", "jenkins.azure-keyvault.sp.client_id")
                .orElse("false");
        if (clientId.equals("false")) {
            return Optional.empty();
        }

        String clientSecret = getPropertyByEnvOrSystemProperty("AZURE_KEYVAULT_SP_CLIENT_SECRET", "jenkins.azure-keyvault.sp.client_secret")
                .orElseGet(() -> getPropertyByFile("AZURE_KEYVAULT_SP_CLIENT_SECRET_FILE", "jenkins.azure-keyvault.sp.client_secret_file")
                        .orElseThrow(IllegalArgumentException::new));
        String subscriptionId = getPropertyByEnvOrSystemProperty("AZURE_KEYVAULT_SP_SUBSCRIPTION_ID", "jenkins.azure-keyvault.sp.subscription_id")
                .orElseThrow(IllegalArgumentException::new);
        String tenantId = getPropertyByEnvOrSystemProperty("AZURE_KEYVAULT_SP_TENANT_ID", "jenkins.azure-keyvault.sp.tenant_id")
                .orElseThrow(IllegalArgumentException::new);

        if (optionalCredentials.isPresent() && optionalCredentials.get() instanceof AzureCredentials &&
                azureCredentialIsEqual(
                        (AzureCredentials) optionalCredentials.get(),
                        clientId,
                        clientSecret,
                        subscriptionId,
                        tenantId,
                        scope)
        ) {
            // don't overwrite the credential if it matches what we currently have so as we don't save to disk all the time
            return Optional.empty();
        }

        AzureCredentials azureCredentials = new AzureCredentials(scope, GENERATED_ID, GENERATED_DESCRIPTION, subscriptionId, clientId, clientSecret);
        azureCredentials.setTenant(tenantId);

        storeCredential(azureCredentials);
        return Optional.of(azureCredentials.getId());
    }

    private boolean azureCredentialIsEqual(AzureCredentials creds, String clientId, String clientSecret, String subscriptionId, String tenantId, CredentialsScope scope) {
        return StringUtils.equals(creds.getClientId(), clientId) &&
                StringUtils.equals(creds.getPlainClientSecret(), clientSecret) &&
                StringUtils.equals(creds.getSubscriptionId(), subscriptionId) &&
                StringUtils.equals(creds.getTenant(), tenantId) &&
                ObjectUtils.equals(creds.getScope(), scope);
    }

    private Optional<String> getPropertyByEnvOrSystemProperty(String envVariable, String systemProperty) {
        String envResult = System.getenv(envVariable);
        if (envResult != null) {
            return Optional.of(envResult);
        }

        String systemResult = System.getProperty(systemProperty);
        if (systemResult != null) {
            return Optional.of(systemResult);
        }

        return Optional.empty();
    }

    private Optional<String> getPropertyByFile(String envVariable, String systemProperty) {
        String envResult = System.getenv(envVariable);
        if (envResult != null) {
            Path pathToSecret = Paths.get(envResult);
            try {
                return Optional.of(Files.readAllLines(pathToSecret).get(0));
            } catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }

        String systemResult = System.getProperty(systemProperty);
        if (systemResult != null) {
            Path pathToSecret = Paths.get(systemResult);
            try {
                return Optional.of(Files.readAllLines(pathToSecret).get(0));
            } catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }

        return Optional.empty();
    }

    private void storeCredential(AzureBaseCredentials credentials) {
        SystemCredentialsProvider instance = SystemCredentialsProvider.getInstance();
        for (int i = 0; i < instance.getCredentials().size(); i++) {
            Credentials cred = instance.getCredentials().get(i);
            if (cred instanceof IdCredentials) {
                IdCredentials idCredentials = (IdCredentials) cred;
                if (idCredentials.getId().equals(credentials.getId())) {
                    instance.getCredentials().remove(i);
                    break;
                }

            }
        }

        instance.getCredentials().add(credentials);

        try {
            instance.save();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @DataBoundSetter
    public void setCredentialID(String credentialID) {
        this.credentialID = credentialID;
        save();
        refresh();
    }

    @POST
    public FormValidation doReloadCache() {
        Jenkins.get().checkPermission(Jenkins.ADMINISTER);

        if (StringUtils.isBlank(keyVaultURL)) {
            return FormValidation.error("Key vault url is required");
        }

        if (StringUtils.isBlank(credentialID)) {
            return FormValidation.error("Credential ID is required");
        }

        refresh();

        return FormValidation.ok("Cache reloaded");
    }

    @POST
    @SuppressWarnings("unused")
    public FormValidation doTestConnection(
            @QueryParameter("keyVaultURL") final String keyVaultURL,
            @QueryParameter("credentialID") final String credentialID
    ) {
        Jenkins.get().checkPermission(Jenkins.ADMINISTER);

        if (StringUtils.isBlank(keyVaultURL)) {
            return FormValidation.error("Key vault url is required");
        }

        if (StringUtils.isBlank(credentialID)) {
            return FormValidation.error("Credential ID is required");
        }

        try {
            SecretClient client = SecretClientCache.get(credentialID, keyVaultURL);

            Long numberOfSecrets = client.listPropertiesOfSecrets().stream().count();
            return FormValidation.ok(String.format("Success, found %d secrets in the vault", numberOfSecrets));
        } catch (RuntimeException e) {
            LOG.log(Level.WARNING, "Failed testing connection", e);
            return FormValidation.error(e, e.getMessage());
        }
    }

    @POST
    @SuppressWarnings("unused")
    public ListBoxModel doFillCredentialIDItems(@AncestorInPath Item context) {
        return AzureKeyVaultUtil.doFillCredentialIDItems(context);
    }

    public static AzureKeyVaultGlobalConfiguration get() {
        return ExtensionList.lookupSingleton(AzureKeyVaultGlobalConfiguration.class);
    }
}
