/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.logic;


import java.io.IOException;
import java.io.InputStream;
import java.util.function.BiPredicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.script.ScriptContext;
import javax.script.ScriptException;

import net.shibboleth.shared.annotation.ParameterName;
import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.collection.Pair;
import net.shibboleth.shared.component.ComponentInitializationException;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.resource.Resource;
import net.shibboleth.shared.scripting.AbstractScriptEvaluator;
import net.shibboleth.shared.scripting.EvaluableScript;

import org.slf4j.Logger;

/**
 * A {@link BiPredicate} which calls out to a supplied script.
 *
 * @param <T> first input type
 * @param <U> second input type
 * @since 8.2.0
 */
public class ScriptedBiPredicate<T,U> extends AbstractScriptEvaluator implements BiPredicate<T,U> {

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(ScriptedBiPredicate.class);

    /** Input type 1. */
    @Nullable private Pair<Class<T>,Class<U>> inputTypes;

    /**
     * Constructor.
     *
     * @param theScript the script we will evaluate.
     * @param extraInfo debugging information.
     */
    protected ScriptedBiPredicate(@Nonnull @NotEmpty final EvaluableScript theScript,
            @Nullable @NotEmpty final String extraInfo) {
        super(theScript);
        setOutputType(Boolean.class);
        setReturnOnError(false);
        setLogPrefix("Scripted BiPredicate from " + extraInfo + ":");
    }

    /**
     * Constructor.
     *
     * @param theScript the script we will evaluate.
     */
    protected ScriptedBiPredicate(@Nonnull @NotEmpty final EvaluableScript theScript) {
        super(theScript);
        setOutputType(Boolean.class);
        setReturnOnError(false);
        setLogPrefix("Anonymous BiPredicate:");
    }

    /**
     * Get the input type to be enforced.
     *
     * @return input type
     */
    @Nullable public Pair<Class<T>,Class<U>> getInputTypes() {
        return inputTypes;
    }

    /**
     * Set the input types to be enforced.
     *
     * @param types the input types
     */
    public void setInputTypes(@Nullable final Pair<Class<T>,Class<U>> types) {
        if (types != null && types.getFirst() != null && types.getSecond() != null) {
            inputTypes = types;
        } else {
            inputTypes = null;
        }
    }

    /**
     * Set value to return if an error occurs.
     * 
     * @param flag value to return
     */
    public void setReturnOnError(final boolean flag) {
        setReturnOnError(Boolean.valueOf(flag));
    }

    /** {@inheritDoc} */
    public boolean test(@Nullable final T first, @Nullable final U second) {
        
        final Pair<Class<T>,Class<U>> types = getInputTypes();
        if (null != types) {
            final Class<T> intype1 = types.getFirst();
            final Class<U> intype2 = types.getSecond();
            
            if (null != first && null != intype1 && !intype1.isInstance(first)) {
                log.error("{} Input of type {} was not of type {}", getLogPrefix(), first.getClass(), intype1);
                return (boolean) returnError();
            }
            if (null != second && null != intype2 && !intype2.isInstance(second)) {
                log.error("{} Input of type {} was not of type {}", getLogPrefix(), second.getClass(), intype2);
                return (boolean) returnError();
            }
        }

        final Object result = evaluate(first, second);
        return (boolean) (result != null ? result : returnError());
    }
    
    /**
     * Helper function to sanity check return-on-error object.
     * 
     * @return a boolean-valued error fallback
     * 
     * @throws ClassCastException if the installed fallback is null or non-Boolean
     */
    private boolean returnError() throws ClassCastException {
        final Object ret = getReturnOnError();
        if (ret instanceof Boolean) {
            return (boolean) ret;
        }
        
        throw new ClassCastException("Unable to cast return value to a boolean");
    }
    
    /** {@inheritDoc} */
    @Override
    protected void prepareContext(@Nonnull final ScriptContext scriptContext, @Nullable final Object... input) {
        scriptContext.setAttribute("input1", input != null ? input[0] : null, ScriptContext.ENGINE_SCOPE);
        scriptContext.setAttribute("input2", input != null ? input[1] : null, ScriptContext.ENGINE_SCOPE);
    }

    /**
     * Factory to create {@link ScriptedBiPredicate} from a {@link Resource}.
     *
     * @param <T> first input type
     * @param <U> second input type
     * @param resource the resource to look at
     * @param engineName the language
     * 
     * @return the function
     * 
     * @throws ScriptException if the compile fails
     * @throws IOException if the file doesn't exist.
     * @throws ComponentInitializationException if the scripting initialization fails
     */
    @Nonnull public static <T,U> ScriptedBiPredicate<T,U> resourceScript(
            @Nonnull @NotEmpty @ParameterName(name="engineName") final String engineName,
            @Nonnull @ParameterName(name="resource") final Resource resource)
                    throws ScriptException, IOException, ComponentInitializationException {
        try (final InputStream is = resource.getInputStream()) {
            final EvaluableScript script = new EvaluableScript();
            script.setEngineName(engineName);
            script.setScript(is);
            script.initialize();
            return new ScriptedBiPredicate<>(script, resource.getDescription());
        }
    }

    /**
     * Factory to create {@link ScriptedBiPredicate} from a {@link Resource}.
     *
     * @param <T> first input type
     * @param <U> second input type
     * @param resource the resource to look at
     * 
     * @return the function
     * 
     * @throws ScriptException if the compile fails
     * @throws IOException if the file doesn't exist.
     * @throws ComponentInitializationException if the scripting initialization fails
     */
    @Nonnull public static <T,U> ScriptedBiPredicate<T,U> resourceScript(
            @Nonnull @ParameterName(name="resource") final Resource resource)
                    throws ScriptException, IOException, ComponentInitializationException {
        return resourceScript(DEFAULT_ENGINE, resource);
    }

    /**
     * Factory to create {@link ScriptedBiPredicate} from inline data.
     *
     * @param <T> first input type
     * @param <U> second input type
     * @param scriptSource the script, as a string
     * @param engineName the language
     * 
     * @return the function
     * 
     * @throws ScriptException if the compile fails
     * @throws ComponentInitializationException if the scripting initialization fails
     */
    @Nonnull public static <T,U> ScriptedBiPredicate<T,U> inlineScript(
            @Nonnull @NotEmpty @ParameterName(name="engineName") final String engineName,
            @Nonnull @NotEmpty @ParameterName(name="scriptSource") final String scriptSource)
                    throws ScriptException, ComponentInitializationException {
        final EvaluableScript script = new EvaluableScript();
        script.setEngineName(engineName);
        script.setScript(scriptSource);
        script.initialize();
        return new ScriptedBiPredicate<>(script, "Inline");
    }

    /**
     * Factory to create {@link ScriptedBiPredicate} from inline data.
     *
     * @param <T> first input type
     * @param <U> second input type
     * @param scriptSource the script, as a string
     * 
     * @return the function
     * 
     * @throws ScriptException if the compile fails
     * @throws ComponentInitializationException if the scripting initialization fails
     */
    @Nonnull public static <T,U> ScriptedBiPredicate<T,U> inlineScript(
            @Nonnull @NotEmpty @ParameterName(name="scriptSource") final String scriptSource)
                    throws ScriptException, ComponentInitializationException {
        final EvaluableScript script = new EvaluableScript();
        script.setScript(scriptSource);
        script.initialize();
        return new ScriptedBiPredicate<>(script, "Inline");
    }
    
}