package com.atlassian.sal.api.pluginsettings;

/**
 * Factory for creating {@link PluginSettings} instances, each scoped to a namespace.
 * <p>
 * A namespace is a scope or container that isolates related settings. Within each namespace, individual settings are
 * stored using keys. Some products may restrict valid namespaces to product-specific entity identifiers.
 *
 * @since 2.0
 * @see PluginSettings
 */
public interface PluginSettingsFactory {

    /**
     * @deprecated since 7.2. The method name "createSettingsForKey" is misleading because the parameter represents a
     * namespace (scope), not a key. The returned {@link PluginSettings} instance defines methods like
     * {@link PluginSettings#get(String key)} where "key" refers to individual settings within that namespace. Use
     * {@link #createSettingsForNamespace(String)} instead, which clarifies this namespace vs. key distinction.
     */
    @Deprecated(forRemoval = true)
    default PluginSettings createSettingsForKey(String namespace) {
        return createSettingsForNamespace(namespace);
    }

    /**
     * Creates a {@link PluginSettings} instance scoped to a specific namespace.
     * <p>
     * A namespace acts as a scope or container for related settings. Within each namespace, you can store individual
     * settings using keys via methods like {@link PluginSettings#get(String)},
     * {@link PluginSettings#put(String, Object)}, etc.
     *
     * @param namespace the namespace scope for the settings. Can be {@code null} to retrieve the "global" namespace
     *                  (equivalent to {@link #createGlobalSettings()})
     * @return a {@link PluginSettings} instance scoped to the specified namespace
     * @throws IllegalArgumentException if the product does not support the specified namespace
     * @since 7.2
     */
    default PluginSettings createSettingsForNamespace(String namespace) {
        return createSettingsForKey(namespace);
    }

    /**
     * Creates a {@link PluginSettings} instance scoped to the global namespace.
     * <p>
     * The global namespace is shared across all components of the product. When storing settings in the global
     * namespace, callers should use specific, unique keys to avoid conflicts with other unrelated settings.
     *
     * @return a {@link PluginSettings} instance scoped to the global namespace
     */
    PluginSettings createGlobalSettings();
}
