package com.atlassian.maven.plugins.studio.wait

import com.atlassian.maven.plugins.studio.AbstractFireballMojo
import com.atlassian.maven.plugins.studio.util.Application
import org.codehaus.cargo.container.internal.util.HttpUtils
import org.codehaus.cargo.container.internal.util.HttpUtils.HttpResult
import org.codehaus.cargo.util.log.LogLevel
import org.codehaus.cargo.util.log.Logger
import org.codehaus.cargo.util.log.SimpleLogger

public abstract class AbstractWaitStudioMojo extends AbstractFireballMojo
{
    private static final long WAIT_TIMEOUT = 5 * 60 * 1000;

    protected abstract Application getApplication()

    protected String getPingUrl()
    {
        return getApplication().getBaseUrl();
    }

    public void execute()
    {
        if (!skip)
        {
            log.info("Waiting for ${application} to start.")

            final long start = System.currentTimeMillis();
            final long end = start + WAIT_TIMEOUT;
            final HttpUtils httpUtils = new HttpUtils();
            httpUtils.setLogger(newDebugLevelLogger())
            final HttpResult result = new HttpResult();
            boolean started = false;
            boolean interrupted = false;

            // keep retrieving from the url until a good response is returned, under a time limit.
            while (!started && !interrupted && System.currentTimeMillis() < end)
            {
                // Wait for a second each time through the loop to avoid hammering Tomcat with requests before it's
                // ready to serve them, as well as to avoid filling the build log with thousands of failed ping messages
                Thread.sleep(1000)
                {
                    interrupted = true;
                    return true;
                }
                // In general this should block until the app is able to respond,
                // however if Tomcat is not yet properly initialized the connection is refused
                // and this returns false immediately
                if (!interrupted && httpUtils.ping(new URL(getPingUrl()), result))
                {
                    started = true;
                }
            }

            if (started)
            {
                log.info("${application} is up.")
            }
            else
            {
              // report the latest error response.
              StringBuilder sb = new StringBuilder();
              sb.append(application.toString())
                .append(" is not up after ")
                .append((System.currentTimeMillis() - start) / 1000)
                .append(" seconds.");
              if (result.responseCode != -1)
              {
                  sb.append(" Response Code:").append(result.responseCode);
              }
              if (result.responseMessage != null)
              {
                sb.append(" Response Message:").append(result.responseMessage);
              }
              if (result.responseBody != null)
              {
                sb.append(" Response Body:").append(result.responseBody);
              }

              throw new RuntimeException(sb.toString());
            }
        }
    }

    private Logger newDebugLevelLogger()
    {
        def logger = new SimpleLogger()
        logger.setLevel(LogLevel.DEBUG)
        return logger
    }
}