package com.atlassian.maven.plugins.studio.util

import com.atlassian.util.patch.PatchApplicationFactory
import com.atlassian.util.patch.util.PatchList
import com.atlassian.util.patch.Patch
import com.atlassian.util.patch.PatchResult
import com.atlassian.util.patch.PatchResult.Result
import org.apache.maven.plugin.logging.Log

/**
 * PatchFile is a helper for applying a patch to a file using ant
 * It checks whether a patch applied successfully, failed or was skipped as it
 * was applied previously
 */
class PatchFile
{
  File patchFile
  File dir

  def PatchFile(String dir, File patchFile)
  {
    if(!patchFile.exists()) {
      throw FileNotFoundException('Patch file: ' + patchFile.path + ' not found')
    }

    this.patchFile = patchFile
    this.dir = new File(dir)
  }

    public static boolean applyPatches(String patchesDir, String baseDirectory, Log log) {
        File patchDir = new File(patchesDir)
        if (patchDir.exists() && patchDir.isDirectory())
        {
            log.info("Started Patching")
            patchDir.eachFile {
                if (it.name.endsWith(".patch"))
                {
                    PatchFile patchfile = new PatchFile(baseDirectory, it)
                      
                    // If the patch failed and the ignorePatchFailures is false then fail
                    if(patchfile.apply(log) == PatchState.FAILED)
                    {
                        return false;
                    }
                }
            }
            log.info("Finished Patching")
        }
        
        return true;
    }
    
  /**
   * Patches a file with the .patch file
   * @param ant
   * @return an int flag indicating success, failure or patch skipped
   */
  public PatchState apply(Log log)
  {   
      Result overallResult = Result.SUCCESS;
      log.info("=== Patch File: " + this.patchFile + " ===")
      PatchList insidePatchFile = null
      try
      {
          insidePatchFile = PatchApplicationFactory.loadPatch(this.patchFile)
          insidePatchFile.overrideBaseDirectory(this.dir)
      }
      catch(Exception e)
      {
          log.error("Patch File Loading Failed: " + e)
          overallResult = Result.FAILED
      }
          
      // read: for each patch inside the patch file
      if(overallResult == Result.SUCCESS && insidePatchFile != null)
      {
          for(Patch patch : insidePatchFile)
          {
              Result overallPerPatchResult = Result.SUCCESS;
              log.info(" Attempting to apply patch to: " + patch.getPatchedFile())
              try
              {
                  PatchResult thisResult = patch.apply()
                  List<Result> hunkResults = thisResult.getHunkResults()
                  for(int i = 1; i <= hunkResults.size(); ++i)
                  {
                      Result result = hunkResults.get(i - 1)
                      if(overallPerPatchResult.compareTo(result) >= 0)
                      {
                          overallPerPatchResult = result
                      }
                      switch(result)
                      {
                          case Result.SUCCESS:
                              // it worked cool, keep moving
                              break;
                          
                          case Result.AlreadyApplied:
                              log.warn("  Hunk #" + i + " has already been applied. Skipping.")
                              break;
                          
                          case Result.FAILED:
                          default:
                              log.error("  Hunk #" + i + " has failed.")
                              break;
                      }
                  }
              }
              catch(Exception e)
              {
                  log.error("The patch failed to apply: " + e)
                  overallPerPatchResult = Result.FAILED
              }
              
              log.info(" Patch Application finished: Result {" + overallPerPatchResult.name() + "}")
              
              if(overallResult.compareTo(overallPerPatchResult) >= 0)
              {
                  overallResult = overallPerPatchResult
              }
          }
      }
      
      log.info("=== Patch File Applied: Overall Result: {" + overallResult.name() + "}")
      
      switch(overallResult)
      {
          case Result.SUCCESS:
              return PatchState.SUCCESS
          case Result.AlreadyApplied:
              return PatchState.SKIPPED
          case Result.FAILED:
          default:
              return PatchState.FAILED
      }
  }
}
