package com.atlassian.maven.plugins.studio.util

import com.atlassian.maven.plugins.studio.AbstractFireballMojo

import static org.twdata.maven.mojoexecutor.MojoExecutor.*

import com.atlassian.maven.plugins.studio.AbstractFireballMojo

import org.apache.maven.model.Plugin
import org.twdata.maven.mojoexecutor.MojoExecutor.Element

import static org.twdata.maven.mojoexecutor.MojoExecutor.*

public class ConfluenceApplication extends AbstractTomcatApplication{

    final Plugin sqlPlugin = plugin(
        groupId('org.codehaus.mojo'),
        artifactId('sql-maven-plugin'))
    
    def String database = 'confluencedb'
    
    def ConfluenceApplication(AbstractFireballMojo mojo) {
        this.mojo = mojo
        httpPort = mojo.confluenceHttpPort
        rmiPort = mojo.confluenceRmiPort
        debugPort = mojo.confluenceDebugPort
        debugSuspend = mojo.confluenceSuspend
        webAppName = "confluence"
        contextPath = mojo.confluenceContextPath
        warDirectory = mojo.confluenceExplodedWarDirectory
        homeDirectory = mojo.confluenceHome
        baseUrl = mojo.getBaseUrl("confluence")
    }

    protected String getJvmArgs() {
        return "-Xmx512m -XX:MaxPermSize=256m ${devMode} ${mojo.confJvmArgsExtra} ${mojo.confluencePluginResourceDirectories} ${mojo.halInstanceUriJvmArg}"
    }

    private String getDevMode() {
        mojo.debug ? "-Dconfluence.dev.mode=true -Datlassian.dev.mode=true" : ""
    }
    
    @Override
    protected String getDatasource()
    {
        return getDatasource(database);
    }
    
    protected String getJdbcUrl(String databaseName)
    {
        return "jdbc:${mojo.databaseType}://localhost:5432/${databaseName}"
    }

    protected String getDatasource(String databaseName) {
        switch(mojo.databaseType) {
            case 'postgresql':
                return "cargo.datasource.url=${getJdbcUrl(databaseName)}|cargo.datasource.driver=org.postgresql.Driver|cargo.datasource.username=${mojo.databaseUsername}|cargo.datasource.password=${mojo.databasePassword}|cargo.datasource.type=javax.sql.DataSource|cargo.datasource.jndi=jdbc/DefaultDS"
            default:
                // Note, this must be the database, otherwise, CONF-20739 will hurt you
                return "cargo.datasource.url=jdbc:hsqldb:${mojo.confluenceHome}/database/${databaseName};hsqldb.tx=MVCC|cargo.datasource.driver=org.hsqldb.jdbcDriver|cargo.datasource.username=${mojo.databaseUsername}|cargo.datasource.password=${mojo.databasePassword}|cargo.datasource.type=javax.sql.DataSource|cargo.datasource.jndi=jdbc/DefaultDS";
        }
    }

    @Override
    protected Element[] getSystemProperties(Element... elements)
    {
        List result = new ArrayList(Arrays.asList(super.getSystemProperties(elements)))
        // this is used by com.atlassian.studio.host.common.initialdata.InitialDataServiceImpl.getDatabaseType() to choose the Hibernate dialect
        result.add(element(name('studio.database.type'), mojo.databaseType));
        result.toArray(new Element[result.size()])
    }

    @Override
    public void start(boolean parallel)
    {
        createDatabaseIfNecessary()
        super.start(parallel);
    }
    
    private boolean createDatabaseIfNecessary() {
        switch(mojo.databaseType) {
            case 'postgresql':
                executeSqlCommand("drop database if exists ${database};create database ${database};", 'org.postgresql.Driver', 'postgres')
                return true;
            default:
                return false
        }
    }
    
    
    private void executeSqlCommand(String sqlCommand, String driver, String maintenanceDatabase)
    {
        String jdbcUrl = getJdbcUrl(maintenanceDatabase)
        executeMojo(sqlPlugin, goal('execute'),
            configuration(
                    element(name('skip'), "${mojo.skip}"),
                    element(name('url'), "${jdbcUrl}"),
                    element(name('username'), "${mojo.databaseUsername}"),
                    element(name('password'), "${mojo.databasePassword}"),
                    element(name('autocommit'), 'true'),
                    element(name('sqlCommand'), "${sqlCommand}"),
                    element(name('driver'), driver)
                ),
            executionEnvironment(mojo.project, mojo.session, mojo.pluginManager))
        mojo.log.info("Executed [${sqlCommand}] on [${jdbcUrl}] with user [${mojo.databaseUsername}]")
    }

    String toString() {
        "Confluence"
    }

    @Override
    protected Element[] getTomcatDependencies() {
        switch(mojo.databaseType) {
            case 'postgresql':
                return [
                    element(name('dependency'),
                    element(name('groupId'), 'postgresql'),
                    element(name('artifactId'), 'postgresql'))
                ]
            default:
                return [
                    element(name('dependency'),
                    element(name('groupId'), 'org.hsqldb'),
                    element(name('artifactId'), 'hsqldb'))
                ]
        }
    }
}