package com.atlassian.maven.plugins.studio.util

import com.atlassian.maven.plugins.studio.AbstractFireballMojo

import org.twdata.maven.mojoexecutor.MojoExecutor.Element

import static org.twdata.maven.mojoexecutor.MojoExecutor.*

public abstract class AbstractTomcatApplication implements Application
{
    AbstractFireballMojo mojo
    int httpPort
    int rmiPort
    int debugPort
    boolean debugSuspend
    String webAppName
    String contextPath
    String warDirectory
    String homeDirectory
    String baseUrl

    public void start(boolean parallel)
    {
        unpackContainer()

        executeMojo(mojo.cargoPlugin, goal('start'),
                configuration(
                        element(name('wait'), "false"),
                        element(name('skip'), 'false'),
                        element(name('container'),
                                element(name('containerId'), 'tomcat6x'),
                                element(name('home'), getContainerInstallDir()),
                                element(name('output'), getProductLog()),
                                element(name("systemProperties"), getSystemProperties(
                                        homeProp, productHomeProp, studioInitialDataProp,
                                        studioInitialDataXmlProp, pluginWaitProp)
                                ),
                                element(name("timeout"), getTimeout(parallel)),
                                element(name("log"), getProductCargoLog()),
                                element(name("dependencies"), tomcatDependencies)
                        ),

                        element(name('configuration'),
                                element(name('home'), getContainerConfigDir()),
                                element(name('type'), 'standalone'),
                                element(name('properties'),
                                        element(name('cargo.datasource.datasource'), datasource),
                                        element(name('cargo.servlet.port'), httpPort.toString()),
                                        element(name('cargo.rmi.port'), rmiPort.toString()),
                                        element(name('cargo.logging'), 'high'),
                                        element(name('cargo.jvmargs'), jvmArgsInt)),
                                element(name('deployables'),
                                        element(name('deployable'),
                                                element(name('location'), "${mojo.explodedDirectory}/${warDirectory}"),
                                                element(name('type'), 'war'),
                                                element(name('properties'), element(name('context'), contextPath)),
                                                element(name('pingURL'), baseUrl),
                                                element(name('pingTimeout'), '60000'))))),
                executionEnvironment(mojo.project, mojo.session, mojo.pluginManager))
    }

    public void stop()
    {
        // this is needed if the application was not started
        // we could check the skip flag in AbstractStopStudioMojo,
        // but the flag could be different from that is used to start the app though.
        unpackContainer();

        executeMojo(mojo.cargoPlugin, goal('stop'),
                configuration(
                        element(name('skip'), 'false'),
                        element(name('container'),
                                element(name('containerId'), 'tomcat6x'),
                                element(name('home'), getContainerInstallDir()),
                                element(name("log"), getProductCargoLog())),
                        element(name('configuration'),
                                element(name('home'), getContainerConfigDir()))
                ),
                executionEnvironment(mojo.project, mojo.session, mojo.pluginManager))
    }

    private void unpackContainer()
    {
        File containerDir = new File(getContainerRootDir())
        if (!containerDir.exists())
        {
            executeMojo(mojo.dependencyPlugin,
                    goal("unpack"),
                    configuration(
                            element(name("artifactItems"),
                                    element(name("artifactItem"),
                                            element(name("groupId"), "org.apache.tomcat"),
                                            element(name("artifactId"), "apache-tomcat"),
                                            element(name("version"), "6.0.20"),
                                            element(name("type"), "zip"))),
                            element(name("outputDirectory"), getContainerRootDir())
                    ),
                    executionEnvironment(mojo.project, mojo.session, mojo.pluginManager));
        }
    }

    protected String getJvmArgsInt()
    {
        String result = "${jvmArgs} ${mojo.jvmArgs?mojo.jvmArgs:""}"

        if (mojo.force32bit)
        {
            //Force the JVM into 32-bit mode. Forcing server mode also because it seems that 1.6 JVMs assume client
            //mode when running in 32-bit mode. 
            result = "-d32 -server ${result}"
        }

        if (mojo.debug)
        {
            return "${result} -Xdebug -Xrunjdwp:transport=dt_socket,address=${debugPort},suspend=${debugSuspend ? 'y' : 'n'},server=y"
        }
        else
        {
            return result
        }
    }

    protected Element[] getSystemProperties(Element... elements)
    {
        List result = new ArrayList(Arrays.asList(elements))
        // TODO: pass system properties
        result.toArray(new Element[result.size()])
    }

    Element getHomeProp()
    {
        BuildUtils.isStudioBuild(mojo.project) ? element(name("studio.home"), mojo.studioHome) :
            element(name("hosted.home"), "${homeDirectory}/hosted-home")
    }

    Element getProductHomeProp()
    {
        element(name("${webAppName}.home"), homeDirectory)
    }

    Element getStudioInitialDataProp()
    {
        return element(name("studio.initial.data.properties"), "${mojo.studioHome}/studio-initial-data.properties");
    }

    Element getStudioInitialDataXmlProp()
    {
        return element(name("studio.initial.data.xml"), "${mojo.studioHome}/studio-initial-data.xml");
    }

    Element getPluginWaitProp()
    {
        return element(name("atlassian.plugins.enable.wait"), "600")
    }

    private String getContainerRootDir()
    {
        "$mojo.project.build.directory/container/tomcat6x"
    }

    private String getContainerInstallDir()
    {
        "${getContainerRootDir()}/apache-tomcat-6.0.20"
    }

    private String getContainerConfigDir()
    {
        "$mojo.project.build.directory/containers/${webAppName}"
    }

    private String getProductLog()
    {
        return "$mojo.project.build.directory/${webAppName}.log"
    }

    private String getProductCargoLog()
    {
        return "$mojo.project.build.directory/${webAppName}.cargo.log"
    }

    private String getTimeout(boolean parallel)
    {
        if (parallel)
        {
            return "0";
        }
        else
        {
            return "300000";
        }
    }

    protected String blankIfNull(String s)
    {
      if (s == null || s.toLowerCase().equals("null"))
      {
        return ""
      }
      else
      {
        return s
      }
    }

    protected Element[] getTomcatDependencies()
    {
        return [
                element(name('dependency'),
                        element(name('groupId'), 'hsqldb'),
                        element(name('artifactId'), 'hsqldb')),
                element(name('dependency'),
                        element(name('groupId'), 'jta'),
                        element(name('artifactId'), 'jta')),
                element(name('dependency'),
                        element(name('groupId'), 'ots-jts'),
                        element(name('artifactId'), 'ots-jts')),
                element(name('dependency'),
                        element(name('groupId'), 'jotm'),
                        element(name('artifactId'), 'jotm')),
                element(name('dependency'),
                        element(name('groupId'), 'jotm'),
                        element(name('artifactId'), 'jotm-jrmp_stubs')),
                element(name('dependency'),
                        element(name('groupId'), 'jotm'),
                        element(name('artifactId'), 'jotm-iiop_stubs')),
                element(name('dependency'),
                        element(name('groupId'), 'jotm'),
                        element(name('artifactId'), 'jonas_timer')),
                element(name('dependency'),
                        element(name('groupId'), 'jotm'),
                        element(name('artifactId'), 'objectweb-datasource')),
                element(name('dependency'),
                        element(name('groupId'), 'carol'),
                        element(name('artifactId'), 'carol')),
                element(name('dependency'),
                        element(name('groupId'), 'carol'),
                        element(name('artifactId'), 'carol-properties')),
                element(name('dependency'),
                        element(name('groupId'), 'xapool'),
                        element(name('artifactId'), 'xapool')),
                element(name('dependency'),
                        element(name('groupId'), 'commons-logging'),
                        element(name('artifactId'), 'commons-logging')),
                element(name('dependency'),
                        element(name('groupId'), 'log4j'),
                        element(name('artifactId'), 'log4j'))
        ]
    }

    protected abstract String getJvmArgs()

    protected abstract String getDatasource()
}
