package com.atlassian.maven.plugins.studio

import org.apache.maven.artifact.factory.ArtifactFactory
import org.apache.maven.artifact.metadata.ArtifactMetadataSource
import org.apache.maven.artifact.repository.ArtifactRepository
import org.apache.maven.artifact.resolver.ArtifactResolver
import org.apache.maven.project.MavenProjectBuilder

abstract class AbstractFireballMojo extends AbstractStudioMojo
{
    /**
     * Directory for the exploded war
     *
     * @parameter expression="${fireball.exploded.dir}" default-value="${project.build.directory}/fireball"
     */
    String explodedDirectory

    /**
     * Directory for the home directory
     *
     * @parameter expression="${fireball.home.dir}" default-value="${project.build.directory}/home"
     */
    String homeDirectory

    /**
     * Whether or not to skip the tests
     *
     * @parameter expression="${maven.test.it.skip}" default-value="false"
     */
    boolean skip

    /**
     * When in debug mode, whether to wait once set up is complete or not
     *
     * @parameter expression="${studio.wait}" default-value="true"
     */
    boolean wait

    /**
     * Whether applications should be run in debug mode
     *
     * @parameter expression="${studio.debug}" default-value="false"
     */
    boolean debug

    /**
     * Whether JIRA should suspend or not
     *
     * @parameter expression="${jira.debug.suspend}" default-value="false"
     */
    boolean jiraSuspend

    /**
     * Whether Crowd should suspend or not
     *
     * @parameter expression="${crowd.debug.suspend}" default-value="false"
     */
    boolean crowdSuspend

    /**
     * Whether Confluence should suspend or not
     *
     * @parameter expression="${confluence.debug.suspend}" default-value="false"
     */
    boolean confluenceSuspend

    /**
     * Whether Bamboo should suspend or not
     *
     * @parameter expression="${bamboo.debug.suspend}" default-value="false"
     */
    boolean bambooSuspend

    /**
     * Whether FishEye should suspend or not
     *
     * @parameter expression="${fisheye.debug.suspend}" default-value="false"
     */
    boolean fisheyeSuspend

    /**
     * Whether JIRA should log memory usage (detailed garbage collection and heap dump on first OutOfMemory error.
     *
     * @parameter expression="${jira.memory.log}" default-value="false"
     */
    boolean jiraMemoryLog

    /**
     * Whether studio apps should be forced to start with a 32-bit JVM. This will save some memory.
     *
     * @parameter expression="${studio.32bit}" default-value="false"
     */
    boolean force32bit;

    /**
     * Extra JVM arguments to be used to start JIRA, such as "-agentlib:yjpagent",
     * which is the argument used to profile JIRA using YourKit.
     *
     * @parameter expression="${jira.jvmargs.extra}" default-value=""
     */
    String jiraJvmArgsExtra
    
    /**
     * @parameter expression="${studio.database.type}" default-value="hsqldb"
     */
    String databaseType

    /**
    * @parameter expression="${studio.database.username}" default-value="sa"
    */
    String databaseUsername
        
    /**
    * @parameter expression="${studio.database.password}" default-value=""
    */
    String databasePassword
    
    public void setJiraJvmArgsExtra(String val)
    {
         jiraJvmArgsExtra = stripQuotes(val);
    }

    /**
     * Extra JVM arguments to be used when starting Confluence.
     *
     * @parameter expression="${conf.jvmargs.extra}" default-value=""
     */
    String confJvmArgsExtra
    
    public void setConfJvmArgsExtra(String val)
    {
        confJvmArgsExtra = stripQuotes(val);
    }

    /**
     * Extra JVM arguments to be used when starting Confluence.
     *
     * @parameter expression="${crowd.jvmargs.extra}" default-value=""
     */
    String crowdJvmArgsExtra

    public void setCrowdJvmArgsExtra(String val)
    {
        crowdJvmArgsExtra = stripQuotes(val);
    }

    /**
     * Extra JVM arguments to be used when starting Bamboo.
     *
     * @parameter expression="${bamboo.jvmargs.extra}" default-value=""
     */
    String bambooJvmArgsExtra

    public void setBambooJvmArgsExtra(String val) {
        bambooJvmArgsExtra = stripQuotes(val);
    }

    /**
     * @component role="org.apache.maven.project.MavenProjectBuilder"
     * @required
     * @readonly
     */
    MavenProjectBuilder mavenProjectBuilder;

    /**
     * @component
     * @required
     * @readonly
     */
    ArtifactFactory artifactFactory;

    /**
     * @parameter default-value="${localRepository}"
     * @required
     * @readonly
     */
    ArtifactRepository localRepository;

    /**
     * @parameter default-value="${project.remoteArtifactRepositories}"
     * @required
     * @readonly
     */
    java.util.List remoteRepositories;

    /**
     * @component
     * @required
     * @readonly
     */
    ArtifactMetadataSource artifactMetadataSource;

    /**
     * @component
     * @required
     * @readonly
     */
    ArtifactResolver artifactResolver;
    
    protected String stripQuotes(String s)
    {
        if (s == null || s.equalsIgnoreCase("null"))
        {
            return "";
        }
        if (s.startsWith('"') || s.startsWith("'"))
        {
            s = s.substring(1)
        }
        if (s.endsWith('"') || s.endsWith("'"))
        {
            s = s.substring(0, s.length() - 1)
        }
        return s
    }

    /**
     * The location to extract crowd to under the exploded directory
     * @parameter expression="${crowd.exploded.dir}" default-value="crowd"
     */
    String crowdExplodedWarDirectory

    /**
     * The location to extract Bamboo to under the exploded directory
     * @parameter expression="${bamboo.exploded.dir}" default-value="bamboo"
     */
    String bambooExplodedWarDirectory

    /**
     * The location to extract Confluence to under the exploded directory
     * @parameter expression="${confluence.exploded.dir}" default-value="confluence"
     */
    String confluenceExplodedWarDirectory

    /**
     * The location to extract Crucible to under the exploded directory
     * @parameter expression="${fisheye.exploded.dir}" default-value="fisheye"
     */
    String fisheyeExplodedDirectory

    /**
     * The location to extract JIRA to under the exploded directory
     * @parameter expression="${jira.exploded.dir}" default-value="jira"
     */
    String jiraExplodedWarDirectory

    /**
     * The SVN root directory
     * @parameter expression="${svn.home.dir}" default-value="${project.build.directory}/home/svn-home"
     */
    String svnHomeDirectory

    /**
     * The SVN public url
     * @parameter expression="${svn.public.url}" default-value="${project.build.directory}/svn"
     */
    String svnPublicUrl

    /**
     * Get the SVN URL in in 'file:///xyz' form
     */
    String getSvnBaseUrl()
    {
        if (System.properties['os.name'].toLowerCase().contains('windows')) {
            // 3 slashes are needed for Windows, so this will evaluate to e.g. file:///C:\something
            return "file:///${svnHomeDirectory}"
        } else {
            // Meanwhile, on Unix, it should be e.g. file:///home/something. 
            // Having 4 slashes on UNIX doesn't break the path, but it does affect tests:
            // when we re-create AppLinks after enabling an app, we set up the path with 3 slashes, and then compare against this value.
            return "file://${svnHomeDirectory}"
        }
    }
    /**
     * Get the SVN PUBLIC URL
     */
    String getSvnPublicUrl()
    {
        if (System.properties['os.name'].toLowerCase().contains('windows')) {
            return "file:///${svnPublicUrl}"
        } else {
            return "file://${svnPublicUrl}"
        }
    }

    /**
     * The Studio home directory
     *
     * @parameter expression="${studio.home.dir}" default-value="studio-home"
     */
    String studioHomeDirectory

    String getStudioHome()
    {
        return "${homeDirectory}/${studioHomeDirectory}"
    }

    /**
     * The name of the target directory (under homeDirectory) where jira home will be
     *
     * @parameter expression="${jira.home.directory}" default-value="jira-home"
     */
    String jiraHomeDirectory

    String getJiraHome()
    {
        return "${homeDirectory}/${jiraHomeDirectory}"
    }

    /**
     * Extra directories to plugin resources to pass to all applications.
     *
     * @parameter expression="${extra.plugin.resource.dirs}"
     */
    String extraPluginResourcesDir

    /**
     * Directories to plugin resources for JIRA.
     *
     * @parameter expression="${jira.plugin.resource.dirs}"
     */
    String jiraPluginResourcesDir

    String getJiraPluginResourceDirectories()
    {
      return jiraPluginResourcesDir == null ? "" : "-Dplugin.resource.directories=" + getPluginResourcesDirPaths(jiraPluginResourcesDir);
    }

    /**
     * The name of the target directory (under homeDirectory) where crowd home will be
     *
     * @parameter expression="${crowd.home.directory}" default-value="crowd-home"
     */
    String crowdHomeDirectory

    String getCrowdHome()
    {
        return "${homeDirectory}/${crowdHomeDirectory}"
    }


    /**
     * The name of the target directory (under homeDirectory) where confluence home will be
     *
     * @parameter expression="${confluence.home.directory}" default-value="confluence-home"
     */
    String confluenceHomeDirectory

    String getConfluenceHome()
    {
        return "${homeDirectory}/${confluenceHomeDirectory}"
    }

    /**
     * Directories to plugin resources for Confluence.
     *
     * @parameter expression="${confluence.plugin.resource.dirs}"
     */
    String confluencePluginResourcesDir

    String getConfluencePluginResourceDirectories()
    {
      return confluencePluginResourcesDir == null ? "" : "-Dplugin.resource.directories=" + getPluginResourcesDirPaths(confluencePluginResourcesDir);
    }


    /**
     * The name of the target directory (under homeDirectory) where bamboo home will be
     *
     * @parameter expression="${bamboo.home.directory}" default-value="bamboo-home"
     */
    String bambooHomeDirectory

    String getBambooHome()
    {
        return "${homeDirectory}/${bambooHomeDirectory}"
    }

    /**
     * Directories to plugin resources for bamboo.
     *
     * @parameter expression="${bamboo.plugin.resource.dirs}"
     */
    String bambooPluginResourcesDir

    String getBambooPluginResourceDirectories()
    {
      return bambooPluginResourcesDir == null ? "" : "-Dplugin.resource.directories=" + getPluginResourcesDirPaths(bambooPluginResourcesDir);
    }


    /**
     * The name of the target directory (under homeDirectory) where fisheye home will be
     *
     * @parameter expression="${fisheye.home.directory}" default-value="fisheye-home"
     */
    String fisheyeHomeDirectory

    String getFisheyeHome()
    {
        return "${homeDirectory}/${fisheyeHomeDirectory}"
    }

    /**
     * Directories to plugin resources for fisheye.
     *
     * @parameter expression="${fisheye.plugin.resource.dirs}"
     */
    String fisheyePluginResourcesDir

    String getFisheyePluginResourceDirectories()
    {
      return fisheyePluginResourcesDir == null ? "" : "-Dplugin.resource.directories=" + getPluginResourcesDirPaths(fisheyePluginResourcesDir);
    }

    /**
     * The studio home directory source data
     *
     * @parameter default-value="${basedir}/src/test/resources/home/studio-home"
     */
    String studioHomeData

    /**
     * Whether Bamboo is enabled or not
     *
     * @parameter expression="${bamboo.enabled}" default-value="true"
     */
    boolean bambooEnabled

    /**
     * Exists so FisheyeApplication can use it
     */
    public AntBuilder getAnt()
    {
        return super.ant
    }

  /**
   * Builds the plugin resources directory with the extra plugin resources
   * if they exist.
   * Also removes all whitespaces to makes sure the plugin resources path is
   * in the correct format.
   * @param pluginResourcesDirs the set of paths to format correctly
   * @return the resources paths that may include the extra plugin resource dirs
   */
    private String getPluginResourcesDirPaths(String pluginResourcesDirs)
    {
        String resourcesDirs = "";

        // Ensure that extra resources exists and is not empty or empty quotes
        if (extraPluginResourcesDir != null &&
                !extraPluginResourcesDir.isEmpty() &&
                !extraPluginResourcesDir.equals("\"\""))
        {
            resourcesDirs = extraPluginResourcesDir + ",";
        }

        resourcesDirs += pluginResourcesDirs;
        resourcesDirs = resourcesDirs.replaceAll("[\\s]+", "");

        return resourcesDirs;
    }

    /**
     * The WebDAV root directory
     * @parameter expression="${webdav.home.dir}" default-value="${project.build.directory}/home/webdav-home"
     */
    String webDavHomeDirectory

    /**
     * The SVN Importer staging directory
     * @parameter expression="${svn.import.staging.dir}" default-value="${project.build.directory}/home/svn-import-staging"
     */
    String svnImportStagingDirectory

    /**
     * The HAL URI to query for retrieving license information about the instance
     * @parameter expression="${studio.hal.instance.uri}" default-value="file://${project.build.testOutputDirectory}/xml/devmode-hal-licenses.xml"
     */
    String halInstanceUri

    /**
     * Returns the required JVM argument to set the HAL instance URI.
     *
     * @return required JVM argument to set the HAL instance URI
     */
    
    String getHalInstanceUriJvmArg()
    {
        return halInstanceUri == null ? "" : "-Dstudio.hal.instance.uri=" + halInstanceUri;
    }
}
