package com.atlassian.maven.plugins.studio.war

import com.atlassian.maven.plugins.studio.AbstractStudioMojo
import com.atlassian.maven.plugins.studio.util.BuildUtils
import com.atlassian.maven.plugins.studio.util.PatchFile
import org.apache.maven.artifact.Artifact
import org.apache.maven.execution.MavenSession
import org.apache.maven.model.Plugin
import org.apache.maven.plugin.MojoExecutionException
import org.apache.maven.plugin.PluginManager
import org.apache.maven.project.MavenProject
import org.apache.maven.project.MavenProjectHelper
import static org.twdata.maven.mojoexecutor.MojoExecutor.artifactId
import static org.twdata.maven.mojoexecutor.MojoExecutor.configuration
import static org.twdata.maven.mojoexecutor.MojoExecutor.element
import static org.twdata.maven.mojoexecutor.MojoExecutor.executeMojo
import static org.twdata.maven.mojoexecutor.MojoExecutor.executionEnvironment
import static org.twdata.maven.mojoexecutor.MojoExecutor.goal
import static org.twdata.maven.mojoexecutor.MojoExecutor.groupId
import static org.twdata.maven.mojoexecutor.MojoExecutor.name
import static org.twdata.maven.mojoexecutor.MojoExecutor.plugin

/**
 * @requiresDependencyResolution runtime
 * @goal package-war
 */
class PackageWarMojo extends AbstractStudioMojo
{
    final Plugin yuicompressorPlugin = plugin(
            groupId('net.sf.alchim'),
            artifactId('yuicompressor-maven-plugin'))

    /**
     * The Maven Session
     *
     * @parameter expression="${session}"
     * @required
     * @readonly
     */
    MavenSession session

    /**
     * The Maven PluginManager
     *
     * @component
     * @required
     * @readonly
     */
    PluginManager pluginManager

    /**
     * The Maven Projectpom
     *
     * @parameter expression="${project}"
     * @required
     * @readonly
     */
    MavenProject project

    /**
     * Maven ProjectHelper.
     *
     * @component
     * @readonly
     */
    MavenProjectHelper projectHelper

    /**
     * Whether or not to take transitive dependencies in account.
     *
     * @parameter expression="${include.transitive.dependencies}" default-value="false"
     * @required
     */
    boolean includeTransitiveDependencies

    /**
     * Work directory for the war
     *
     * @parameter expression="${war.work.dir}" default-value="${project.build.directory}/war/work"
     * @required
     */
    String workDirectory

    /**
     * Directory for the exploded war
     *
     * @parameter expression="${war.exploded.dir}" default-value="${project.build.directory}/war/exploded"
     * @required
     */
    String explodedDirectory

    /**
     * Map the contents of the exploded war from this pattern to *
     *
     * @parameter
     */
    String globMapperFrom

    /**
     * The source web archive
     *
     * @parameter
     */
    String sourceArtifactId

    /**
     * The library directory of the exploded war
     *
     * @parameter expression="${src.lib.dir}" default-value="/WEB-INF/lib"
     * @required
     */
    String libDirectory

    /**
     * The directory in which the bundled plugins lives, relative to explodedDirectory
     *
     * @parameter expression="${bundled.plugins.dir}" default-value="/WEB-INF/classes"
     * @required
     */
    String bundledPluginsDirectory

    /**
     * Libraries to exclude from the web app, supports ant wildcards
     *
     * @parameter
     */
    ArrayList<String> libraryExcludes

    /**
     * Libraries to prevent from being deleted by the excludes and also by the deleting of old versions of libraries.
     * This is mainly here for JIRA so that it can do a blanket delete of all language packs, but exclude the UK
     * language back from that list of things to delete
     *
     * @parameter
     */
    ArrayList<String> libraryIncludes
    
    /**
     * Libraries to be exploded to WEB-INF/classes.
     * This is mainly here for JIRA so that properties file and xml file from libraries can be patched.
     * 
     * @parameter
     */
    ArrayList<String> libraryExplodes

    /**
     * The file name of the zip file
     *
     * @parameter expression="${bundled.plugins.output}" default-value="atlassian-bundled-plugins.zip"
     */
    String bundledPluginsOutputFile

    /**
     * The list of plugins to be excluded
     * 
     * @parameter
     */
    ArrayList excludePlugins // If specified as List, got InstantiationException when maven tries to instantiate an interface

   /**
    * The list of directories in which extra contents live, which will be copied to explodedDirectory
    *
    * @parameter expression="${war.extra.dir}"
    */
    ArrayList extraContentsDirectories // If specified as List, got InstantiationException when maven tries to instantiate an interface

    /**
     * Directory of extra things to include in the bundled plugins
     *
     * @parameter
     */
    String bundledPluginsIncludeDirectory

   /**
    * @parameter default-value=""
    */
    String baseDir = ""

    /**
     * Whether the package stage should be skipped
     *
     * @parameter expression="${studio.package.skip}" default-value="false"
     */
    boolean skipPackage

    /**
     * Whether the unpack stage should be skipped (useful for speeding up studio deploys)
     *
     * @parameter expression="${studio.unpack.skip}" default-value="false"
     */
    boolean skipUnpack

    /**
     * If ignorePatchFailures is true, failure to apply any patches won't fail the build
     *
     * @parameter expression="${ignore.patch.failures}" default-value="false"
     */
    boolean ignorePatchFailures

    /**
     * The directory to put classpath resources into the application
     *
     * @parameter expression="${classpath.resource.directory}" default-value="WEB-INF/classes"
     */
    String classpathResourceDirectory

    /**
     * The webapp directory to put minified JavaScript resources into the application
     *
     * @parameter expression="${javascript.resource.directory}" default-value="includes/js"
     */
    String javascriptResourceDirectory

    /**
     * Whether we are in dev mode (and so include dependencies for dev mode)
     *
     * @parameter expression="${studio.debug}" default-value="false"
     */
    boolean debug

    void execute()
    {
        ant.project.addBuildListener(antTaskListener)

        ant.mkdir(dir: explodedDirectory)
        ant.mkdir(dir: workDirectory)

        def artifacts = includeTransitiveDependencies ? project.artifacts : project.dependencyArtifacts;
        def sourceArtifacts = sourceArtifactId ? artifacts.findAll { it.artifactId == sourceArtifactId } :
            artifacts.findAll { it.type == 'war' || it.type == 'zip'}

        if (!skipUnpack)
        {
            if (sourceArtifacts.size() < 1)
                throw new MojoExecutionException("No source artifacts found!")

            // unpack war and/or zip dependencies
            sourceArtifacts.each {dependency ->
                ant.unzip(src: dependency.file, dest: explodedDirectory, overwrite: true) {
                    if (globMapperFrom != null)
                    {
                        globmapper(from: globMapperFrom, to: "*")
                    }
                }
            }
        }

        extraContentsDirectories.each { extraContentsDirectory ->
          ant.copyWithPerms(todir: "$explodedDirectory", overwrite: true) {
              fileset(dir: "$extraContentsDirectory")
          }
        }

        // Delete all library excludes
        if (libraryExcludes != null && !libraryExcludes.empty)
        {
            ant.delete(failonerror: false) { 
                fileset(dir: "$explodedDirectory/$baseDir/$libDirectory") {
                    libraryExcludes.each {
                        include(name: it)
                    }
                    if (libraryIncludes != null && !libraryIncludes.empty)
                    {
                        libraryIncludes.each {
                            exclude(name: it)
                        }
                    }
                }
            }
        }

        def webInfLibDependencies = artifacts.findAll { isJarToBePackaged(it) && it.scope != 'bundled' &&
            it.scope != "gapps-bundled" && it.scope != "dev-bundled" && !sourceArtifacts.contains(it)}
        def bundledDependencies = artifacts.findAll { isJarToBePackaged(it) &&
                it.scope == 'bundled' && !sourceArtifacts.contains(it)}
        def gappsBundledDependencies = artifacts.findAll { isJarToBePackaged(it) &&
                it.scope == 'gapps-bundled' && !sourceArtifacts.contains(it)}
        def devBundledDependencies = artifacts.findAll { isJarToBePackaged(it) &&
                it.scope == 'dev-bundled' && !sourceArtifacts.contains(it)}

        //Check that webInfLibDependencies is not empty otherwise all jars in lib will be deleted
        if (webInfLibDependencies.size() > 0)
        {
          // delete old versions of libraries being copied to WEB-INF/lib
          ant.delete(quiet: true) {
              fileset(dir: "$explodedDirectory/$baseDir/$libDirectory") {
                  webInfLibDependencies.each {
                      include(name: "$it.artifactId-*.jar")
                  }
                  if (libraryIncludes != null && !libraryIncludes.empty)
                  {
                      libraryIncludes.each {
                          exclude(name: it)
                      }
                  }
              }
          }
        }

        // copying libraries to WEB-INF/lib
        ant.copyWithPerms(todir: "$explodedDirectory/$baseDir/$libDirectory", flatten: true) {
            resources { webInfLibDependencies.each { file(file: it.file) } }
        }


        // add plugins2 plugins to bundled plugins
        def previousBundledPlugins = "$explodedDirectory/$baseDir/$bundledPluginsDirectory/$bundledPluginsOutputFile"
        if (new File(previousBundledPlugins).exists() || bundledDependencies)
        {
            ant.zipWithPerms(destfile: "$workDirectory/$bundledPluginsOutputFile", compress: false) {
                if (new File(previousBundledPlugins).exists())
                {
                    zipfileset(src: previousBundledPlugins) {
                        bundledDependencies.each {
                            exclude(name: "$it.artifactId-*.jar")
                        }
                        excludePlugins.each {excludePlugin ->
                            exclude(name: "$excludePlugin")
                        }
                    }
                }
                else
                {
                    log.info "There is no bundled plugins zip file at $previousBundledPlugins"
                }
                bundledDependencies.each { fileset(dir: it.file.parent, includes: it.file.name) }
                if (bundledPluginsIncludeDirectory != null && new File(bundledPluginsIncludeDirectory).exists())
                {
                    fileset(dir: bundledPluginsIncludeDirectory)
                }
                // Gapps plugins, if we're in debug and not in gapps mode, don't bundle them
                if (!debug || gappsEnabled)
                {
                    gappsBundledDependencies.each{fileset(dir: it.file.parent, includes: it.file.name)}
                }
                if (debug)
                {
                    devBundledDependencies.each{fileset(dir: it.file.parent, includes: it.file.name)}                    
                }
            }
            ant.copyWithPerms(file: "$workDirectory/$bundledPluginsOutputFile", todir: "$explodedDirectory/$baseDir/$bundledPluginsDirectory", overwrite: true)
        }

        // For the two tasks below, if we've skipped the unpack stage we don't want to overwrite files because it means
        // that if they are older, they have definitely already been copied across


        // Unpack common resources
        // This is studio specific and not used for hosted
        if (BuildUtils.isStudioBuild(project))
        {
          executeMojo(dependencyPlugin, goal("unpack-dependencies"),
                  configuration(
                          element(name("excludeTransitive"), "true"),
                          element(name("includeGroupIds"), "com.atlassian.studio"),
                          element(name("includeArtifactIds"), "studio-webapps-common"),
                          element(name("excludes"), "META-INF/**"),
                          element(name("outputDirectory"), "$explodedDirectory/$baseDir/$classpathResourceDirectory")
                  ),
                  executionEnvironment(project, session, pluginManager))
        }

        // copy resources and compiled classes across
        if (new File("$project.build.directory/classes").exists())
        {
            overwriteClasses()
        }

        // copy webapp resources across
        if (new File("$project.basedir/src/main/webapp").exists())
        {
            ant.copyWithPerms(todir: "$explodedDirectory/$baseDir", overwrite: !skipUnpack) {
                fileset(dir: "$project.basedir/src/main/webapp")
            }
        }
        
        // minify JavaScript and copy resources across
        if (new File("$project.basedir/src/main/javascript").exists() && javascriptResourceDirectory)
        {
            ant.copyWithPerms(todir: "$explodedDirectory/$baseDir/$javascriptResourceDirectory", overwrite: !skipUnpack) {
                fileset(dir: "$project.basedir/src/main/javascript")
            }
            executeMojo(yuicompressorPlugin, goal("compress"),
                    configuration(
                            element(name("sourceDirectory"), "$project.basedir/src/main/javascript"),
                            element(name("outputDirectory"), "$explodedDirectory/$baseDir/$javascriptResourceDirectory"),
                            element(name("force"), "true")
                    ),
                    executionEnvironment(project, session, pluginManager)
            )
        }

        // exploding libraries to WEB-INF/classes
        libraryExplodes.each {lib ->
            ant.unzip(dest: "$explodedDirectory/$baseDir/$classpathResourceDirectory", overwrite: true) {
                patternset {
                    exclude(name: "META-INF/MANIFEST.MF")
                }
                fileset(dir: "$explodedDirectory/$baseDir/$libDirectory") {
                    filename(name: lib)
                }
            }
        }

        // apply any patches
        if(!PatchFile.applyPatches("$project.basedir/src/main/patches", "$explodedDirectory/$baseDir", log))
        {
            if(!ignorePatchFailures)
            {
                fail("There was a patch failure")
            }
        }

        if (BuildUtils.isStudioBuild(project))
        {
          // create a gapps plugin list so that gapps plugins can be removed at a future stage
          File gappsPluginListFile = new File(explodedDirectory, gappsPluginListFileName)
          BufferedWriter writer = new BufferedWriter(new FileWriter(gappsPluginListFile))
          gappsBundledDependencies.each {writer.writeLine(it.file.name) }
          writer.close()
        }

        // creating the war and attaching
        if (!skipPackage)
        {
            def packaging = (sourceArtifactId ? sourceArtifacts.iterator().next().type : "war")
            def projectArtifactFile = "$project.build.directory/${project.build.finalName}.$packaging"
            ant.zipWithPerms(destfile: projectArtifactFile, basedir: explodedDirectory, compress: false)
            projectHelper.attachArtifact(project, packaging, new File(projectArtifactFile))
        }
    }

    protected void overwriteClasses()
    {
        ant.copyWithPerms(todir: "$explodedDirectory/$baseDir/$classpathResourceDirectory", overwrite: !skipUnpack) {
            fileset(dir: "$project.build.directory/classes")
        }
    }

    boolean isJarToBePackaged(Artifact artifact)
    {
        return artifact.artifactHandler.extension == 'jar' && artifact.scope != 'provided' && artifact.scope != 'test'
    }
}
