package com.atlassian.maven.plugins.studio.validation

import com.atlassian.maven.plugins.studio.AbstractStudioMojo 
import com.atlassian.plugins.client.service.plugin.PluginServiceClientImpl;
import com.atlassian.plugins.domain.model.plugin.Plugin 
import com.atlassian.plugins.service.plugin.PluginService 

import javax.xml.parsers.DocumentBuilderFactory 
import org.apache.maven.artifact.Artifact 
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.project.MavenProject;

/**
 * @requiresDependencyResolution runtime
 * @goal pac-check
 */
class PacCheckMojo extends AbstractStudioMojo {
    
    /**
     * The Maven Project
     *
     * @parameter expression="${project}"
     * @required
     * @readonly
     */
    MavenProject project
    
    /**
     * @parameter default-value="https://plugins.atlassian.com/server"
     */
    String pacBaseUrl
    
    /**
     * @parameter default-value="false"
     */
    boolean pacLogRequests
    
    volatile PacValidator validator;
    
    volatile PluginService pluginService
    
    volatile List<String> expands
    
    /**
     * @parameter default-value="false"
     */
    boolean fail
    
    /**
    * @parameter default-value="false"
    */
    boolean notFoundIgnored
    
    Locale userLocale = Locale.default
    
    static final String LS = System.getProperty("line.separator")
    
    static final String PLUGIN_DESCRIPTOR = "atlassian-plugin.xml"
    
    //FIXME copied from PackageWarMojo
    /**
     * Whether or not to take transitive dependencies in account.
     *
     * @parameter expression="${include.transitive.dependencies}" default-value="false"
     * @required
     */
    boolean includeTransitiveDependencies
    
    //FIXME copied from PackageWarMojo
    /**
     * The source web archive
     *
     * @parameter
     */
    String sourceArtifactId
    
    /**
     * @parameter default-value="target/validate"
     */
    String workDir
    
    /**
     * @parameter default-value="${localRepository}" 
     */
    private org.apache.maven.artifact.repository.ArtifactRepository localRepository;
    
    
    public void execute() throws MojoExecutionException, MojoFailureException {
        
        //FIXME filter RELEASE / LATEST / version ranges or resolve version from repository
        
        //FIXME copied from PackageWarMojo
        def artifacts = includeTransitiveDependencies ? project.artifacts : project.dependencyArtifacts;
        def sourceArtifacts = sourceArtifactId ? artifacts.findAll { it.artifactId == sourceArtifactId } :
        artifacts.findAll { it.type == 'war' || it.type == 'zip'}
        def webInfLibDependencies = artifacts.findAll {
            isJarToBePackaged(it) && it.scope != 'bundled' &&
                    it.scope != "gapps-bundled" && it.scope != "dev-bundled" && !sourceArtifacts.contains(it)
        }
        def bundledDependencies = artifacts.findAll {
            isJarToBePackaged(it) &&
                    it.scope == 'bundled' && !sourceArtifacts.contains(it)
        }
        def gappsBundledDependencies = artifacts.findAll {
            isJarToBePackaged(it) &&
                    it.scope == 'gapps-bundled' && !sourceArtifacts.contains(it)
        }
        def checkDependencies = new HashSet<Artifact>()
        checkDependencies.addAll(webInfLibDependencies)
        checkDependencies.addAll(bundledDependencies)
        checkDependencies.addAll(gappsBundledDependencies)
        //FIXME refactor stuff out in search, validation and reporting strategy
        Map<Artifact,List<Plugin>> artifactToPluginMap = search(checkDependencies)
        Map<Artifact,String> validationResultMap = validate(artifactToPluginMap)
        report(validationResultMap)
    }
    
    //FIXME copied from PackageWarMojo
    boolean isJarToBePackaged(Artifact artifact) {
        return artifact.artifactHandler.extension == 'jar' && artifact.scope != 'provided' && artifact.scope != 'test'
    }
    
    void report(Map<Artifact,String> validationResultMap) {
        String serializedValidationResult = serialize(validationResultMap)
        if(serializedValidationResult.length() > 0) {
            if(fail) {
                fail(serializedValidationResult)
            }
            else {
                log.warn(serializedValidationResult)
            }
        }
    }
    
    static String serialize ( Map<Artifact,String> validationResultMap) {
        StringBuilder sb = new StringBuilder()
        validationResultMap.each { artifact, validationResult ->
            sb << LS
            sb << validationResult
        }
        return sb.toString();
    }
    
    Map<Artifact,String> validate(Map<Artifact,Plugin> artifactToPluginMap) {
        Map<Artifact,String> validationResultMap=new HashMap<Artifact,String>()
        artifactToPluginMap.each {  artifact, plugins ->
            String validationResult = validator.validate (artifact, plugins, userLocale)
            if(validationResult!=null) {
                validationResultMap.put (artifact, validationResult)
            }
        }
        return validationResultMap
    }
    
    
    Map<Artifact,List<Plugin>> search(Set<Artifact> artifacts) {
        
        Map<Artifact,Plugin> pluginMap=new HashMap<Artifact,Plugin>()
        artifacts.each {artifact ->
            def pluginKey = getPluginKeyForArtifact(artifact)
            def pluginKeys = new LinkedList<String>()
            pluginKeys.add(pluginKey)
            List<Plugin> plugins = pluginService.pluginsByPluginKey(pluginKeys , 1, null, expands)
            pluginMap.put artifact, plugins
            
            //FIXME maybe fallback with category, vendor?
        }
        
        return pluginMap
    }
    
    String getPluginKeyForArtifact(Artifact artifact) {
        //FIXME improve/add exception handling
        
        String unzipDirString = localRepository.pathOf(artifact)
        unzipDirString = unzipDirString.substring(0,unzipDirString.length()-artifact.getFile().getName().length())
        def unzipDir = new File(workDir,unzipDirString)
        def ant = new AntBuilder();
        ant.unzip(  src: artifact.getFile(), dest: unzipDir,  overwrite:"true"){ 
            mapper(type:"flatten")
            patternset(){include(name: PLUGIN_DESCRIPTOR)}
        };
        def descriptor = new File(unzipDir,PLUGIN_DESCRIPTOR)
        def pluginKey = null
        if(descriptor.file) {
            def builder     = DocumentBuilderFactory.newInstance().newDocumentBuilder()
            def inputStream = new FileInputStream(descriptor)
            def plugin     = builder.parse(inputStream).documentElement
            pluginKey = plugin.attributes.getNamedItem('key').nodeValue
        }
        return pluginKey
    }
    
    //expands should be moved in search strategy
    List<String> getExpands() {
        if(expands==null) {
            synchronized(this) {
                if(expands==null) {
                    List<String> expands=new LinkedList<String>()
                    expands.add Plugin.EXPAND_LATEST_VERSION
                    this.expands=expands
                }
            }
        }
        return expands
    }
    
    public void setPluginService(PluginService pluginService) {
        this.pluginService = pluginService
    }
    
    public PluginService getPluginService() {
        if(pluginService == null) {
            synchronized(this) {
                if(pluginService == null) {
                    PluginServiceClientImpl pluginService = new PluginServiceClientImpl()
                    pluginService.setBaseUrl pacBaseUrl
                    pluginService.setLogWebResourceRequests pacLogRequests
                    this.pluginService = pluginService
                }
            }
        }
        return pluginService
    }
    
    public void setValidator(PacValidator validator) {
        this.validator = validator
    }
    
    public PacValidator getValidator() {
        if(validator==null) {
            synchronized(this) {
                if(validator==null) {
                    def validator = new DefaultPacValidator()
                    validator.notFoundIgnored = notFoundIgnored
                    this.validator = validator
                }
            }
        }
        return validator
    }
}
