package com.atlassian.maven.plugins.studio.prepare

/**
 * Prepares Studio for development and functional tests.  This includes preparing the studio home directory, and
 * setting up the studio test properties file.
 *
 * @goal prepare-studio
 */
public class PrepareStudioMojo extends AbstractPrepareFireballMojo
{

    /**
     * The location of studio data files for restoring from backup
     * @parameter expression="${studio.data.location}" default-value="${basedir}/target/test-classes/data"
     */
    String studioDataLocation

    /**
     * The zip file for SVN
     * @parameter default-value="${basedir}/src/test/resources/svnroot.zip"
     */
    String svnDataZip

    /**
     * @parameter expression="${fisheye.shutdown.enabled}" default-value="true"
     */
    boolean fisheyeShutdownEnabled

    public void execute()
    {
        if (skip) {
            log.info("Studio prepare has been skipped as maven.test.skip has been specified")
            return;
        }
        // Copy studio home stuff over
        ant.copyWithPerms(todir: studioHome) {
            fileset(dir: studioHomeData)
        }

        // Setup SVN
        ant.unzip(src: svnDataZip, dest: homeDirectory)

        // zip up the test data to be processed as a zip file, identified by directory name ending with .tozip
        zipUpTestData()

        if (svnPublicUrl != svnBaseUrl)
        {
            try
            {
                ant.exec(executable: "ln") {
                  arg(line: "-s ${convertToFilePath(svnBaseUrl)} ${convertToFilePath(svnPublicUrl)}")
                }
            }
            catch (Throwable t)
            {
                log.warn("Unable to create symlink [${svnPublicUrl}] to [${svnBaseUrl}]. Running on Windows? " +
                        "Subversion public url is now set to [${svnBaseUrl}]")
                svnPublicUrl = svnBaseUrl
            }
        }

        // Studio properties file
        parameterisePropertiesFile("${studioHome}/studio.properties")
        parameterisePropertiesFile("${studioHome}/studio-initial-data.properties")
        parameterisePropertiesFile("${studioHome}/studio-initial-data.xml")
        // Studio test properties file
        ant.copyWithPerms(tofile: "${project.build.directory}/test-classes/studiotest.properties",
                file: "${project.basedir}/src/test/resources/studiotest.properties.sample")
        parameterisePropertiesFile("${project.build.directory}/test-classes/studiotest.properties")

        // prepare WebDAV and SVN importer staging directories
        ant.sequential {
            echo("Preparing SVN Importer directories...")
            mkdir(dir:webDavHomeDirectory)
            mkdir(dir:"${webDavHomeDirectory}/dump_files")
            mkdir(dir:svnImportStagingDirectory)
            echo("done")
        }
    }

    void parameterisePropertiesFile(String file)
    {
        ant.replace(file: file)
        {
            replacefilter(token:"%JIRA-ATTACHMENTS%", value: ("${jiraHome}/attachments"))
            replacefilter(token:"%GREENHOPPER-LICENSE%", value: blankIfNull("${project.build.directory}/test-classes/greenhopper.license"))
            replacefilter(token:"%SVN-HOOKS%", value: blankIfNull("${svnHomeDirectory}/hooks"))
            replacefilter(token:"%JIRA-BASE-URL%", value: blankIfNull(getBaseUrl("jira")))
            replacefilter(token:"%JIRA-HOST-URL%", value: blankIfNull("http://localhost:${jiraHttpPort}"))
            replacefilter(token:"%JIRA-CONTEXT%", value: blankIfNull(jiraContextPath))
            replacefilter(token:"%CONFLUENCE-BASE-URL%", value: blankIfNull(getBaseUrl("confluence")))
            replacefilter(token:"%CONFLUENCE-HOST-URL%", value: blankIfNull("http://localhost:${confluenceHttpPort}"))
            replacefilter(token:"%CONFLUENCE-CONTEXT%", value: blankIfNull(confluenceContextPath))
            replacefilter(token:"%FISHEYE-BASE-URL%", value: blankIfNull(getBaseUrl("fisheye")))
            replacefilter(token:"%FISHEYE-HOST-URL%", value: blankIfNull("http://localhost:${fisheyeHttpPort}"))
            replacefilter(token:"%FISHEYE-CONTROL-PORT%", value: blankIfNull(fisheyeControlPort))
            replacefilter(token:"%FISHEYE-CONTEXT%", value: blankIfNull(fisheyeContextPath))
            replacefilter(token:"%FISHEYE-SHUTDOWN-ENABLED%", value: blankIfNull(fisheyeShutdownEnabled))
            replacefilter(token:"%BAMBOO-BASE-URL%", value: blankIfNull(getBaseUrl("bamboo")))
            replacefilter(token:"%BAMBOO-HOST-URL%", value: blankIfNull("http://localhost:${bambooHttpPort}"))
            replacefilter(token:"%BAMBOO-CONTEXT%", value: blankIfNull(bambooContextPath))
            replacefilter(token:"%CROWD-BASE-URL%", value: blankIfNull(getBaseUrl("crowd")))
            replacefilter(token:"%CROWD-HOST-URL%", value: blankIfNull("http://localhost:${crowdHttpPort}"))
            replacefilter(token:"%CROWD-CONTEXT%", value: blankIfNull(crowdContextPath))
            replacefilter(token:"%SVN-BASE-URL%", value: blankIfNull(svnBaseUrl))
            replacefilter(token:"%SVN-PUBLIC-URL%", value: blankIfNull(svnPublicUrl))
            replacefilter(token:"%BAMBOO-ENABLED%", value: blankIfNull(bambooEnabled))
            replacefilter(token:"%STUDIO-DATA-LOCATION%", value: blankIfNull(studioDataLocation))
            replacefilter(token:"%STUDIO-HOME%", value: blankIfNull(studioHome))
            replacefilter(token:"%GAPPS-ENABLED%", value: blankIfNull(gappsEnabled))
            replacefilter(token:"%STUDIO-GAPPS-DOMAIN%", value: blankIfNull(gappsDomain))
            replacefilter(token:"%STUDIO-WEBDAV-DIRECTORY%", value: blankIfNull(webDavHomeDirectory))
            replacefilter(token:"%STUDIO-SVN-IMPORT-STAGING-DIRECTORY%", value: blankIfNull(svnImportStagingDirectory))
        }
    }

    void zipUpTestData()
    {
        log.info("Zipping up test data files...")
        new File("${project.build.directory}/test-classes/data").listFiles({ file ->
            file.isDirectory() && file.getAbsolutePath().endsWith(".tozip")
        } as FileFilter).each { dir ->
            def destfilename = dir.absolutePath.replaceFirst('\\.tozip$', ".zip")
            log.info("Zipping $dir to $destfilename")
            ant.zipWithPerms(basedir: dir, destfile: destfilename)
        }
    }

    Object blankIfNull(Object value)
    {
        if (value == null)
        {
            return ""
        }
        return value;
    }

    String convertToFilePath(String fileUrl)
    {
        new File(new URI(fileUrl)).getAbsolutePath()
    }

}
