package com.atlassian.maven.plugins.sandbox.scm;

import com.atlassian.maven.plugins.sandbox.ScmException;
import org.apache.maven.plugin.logging.Log;
import org.codehaus.plexus.util.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;

public class HgHandler implements ScmHandler
{
    private static final String HG = "hg";
    private final Log log;
    private final CommandRunner commandRunner;

    public HgHandler(final Log log)
    {
        this.log = log;
        commandRunner = new CommandRunner(log);
    }


    @NotNull
    public String getBranch() throws ScmException
    {
        final ArrayList<String> output = new ArrayList<String>();
        final int exitCode = commandRunner.runCommand(output, HG, "branch");
        if (exitCode!=0 || output.size()!=1)
        {
            throw new ScmException("Unable to get the current branch name", exitCode, output);
        }
        return output.get(0).trim();
    }

    public void assertValidWorkspace(final String urlWeWantToPushTo, final String expectedBranch, final String originalWorkspace) throws ScmException
    {
        final ArrayList<String> output = new ArrayList<String>();
        final int exitCode = commandRunner.runCommand(output, HG, "incoming", urlWeWantToPushTo);
        if (exitCode != 0 && exitCode != 1)
        {
            throw new ScmException("An error has occured when matching local workspace with repository at " + urlWeWantToPushTo + ", consider using " + originalWorkspace, exitCode, output);
        }

        final String workspaceBranch = getBranch();
        if (!workspaceBranch.equals(expectedBranch))
        {
            throw new ScmException("Workspace branch [" + workspaceBranch + "] does not match sandbox branch: [" + expectedBranch + "]" + ", consider using " + originalWorkspace);
        }
    }

    public void createRemoteTagAndUpdateToIt(final String destinationUrl, final String revisionToTag, final String tagName, final String commitMessage) throws ScmException
    {
        final ArrayList<String> output = new ArrayList<String>();
        int exitCode = commandRunner.runCommand(output, HG, "pull", "-u", destinationUrl);
        if (exitCode != 0)
        {
            throw new ScmException("Unable to pull from " + destinationUrl, exitCode, output);
        }

        exitCode = commandRunner.runCommand(output, HG, "tag", "--force", "--message", commitMessage, "--rev", revisionToTag, tagName);
        if (exitCode != 0)
        {
            throw new ScmException("Unable to tag the local repository with " + tagName, exitCode, output);
        }

        exitCode = commandRunner.runCommand(output, HG, "push", destinationUrl);
        if (exitCode != 0)
        {
            throw new ScmException("Unable to push out tag commit to " + destinationUrl);
        }

        exitCode = commandRunner.runCommand(output, HG, "up", "-r", revisionToTag); //hmm, should be tagName?
        if (exitCode != 0)
        {
            log.warn("Unable to update workspace to " + revisionToTag);
        }
    }

    @Nullable
    public String getTagString(final String tagName)
    {
        final String hgTagName = StringUtils.strip(tagName, "/");

        return StringUtils.isEmpty(hgTagName) ? null : hgTagName;
    }
}
