package com.atlassian.maven.plugins.sandbox.scm;

import com.atlassian.maven.plugins.sandbox.ScmException;
import org.apache.maven.plugin.logging.Log;
import org.codehaus.plexus.util.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;

public class GitHandler implements ScmHandler
{
    private final CommandRunner commandRunner;
    private final Log log;
    private final static String GIT = "git";

    public GitHandler(final Log log)
    {
        this.log = log;
        commandRunner = new CommandRunner(log);
    }

    @NotNull
    public String getBranch() throws ScmException
    {
        final ArrayList<String> output = new ArrayList<String>();
        final int exitCode = commandRunner.runCommand(output, GIT, "rev-parse", "--abbrev-ref", "HEAD");
        if (exitCode!=0 || output.size()!=1)
        {
            throw new ScmException("Unable to get the current branch name", exitCode, output);
        }
        return output.get(0).trim();
    }

    public void assertValidWorkspace(final String urlWeWantToPushTo, final String expectedBranch, final String originalWorkspace) throws ScmException
    {
        final ArrayList<String> output = new ArrayList<String>();
        final int exitCode = commandRunner.runCommand(output, GIT, "fetch", "--dry-run", urlWeWantToPushTo);
        if (exitCode != 0)
        {
            throw new ScmException("An error has occured when matching local workspace with repository at " + urlWeWantToPushTo + ", consider using " + originalWorkspace, exitCode, output);
        }
        for (final String s : output)
        {
            if (s.contains("no common commits"))
            {
                throw new ScmException("Local workspace is unrelated to " + urlWeWantToPushTo + ", consider using " + originalWorkspace, exitCode, output);
            }
        }

        final String workspaceBranch = getBranch();
        if (!workspaceBranch.equals(expectedBranch))
        {
            throw new ScmException("Workspace branch [" + workspaceBranch + "] does not match sandbox branch: [" + expectedBranch + "]" + ", consider using " + originalWorkspace);
        }
    }

    public void createRemoteTagAndUpdateToIt(final String destinationUrl, final String revisionToTag, final String tagName, final String commitMessage) throws ScmException
    {
        final ArrayList<String> output = new ArrayList<String>();
        int exitCode = commandRunner.runCommand(output, GIT, "fetch", destinationUrl);
        if (exitCode != 0)
        {
            throw new ScmException("Unable to pull from " + destinationUrl, exitCode, output);
        }

        exitCode = commandRunner.runCommand(output, GIT, "tag", "--force", "-m", commitMessage, tagName, revisionToTag);
        if (exitCode != 0)
        {
            throw new ScmException("Unable to tag the local repository with " + tagName, exitCode, output);
        }

        exitCode = commandRunner.runCommand(output, GIT, "push", destinationUrl, tagName);
        if (exitCode != 0)
        {
            throw new ScmException("Unable to push out tag commit to " + destinationUrl);
        }

        exitCode = commandRunner.runCommand(output, GIT, "checkout", tagName);
        if (exitCode != 0)
        {
            log.warn("Unable to update workspace to " + tagName);
        }
    }

    @Nullable
    public String getTagString(final String tagName)
    {
        final String finalTagName = StringUtils.strip(tagName, "/");

        return StringUtils.isEmpty(finalTagName) ? null : finalTagName;
    }
}
