package com.atlassian.maven.plugins.sandbox;

import com.atlassian.sandbox.model.SandboxRepository;
import com.atlassian.sandbox.model.SandboxResult;
import com.google.gson.Gson;
import org.apache.commons.io.FileUtils;
import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.artifact.repository.ArtifactRepositoryFactory;
import org.apache.maven.artifact.repository.layout.DefaultRepositoryLayout;
import org.apache.maven.model.DeploymentRepository;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.codehaus.plexus.util.StringUtils;

import java.io.File;
import java.io.IOException;

/**
 * Modifies the distribution management of the project so that all builds dep
 *
 * @phase verify
 * @goal sandbox
 */
public class SandboxMojo extends AbstractSandboxMojo
{
    private static final Gson GSON = new Gson();

    private static final String SANDBOX_RESULT_FILE_NAME = "sandbox.json";

    /**
     * @component
     */
    private ArtifactRepositoryFactory repositoryFactory;

    /**
     * The output directory of the project.
     *
     * @parameter default-value="${project.build.directory}" expression="${sandbox.outputDirectory}"
     */
    protected String outputDirectory;

    public void execute()
        throws MojoExecutionException, MojoFailureException
    {
        if (isEnabled())
        {
            getLog().info("Sandbox enabled");
            setArtifactRepositories();

            SandboxResult sandboxResult = createResult();
            writeResult(sandboxResult);
        }
    }

    private SandboxResult createResult() throws MojoExecutionException
    {
        SandboxResult sandboxResult = new SandboxResult();
        final ArtifactRepository artifactRepository = getProject().getDistributionManagementArtifactRepository();
        SandboxRepository sandboxRepository = new SandboxRepository(artifactRepository.getId(), artifactRepository.getUrl());
        sandboxResult.addRepository(sandboxRepository);
        return sandboxResult;
    }

    private void writeResult(final SandboxResult sandboxResult) throws MojoExecutionException
    {
        final File sandboxFile = new File(outputDirectory, SANDBOX_RESULT_FILE_NAME);
        if (!sandboxFile.getParentFile().exists())
        {
            sandboxFile.getParentFile().mkdirs();
        }

        try
        {
            FileUtils.writeStringToFile(sandboxFile, GSON.toJson(sandboxResult));
        }
        catch (IOException e)
        {
            throw new MojoExecutionException("Could not write sandbox result for this module to '" + sandboxFile.getAbsolutePath() + "'", e);
        }
    }

    private void setArtifactRepositories() throws MojoExecutionException
    {
        getLog().info("Sandboxing deployment repositories to " + getRepositorySandboxUrl());
        //Set release
        ArtifactRepository releaseRepository = createDeployableArtifactRepository(getProject().getDistributionManagement().getRepository());
        getProject().setReleaseArtifactRepository(releaseRepository);

        //Set snapshot
        ArtifactRepository snapshotRepository = createDeployableArtifactRepository(getProject().getDistributionManagement().getSnapshotRepository());
        getProject().setSnapshotArtifactRepository(snapshotRepository);
    }

    private ArtifactRepository createDeployableArtifactRepository(DeploymentRepository deploymentRepository) throws MojoExecutionException
    {
        final String newDeploymentUrl;
        if(!getRepositorySandboxUrl().startsWith("dav:"))
        {
            newDeploymentUrl = "dav:" + getRepositorySandboxUrl() + "/" + getRepositoryName(deploymentRepository.getUrl());
        }
        else
        {
            newDeploymentUrl = getRepositorySandboxUrl() + "/" + getRepositoryName(deploymentRepository.getUrl());    
        }
        getLog().info("Repository " + deploymentRepository.getUrl() + " has been sandboxed to " + newDeploymentUrl);
        return repositoryFactory.createDeploymentArtifactRepository(deploymentRepository.getId(), newDeploymentUrl, new DefaultRepositoryLayout(), deploymentRepository.isUniqueVersion());
    }

    public String getRepositoryName(String url)
    {
        String[] parts = StringUtils.split(url, "/");
        if (parts.length > 0)
        {
            return parts[parts.length - 1];
        }
        return null;
    }
}
