package com.atlassian.maven.plugins.sandbox;

import com.atlassian.maven.plugins.sandbox.scm.ScmConnectionWrapper;
import com.atlassian.maven.plugins.sandbox.scm.ScmHandler;
import com.atlassian.maven.plugins.sandbox.scm.ScmHandlerFactory;
import com.atlassian.maven.plugins.sandbox.scm.ScmRoHandler;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.shared.release.versions.DefaultVersionInfo;
import org.apache.maven.shared.release.versions.VersionInfo;
import org.apache.maven.shared.release.versions.VersionParseException;
import org.codehaus.plexus.components.interactivity.Prompter;
import org.codehaus.plexus.components.interactivity.PrompterException;
import org.codehaus.plexus.util.StringUtils;
import org.codehaus.plexus.util.xml.Xpp3Dom;

import java.net.InetAddress;
import java.net.UnknownHostException;

import static org.twdata.maven.mojoexecutor.MojoExecutor.configuration;
import static org.twdata.maven.mojoexecutor.MojoExecutor.element;
import static org.twdata.maven.mojoexecutor.MojoExecutor.executeMojo;
import static org.twdata.maven.mojoexecutor.MojoExecutor.executionEnvironment;
import static org.twdata.maven.mojoexecutor.MojoExecutor.goal;

/**
 * @goal release
 * @requiresProject true
 * @aggregator
 */
public class ReleaseMojo extends AbstractSandboxMojo
{
    // ------------------------------------------------------------------------------------------------------- Constants
    // ------------------------------------------------------------------------------------------------- Type Properties

    /**
     * Component used to prompt for input.
     *
     * @component
     */
    private Prompter prompter;

    /**
     * @parameter expression="${sandbox.developmentVersion}"
     */
    private String developmentVersion;

    /**
     * @parameter expression="${sandbox.releaseVersion}"
     */
    private String releaseVersion;

    /**
     * @parameter expression="${sandbox.extraArguments}"
     */
    private String extraArguments;

    /**
     * @parameter expression="${sandbox.extraPrepareArguments}"
     */
    private String extraPrepareArguments;

    /**
     * @parameter expression="${sandbox.extraPerformArguments}"
     */
    private String extraPerformArguments;

    private String sandboxTagUrl;
    private String releaseTagUrl;
    private ScmHandler scmDevHandler;
    private ScmRoHandler scmReadOnlyHandler;

    // ---------------------------------------------------------------------------------------------------- Dependencies
    // ---------------------------------------------------------------------------------------------------- Constructors
    // ----------------------------------------------------------------------------------------------- Interface Methods
    // -------------------------------------------------------------------------------------------------- Action Methods
    // -------------------------------------------------------------------------------------------------- Public Methods

    @Override
    public void execute() throws MojoExecutionException, MojoFailureException
    {
        scmDevHandler = ScmHandlerFactory.createScmUtils(getLog(), getProject().getScm());
        scmReadOnlyHandler = ScmHandlerFactory.createScmRoUtils(getLog(), getProject().getScm());
        checkSandboxKeySpecified();

        /**
         * hack to get around our svn replication being slow. You might want to re-run the release goal without going through this again
         */
        if (isPromptAndMetadataStorageRequired())
        {
            getLog().info("Staging release at " + getRepositorySandboxUrl());

            promptForVersions();

            validateReleaseAndDevelopmentVersions();

            storeSandboxMetadata();
        }
        else
        {
            sandboxTagUrl= scmReadOnlyHandler.getTagString(getTagName());
            releaseTagUrl = scmReadOnlyHandler.getTagString(getProject().getArtifactId() + "-" + releaseVersion);
        }

        makeSessionNonInteractive();

        executeMojo(RELEASE_PLUGIN,
                    goal("prepare"),
                    getConfigurationForPrepare(),
                    executionEnvironment(
                            getProject(),
                            getSession(),
                            getBuildPluginManager()
                    ));

        getLog().info("Executing 'perform' goal of release plugin " + RELEASE_PLUGIN.getVersion());
        executeMojo(RELEASE_PLUGIN,
                    goal("perform"),
                    getConfigurationForPerform(),
                    executionEnvironment(
                            getProject(),
                            getSession(),
                            getBuildPluginManager()
                    ));

        printPostReleaseHelp();
    }

    private boolean isPromptAndMetadataStorageRequired() throws MojoFailureException
    {
        try
        {
            VersionInfo versionInfo = new DefaultVersionInfo(getProject().getVersion());
            return versionInfo.isSnapshot();
        }
        catch (VersionParseException e)
        {
            throw new MojoFailureException(e.getMessage(), e);
        }
    }

    private void promptForVersions() throws MojoFailureException
    {
        if (StringUtils.isEmpty(releaseVersion) || StringUtils.isEmpty(developmentVersion))
        {
            getLog().info("Press enter to accept defaults");
        }

        try
        {
            if (StringUtils.isEmpty(releaseVersion))
            {
                VersionInfo currentVersionInfo = new DefaultVersionInfo(getProject().getVersion());
                releaseVersion = prompter.prompt("What is the release version for \"" + getProject().getName() + "\"?", currentVersionInfo.getReleaseVersionString());
            }

            if (StringUtils.isEmpty(developmentVersion))
            {
                VersionInfo currentVersionInfo = new DefaultVersionInfo(releaseVersion);
                developmentVersion = currentVersionInfo.getNextVersion().getSnapshotVersionString();
                developmentVersion = prompter.prompt("What is the new development version for \"" + getProject().getName() + "\"?", developmentVersion);
            }
        }
        catch (VersionParseException e)
        {
            throw new MojoFailureException(e.getMessage(), e);
        }
        catch (PrompterException e)
        {
            throw new MojoFailureException(e.getMessage(), e);
        }
    }

    private void validateReleaseAndDevelopmentVersions() throws MojoFailureException, MojoExecutionException
    {
        if (StringUtils.isNotEmpty(developmentVersion) && !developmentVersion.endsWith("-SNAPSHOT"))
        {
            throw new MojoFailureException("-Dsandbox.developmentVersion should end with -SNAPSHOT");
        }

        if (StringUtils.isNotEmpty(releaseVersion) && releaseVersion.endsWith("-SNAPSHOT"))
        {
            throw new MojoExecutionException("-Dsandbox.releaseVersion should NOT end with -SNAPSHOT");
        }
    }

    private void storeSandboxMetadata() throws MojoFailureException
    {
        sandboxTagUrl = scmReadOnlyHandler.getTagString(getTagName());
        releaseTagUrl = scmReadOnlyHandler.getTagString(getProject().getArtifactId() + "-" + releaseVersion);

        getLog().info("Release will be tagged at '" + sandboxTagUrl + "'");

        getSandboxService().storeReleaseMetadata(getSandboxKey(), sandboxTagUrl, releaseTagUrl);

        final String readOnlyConnection = getProject().getScm().getConnection();
        setPropertyIfNotBlank(SandboxMetadataProperties.SCM_CONNECTION, readOnlyConnection);
        final String readWriteConnection = getProject().getScm().getDeveloperConnection();
        setProperty(SandboxMetadataProperties.SCM_DEVELOPER_CONNECTION, readWriteConnection);
        setProperty(SandboxMetadataProperties.WORKSPACE, getWorkspace());

        final String branchName;
        try
        {
            branchName = scmDevHandler.getBranch();
        }
        catch (ScmException e)
        {
            throw new MojoFailureException("Unable to retrieve branch name", e);
        }

        setProperty(SandboxMetadataProperties.BRANCH, branchName);
        
        getLog().info("Successfully stored the release metadata.");
    }

    String getWorkspace()
    {
        String hostName;
        try
        {
            hostName = InetAddress.getLocalHost().getHostName();
        }
        catch (UnknownHostException e)
        {
            hostName = "unknown";
        }

        return hostName + ":" + getSession().getExecutionRootDirectory();
    }

    private Xpp3Dom getConfigurationForPrepare()
    {
        final StringBuilder sb = new StringBuilder();

        //Append extra arguments if there are any available
        String customSettingsArgument = getCustomSettingsArgument();
        if (StringUtils.isNotEmpty(customSettingsArgument))
        {
            sb.append(" ");
            sb.append(customSettingsArgument);
        }
        if (StringUtils.isNotEmpty(extraArguments))
        {
            sb.append(" ");
            sb.append(extraArguments);
        }
        if (StringUtils.isNotEmpty(extraPrepareArguments))
        {
            sb.append(" ");
            sb.append(extraPrepareArguments);
        }

        if (StringUtils.isNotEmpty(developmentVersion) && StringUtils.isNotEmpty(releaseVersion))
        {
            return configuration(
                    element("releaseVersion", releaseVersion),
                    element("developmentVersion", developmentVersion),
                    element("tag", getTagName()),
                    element("arguments", sb.toString())
            );
        }
        return configuration();
    }

    private Xpp3Dom getConfigurationForPerform()
    {
        StringBuilder sb = new StringBuilder();
        sb.append(" -Prelease");
        sb.append(" -Dsandbox.sandboxRepositoryUrl=").append(getSandboxRepositoryUrl());
        sb.append(" -Dsandbox.key=").append(getSandboxKey());
        sb.append(" -Dsandbox.repositoryBaseUrl=").append(getMavenRepositoryBaseUrl());
        sb.append(" -Dsandbox.promoterUrl=").append(getPromoterUrl());

        //Append extra arguments if there are any available
        String customSettingsArgument = getCustomSettingsArgument();
        if (StringUtils.isNotEmpty(customSettingsArgument))
        {
            sb.append(" ");
            sb.append(customSettingsArgument);
        }
        if (StringUtils.isNotEmpty(extraPerformArguments))
        {
            sb.append(" ");
            sb.append(extraPerformArguments);
        }

        if (StringUtils.isNotEmpty(extraArguments))
        {
            sb.append(" ");
            sb.append(extraArguments);
        }

        return configuration(
                element("arguments", sb.toString())
        );
    }

    // -------------------------------------------------------------------------------------- Basic Accessors / Mutators

    private String getTagName()
    {
        return "sandbox-" + getSandboxKey();
    }

    private ScmConnectionWrapper getRepositoryUrl() throws MojoFailureException
    {
        return new ScmConnectionWrapper(getProject().getScm().getConnection());
    }

    private void printPostReleaseHelp()
    {
        printLine();
        getLog().info("Release has been successfully built");
        printLine();

        printBlankLine();
        getLog().info("Sandbox: " + getRepositorySandboxUrl());
        getLog().info("Sandbox Tag: '" + sandboxTagUrl + "'");
        getLog().info("Release Tag: '" + releaseTagUrl + "' (Does not exist until the release is promoted)");
        printBlankLine();

        printLine();
        getLog().info("To promote your release (after manual validation) run:");
        getLog().info('\t' + "mvn sandbox:promote -Dsandbox.key=" + getSandboxKey());
        printLine();
    }

    private void printLine()
    {
        getLog().info("------------------------------------------------------------------------");
    }

    private void printBlankLine()
    {
        getLog().info("");
    }
}
