package com.atlassian.maven.plugins.sandbox;

import com.atlassian.sandbox.promotion.api.SandboxService;
import com.atlassian.util.concurrent.Lazy;
import com.atlassian.util.concurrent.Supplier;
import com.atlassian.xmlrpc.RedstoneBinder;
import org.apache.maven.execution.MavenSession;
import org.apache.maven.model.Plugin;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.BuildPluginManager;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.project.MavenProject;
import org.apache.maven.settings.Server;
import org.apache.maven.settings.Settings;
import org.codehaus.plexus.util.StringUtils;
import org.sonatype.plexus.components.sec.dispatcher.SecDispatcher;
import org.sonatype.plexus.components.sec.dispatcher.SecDispatcherException;

import java.io.File;
import java.net.URL;
import java.util.Map;

import static org.twdata.maven.mojoexecutor.MojoExecutor.plugin;

public abstract class AbstractSandboxMojo extends AbstractMojo
{
    // ------------------------------------------------------------------------------------------------------- Constants

    protected static final Plugin RELEASE_PLUGIN = plugin("org.apache.maven.plugins", "maven-release-plugin", "2.5"); //should match pom.xml of sandbox plugin

    // ------------------------------------------------------------------------------------------------- Type Properties

    private final Supplier<SandboxService> sandboxService = Lazy.supplier(new Supplier<SandboxService>()
    {
        public SandboxService get()
        {
            try
            {
                getLog().info("Connecting to promoter service at " + promoterUrl);
                return new RedstoneBinder().bind(SandboxService.class, new URL(promoterUrl));
            }
            catch (Exception e)
            {
                throw new RuntimeException("Could not bind to Sandbox promoter service: " + e.getMessage(), e);
            }
        }
    });

    /**
     * The Maven Project
     *
     * @parameter expression="${project}"
     * @required
     * @readonly
     */
    private MavenProject project;

    /**
     * The Maven Session Object
     *
     * @parameter expression="${session}"
     * @required
     * @readonly
     */
    private MavenSession session;

    /**
     * The Maven PluginManager Object
     *
     * @component
     * @required
     */
    private BuildPluginManager buildPluginManager;

    /**
     * @parameter expression="${settings}
     * @readonly
     */
    private Settings settings;

    /**
     * Used for auth details
     *
     * @parameter expression="${sandbox.serverId}" default-value="atlassian-private"
     * @required
     */
    private String serverId;

    /**
     * URL to the sandbox repository server
     *
     * @parameter expression="${sandbox.repositoryUrl}" default-value="https://sandbox.atlassian.com/sandbox/"
     */
    private String sandboxRepositoryUrl;

    /**
     * @parameter expression="${sandbox.key}"
     */
    private String sandboxKey;

    /**
     * @parameter expression="${sandbox.baseRepositoryUrl}" default-value="https://maven.atlassian.com/"
     */
    private String repositoryBaseUrl;

    /**
     * @parameter expression="${sandbox.promoterUrl}" default-value="http://sandbox-promoter.buildeng.atlassian.com:8080/xmlrpc/"
     */
    private String promoterUrl;

    /**
     * @parameter expression="${sandbox.maven.settings}"
     */
    private String sandboxSettingsFilename;

    /** @component role-hint="mng-4384" */
    private SecDispatcher secDispatcher;

    protected final Supplier<Map<String, String>> sandbox2releaseTags = Lazy.supplier(new Supplier()
    {
        public Map<String, String> get()
        {
            return sandboxService.get().getReleasesInSandbox(sandboxKey);
        }
    });

    // ---------------------------------------------------------------------------------------------------- Dependencies
    // ---------------------------------------------------------------------------------------------------- Constructors
    // ----------------------------------------------------------------------------------------------- Interface Methods
    // -------------------------------------------------------------------------------------------------- Action Methods
    // -------------------------------------------------------------------------------------------------- Public Methods

    protected boolean isEnabled()
    {
        return StringUtils.isNotEmpty(getSandboxRepositoryUrl()) &&
               StringUtils.isNotEmpty(getSandboxKey()) &&
               StringUtils.isNotEmpty(getMavenRepositoryBaseUrl());
    }

    // -------------------------------------------------------------------------------------- Basic Accessors / Mutators

    protected MavenProject getProject()
    {
        return project;
    }

    protected MavenSession getSession()
    {
        return session;
    }

    protected BuildPluginManager getBuildPluginManager()
    {
        return buildPluginManager;
    }

    protected Settings getSettings()
    {
        return settings;
    }

    protected String getSandboxRepositoryRootUrl()
    {
        if (!StringUtils.isEmpty(sandboxRepositoryUrl))
        {
            return StringUtils.strip(sandboxRepositoryUrl);
        }
        return sandboxRepositoryUrl;
    }

    protected String getRepositorySandboxUrl()
    {
        String url = getSandboxRepositoryRootUrl();
        if (!url.endsWith("/"))
        {
            url += "/";
        }

        return url + getSandboxKey() + "/";
    }

    protected String getSandboxRepositoryUrl()
    {
        return sandboxRepositoryUrl;
    }

    protected String getSandboxKey()
    {
        return sandboxKey;
    }

    protected String getMavenRepositoryBaseUrl()
    {
        return repositoryBaseUrl;
    }

    protected String getPromoterUrl()
    {
        return promoterUrl;
    }

    protected Server getServer()
    {
        return settings.getServer(serverId);
    }

    protected String getSandboxSettingsFilename()
    {
        return sandboxSettingsFilename;
    }


    protected void setProperty(String propertyKey, String propertyValue) throws MojoFailureException
    {
        getSandboxService().setProperty(getSandboxKey(), propertyKey, propertyValue);
    }

    protected String getProperty(final String propertyKey) throws MojoFailureException
    {
        return getProperties().get(propertyKey);
    }

    protected void setPropertyIfNotBlank(String propertyKey, String propertyValue) throws MojoFailureException
    {
        if (propertyValue!=null && !propertyValue.trim().isEmpty())
        {
            getSandboxService().setProperty(getSandboxKey(), propertyKey, propertyValue);
        }
    }

    protected Map<String, String> getProperties() throws MojoFailureException
    {
        return getSandboxService().getProperties(sandboxKey);
    }

    protected String getCustomSettingsArgument()
    {
        final StringBuilder sb = new StringBuilder();
        if(sandboxSettingsFilename == null){
            return sb.toString();
        }
        
        final File settingsPath = new File(sandboxSettingsFilename);
        if (settingsPath.exists())
        {
            sb.append("-s ");
            sb.append(settingsPath.getAbsolutePath());
        }
        return sb.toString();
    }

    protected SandboxService getSandboxService()
    {
        return sandboxService.get();
    }


    protected void checkSandboxKeySpecified() throws MojoFailureException
    {
        if (StringUtils.isEmpty(getSandboxKey()))
        {
            throw new MojoFailureException("-Dsandbox.key was not specified or was empty");
        }
    }

    protected void makeSessionNonInteractive()
    {
        settings.setInteractiveMode(false);
    }

    protected String decryptPasswordIfNecessary(String password)
    {
        if (secDispatcher != null)
        {
            try
            {
                password = secDispatcher.decrypt(password);
            }
            catch (SecDispatcherException e)
            {
                getLog().error("SecDispatcher error decrypting password.");
            }
        }
        else
        {
            getLog().info("Could not find SecDispatcher, won't be able to decrypt passwords.");
        }

        return password;
    }
}
