package com.atlassian.maven.plugins.sandbox.scm;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import com.atlassian.maven.plugins.sandbox.ScmException;
import org.apache.maven.plugin.logging.Log;
import org.codehaus.plexus.util.StringUtils;
import org.jetbrains.annotations.Nullable;

/**
 * Lifted from org.apache.maven.scm.provider.svn.SvnTagBranchUtils
 */
public class SvnUtil
{
    public static final String SVN_TRUNK = "trunk";

    public static final String SVN_BRANCHES = "branches";

    public static final String SVN_TAGS = "tags";

    public static final String[] SVN_BASE_DIRS = new String[]{SVN_TRUNK, SVN_BRANCHES, SVN_TAGS};

    private SvnUtil()
    {
    }


    @Nullable
    public static String resolveTagUrl(String repositoryUrl, String branchTagName)
    {
        String projectRoot = getProjectRoot(repositoryUrl);
        branchTagName = StringUtils.strip(branchTagName, "/");

        if (StringUtils.isEmpty(branchTagName))
        {
            return null;
        }

        // Look for a query string as in ViewCVS urls
        String queryString = null;
        if (repositoryUrl.contains("?"))
        {
            queryString = repositoryUrl.substring(repositoryUrl.indexOf("?"));
            // if repositoryUrl contains a query string, remove it from repositoryUrlRoot; will be re-appended later
            projectRoot = StringUtils.replace(projectRoot, queryString, "");
        }

        if (branchTagName.contains("://"))
        {
            // branch/tag is already an absolute url so just return it.
            return branchTagName;
        }

        // Look for any "branches/" or "tags/" specifiers in the branchTagName. If one occurs,
        // don't append the subdir to the projectRoot when appending the name
        for (final String SVN_BASE_DIR : SVN_BASE_DIRS)
        {
            if (branchTagName.startsWith(SVN_BASE_DIR + "/"))
            {
                return addSuffix(appendPath(projectRoot, branchTagName), queryString);
            }
        }

        return addSuffix(appendPath(appendPath(projectRoot, SVN_TAGS), branchTagName), queryString);
    }

    private static String getProjectRoot(String repoPath)
    {
        for (final String SVN_BASE_DIR : SVN_BASE_DIRS)
        {
            String base = "/" + SVN_BASE_DIR;
            int pos = repoPath.lastIndexOf(base + "/");
            if (repoPath.endsWith(base))
            {
                return repoPath.substring(0, repoPath.length() - base.length());
            }
            else if (pos >= 0)
            {
                return repoPath.substring(0, pos);
            }
        }

        // At this point we were unable to locate the project root of this url
        // so assume that the repository url specified is the project root
        return appendPath(repoPath, null);
    }

    private static String appendPath(String basePath, @Nullable String addlPath)
    {
        basePath = StringUtils.stripEnd(basePath, "/");

        if (StringUtils.isEmpty(addlPath))
        {
            return basePath;
        }
        else
        {
            return basePath + "/" + StringUtils.stripStart(addlPath, "/");
        }
    }

    private static String addSuffix(String baseString, String suffix)
    {
        return (suffix != null) ? baseString + suffix : baseString;
    }

    public static int tag(final Log log, final String sandboxTagUrl, final String releaseTagUrl, final String commitMessage) throws ScmException
    {
        String command = null;
        try
        {
            ProcessBuilder processBuilder = new ProcessBuilder("svn", "cp", sandboxTagUrl, releaseTagUrl, "-m", commitMessage);

            command = org.apache.commons.lang.StringUtils.join(processBuilder.command(), ' ');

            log.info("Running '" + command + "'");

            Process process = processBuilder.start();

            return process.waitFor();
        }
        catch (Exception e)
        {
            throw new ScmException("An error has occured when running svn command: " + command, e);
        }
    }
}
