package com.atlassian.maven.plugins.sandbox;

import com.atlassian.maven.plugins.sandbox.scm.ScmConnectionWrapper;
import com.atlassian.maven.plugins.sandbox.scm.ScmHandler;
import com.atlassian.maven.plugins.sandbox.scm.ScmHandlerFactory;
import com.atlassian.sandbox.promotion.api.SandboxService;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.settings.Server;

import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * Promote a Sandbox from the command line
 *
 * @goal promote
 * @requiresProject false
 */
public class PromoteMojo extends AbstractSandboxMojo
{
    // ------------------------------------------------------------------------------------------------- Type Properties

    /**
     * @parameter expression="${sandbox.validatePromotion}" default-value="true"
     */
    private boolean validation;

    /**
     * @parameter expression="${sandbox.skipTagging}" default-value="false"
     */
    private boolean skipTagging;
    private ScmConnectionWrapper developerConnection; //this will be filled in only when skipTagging = false, NPE otherwise
    private ScmHandler scmHandler;

    // ---------------------------------------------------------------------------------------------------- Dependencies
    // ---------------------------------------------------------------------------------------------------- Constructors
    // ----------------------------------------------------------------------------------------------- Interface Methods
    // -------------------------------------------------------------------------------------------------- Action Methods
    // -------------------------------------------------------------------------------------------------- Public Methods

    public void execute() throws MojoExecutionException, MojoFailureException
    {
        scmHandler = ScmHandlerFactory.createScmUtils(getLog(), getProject().getScm().getDeveloperConnection());
        SandboxService sandboxService = getSandboxService();

        Server server = getServer();

        checkSandboxKeySpecified();

        if (isAlreadyPromoted())
        {
            throw new MojoExecutionException("Sandbox '" + getSandboxKey() + "' has already been promoted. To override specify -Dsandbox.validatePromotion=false.");
        }

        if (!skipTagging)
        {
            String developerConnectionString = getProperty(SandboxMetadataProperties.SCM_DEVELOPER_CONNECTION);
            if (developerConnectionString==null)
            {
                //sandbox release has been created by an older (Subversion-only) version of the plugin,
                //so we cannot use the pom.xml connection string and we don't have it in sandbox properties
                //either. Fortunately, for Subversion we store the full tag paths anyway.
                developerConnectionString = "scm:svn:http://host/path";
            }
            developerConnection = new ScmConnectionWrapper(developerConnectionString);
            try
            {
                assertValidWorkspace();
            }
            catch (Exception e)
            {
                throw new MojoExecutionException("Release cannot be tagged. To override specify -Dsandbox.skipTagging=true.", e);
            }
        }

        String password = decryptPasswordIfNecessary(server.getPassword());

        if (!sandboxService.exists(getSandboxKey(), server.getUsername(), password, getSandboxRepositoryRootUrl()))
        {
            throw new MojoFailureException("Sandbox '" + getSandboxKey() + "' does not exist");
        }

        if (!sandboxService.isPromoting(getSandboxKey()))
        {
            markAsPromoted();
            sandboxService.promote(getSandboxKey(), server.getUsername(), password, getSandboxRepositoryUrl(), getMavenRepositoryBaseUrl());
        }
        else
        {
            throw new MojoExecutionException("Sandbox '" + getSandboxKey() + "' is already promoting");
        }

        getLog().info("Sandbox is promoting...");
        while (sandboxService.isPromoting(getSandboxKey()))
        {
            try
            {
                Thread.sleep(TimeUnit.SECONDS.toMillis(5));
            }
            catch (InterruptedException e)
            {
                throw new MojoExecutionException(e.getMessage(), e);
            }
        }

        promotionTasksForSandboxRelease();

        getLog().info("Sandbox has finished promoting");
    }

    private void assertValidWorkspace() throws MojoFailureException, ScmException
    {
        scmHandler.assertValidWorkspace(developerConnection.getUrl(), getExpectedBranch(), getProperty(SandboxMetadataProperties.WORKSPACE));
    }

    private void promotionTasksForSandboxRelease() throws MojoFailureException
    {
        if (!skipTagging)
        {
            final Map<String, String> releases = getReleasesToSandbox();
            final boolean isSandboxRelease = !releases.isEmpty();

            if (isSandboxRelease)
            {
                getLog().info("Updating tags for " + getSandboxKey());

                copySandboxTagsToPromotedTags(releases);
            }
        }
        else
        {
            getLog().info("Skipped tag update");
        }
    }

    private void copySandboxTagsToPromotedTags(final Map<String, String> sandboxReleases) throws MojoFailureException
    {
        try
        {
            for (final Map.Entry<String, String> releaseTags : sandboxReleases.entrySet())
            {
                copySandboxTagToPromotedTag(releaseTags.getKey(), releaseTags.getValue());
            }
        }
        catch (ScmException e)
        {
            throw new MojoFailureException(e.getMessage(), e);
        }
    }

    private void copySandboxTagToPromotedTag(final String sandboxTag, final String releaseTag) throws MojoFailureException, ScmException
    {
        final String commitMessage = "\"Promoted '" + getSandboxKey() + "' from the Sandbox\"";

        scmHandler.createRemoteTagAndUpdateToIt(developerConnection.getUrl(), sandboxTag, releaseTag, commitMessage);

        getLog().info("Release has been tagged at " + releaseTag);
    }

    private boolean isAlreadyPromoted() throws MojoFailureException
    {
        return validation && getProperties().containsKey(SandboxMetadataProperties.PROMOTED_STATUS);
    }

    private void markAsPromoted() throws MojoFailureException
    {
        setProperty(SandboxMetadataProperties.PROMOTED_STATUS, "true");
    }

    // -------------------------------------------------------------------------------------- Basic Accessors / Mutators

    public String getExpectedBranch() throws MojoFailureException
    {
        return getProperty(SandboxMetadataProperties.BRANCH);
    }
}
