package com.atlassian.maven.plugins

import org.apache.commons.collections.CollectionUtils

/**
 * @requiresDependencyResolution test
 * @goal validate
 * @phase validate
 */
class ValidateDependenciesMojo extends AbstractDependencyTrackerMojo
{
    /**
     *
     * @parameter expression="${dependency-tracker.skip}" default-value="false"
     * @required
     * @readonly
     */
    boolean skip = false

    /**
     *
     * @parameter expression="${dependency-tracker.validateArtifactChecksums}" default-value="true"
     * @required
     */
    boolean validateArtifactChecksums

    /**
     *
     * @parameter expression="${dependency-tracker.validatePomChecksums}" default-value="true"
     * @required
     */
    boolean validatePomChecksums

    void execute()
    {
        if (skip)
        {
            log.info "Skipping dependency report validation"
            return
        }

        final File report = new File("${project.basedir}/${reportFileName}")
        if (!report?.exists())
        {
            log.warn("$report.absolutePath doesn't exists, won't validate dependencies")
            return
        }

        def fileReport = report.readLines().collect { DependencyReportLine.from(it) }
        def runtimeReport = reportLines()

        // used to build a map of (compare string, display string)
        def mapByCompareString = { Map memo, DependencyReportLine entry ->
          memo[entry.toString(validateArtifactChecksums, validatePomChecksums)] = entry.toString()
          return memo
        }

        def fileReportMap = fileReport.inject([:], mapByCompareString)
        def runtimeReportMap = runtimeReport.inject([:], mapByCompareString)

        def newDependencies = CollectionUtils.subtract(runtimeReportMap.keySet(), fileReportMap.keySet())
        def removedDependencies = CollectionUtils.subtract(fileReportMap.keySet(), runtimeReportMap.keySet())

        if (!newDependencies && !removedDependencies)
        {
            log.debug "Calculated dependencies are the same as in the report"
            return
        }

        if (newDependencies)
        {
            log.error "The following dependencies are not present in the report:"
            newDependencies.each { log.error "\t${runtimeReportMap[it]}" }
        }

        if (removedDependencies)
        {
            log.error "The following dependencies are present in the report but no longer calculated:"
            removedDependencies.each { log.error "\t${fileReportMap[it]}" }
        }

        String fileName = "${reportFileName}.tmp"
        writeReportToFile(fileName, runtimeReport);

        fail "The dependency report and your resolved dependencies are not in sync. An updated report has been written to <${new File(fileName).absolutePath}>"
    }
}
