package com.atlassian.maven.plugins

import org.apache.maven.artifact.Artifact
import org.apache.maven.project.MavenProject
import org.codehaus.groovy.maven.mojo.GroovyMojo

abstract class AbstractDependencyTrackerMojo extends GroovyMojo
{
    /**
     * The Maven Project
     *
     * @parameter expression="${project}"
     * @readonly
     */
    MavenProject project

    /**
     * The file name to use for reporting
     * @parameter expression="${dependency-tracker.report}" default-value="dependencies.txt"
     * @required
     * @readonly
     */
    String reportFileName = 'dependencies.txt'

    /**
     * The name of the checksum algorithm to be used. Typically md5, sha1, etc.
     * @parameter expression="${dependency-tracker.checksum}" default-value="sha1"
     * @required
     * @readonly
     */
    String checkSum = 'sha1'

    SortedSet<DependencyReportLine> reportLines()
    {
        final SortedSet<DependencyReportLine> report = new TreeSet()
        project.artifacts.each {Artifact a -> report.add(DependencyReportLine.from(a, this.&calculateCheckSum)) }
        return report
    }

    def calculateCheckSum = {File file ->
        if (!file?.exists())
        {
            log.info "File for $file doesn't exists"
            return '?'
        }
        def checkSumName = "${file.absolutePath}.checksum"
        ant.checksum(file: file.absolutePath, property: checkSumName, algorithm: checkSum)
        "{$checkSum}${ant.project.properties[checkSumName]}"
    }

    void writeReportToFile(String fileName, Set<DependencyReportLine> report)
    {
        final def file = "${project.basedir}/${fileName}"
        new File(file).withWriter {writer -> report.each { writer << "$it\n" } }
        log.debug "Wrote dependencies report to $file"
    }
}
