package com.atlassian.maven.plugins.sourcerelease.mojos.git;

import com.atlassian.maven.plugins.sourcerelease.mojos.CheckoutFromTagFilter;
import com.atlassian.maven.plugins.sourcerelease.mojos.git.ProcessExecution.ExecutionResult;
import com.atlassian.maven.plugins.sourcerelease.mojos.git.ProcessExecution.FileStreamConverter;
import com.atlassian.maven.plugins.sourcerelease.mojos.git.ProcessExecution.ListStreamConverter;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;
import org.apache.maven.project.MavenProject;
import org.apache.maven.scm.ScmException;
import org.apache.maven.scm.command.export.ExportScmResult;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

public class GitScmProviderUtils
{
    static RepositoryAndPath splitUrlIntoRepositoryAndPath(String url)
    {
        int fragment = url.indexOf('#');
        if (fragment >= 0)
        {
            return new RepositoryAndPath(url.substring(0, fragment), url.substring(fragment + 1));
        }

        int dotGitPath = url.indexOf(".git/");
        if (dotGitPath >= 0)
        {
            return new RepositoryAndPath(url.substring(0, dotGitPath + 4), url.substring(dotGitPath + 5));
        }

        return new RepositoryAndPath(url, "");
    }

    static String stripLeadingAndEnsureTrailingSlashes(String p)
    {
        if (p.startsWith("/"))
        {
            p = p.substring(1);
        }

        if (!p.equals("") && !p.endsWith("/"))
        {
            p += "/";
        }

        return p;
    }

    /**
     * Export a specific tag from a repository to a local tar file.
     */
    static String makeLocalCopy(RepositoryAndRef rt, File tarfile, File temporaryDirectory) throws IOException, InterruptedException
    {
        String repositoryUrl = rt.getRepositoryUrl();
        ExecutionResult<File> result;
        final File repositoryDirectory;
        List<String> archiveCmdArgs = new ArrayList<String>();
        archiveCmdArgs.add("git");
        archiveCmdArgs.add("archive");
        archiveCmdArgs.add("--format=tar");
        archiveCmdArgs.add(rt.getRefAsString());

        if (repositoryUrl.startsWith("http"))
        {
            // we cant archive remotely, so we need to clone it, and archive that instead
            repositoryDirectory = new File(temporaryDirectory, rt.getExportFilename() + ".clone");
            if(!(repositoryDirectory.mkdir()))
            {
                throw new IOException("Could not create temp directory: " + repositoryDirectory.getAbsolutePath());
            }

            //clone needs just the branch name, not 'refs/heads/branchname'
            String[] refComponents = rt.getRefAsString().split("/");

            final String[] cloneCmdArgs = {
                    "git", "clone", "--depth=1", "-b", refComponents[refComponents.length-1], repositoryUrl, repositoryDirectory.getAbsolutePath()
            };
            ExecutionResult<List<String>> cloneResult = new ProcessExecution().executeCommand(cloneCmdArgs, new ListStreamConverter());

            if (cloneResult.returnCodes() != 0)
            {
                throw new RuntimeException("git process failed: " + StringUtils.join(cloneResult.getErrors(), '\n'));
            }
        }
        else
        {
            repositoryDirectory = null; // we dont need a directory - its remote
            archiveCmdArgs.add("--remote=" + repositoryUrl);
        }
        result = new ProcessExecution().executeCommand(archiveCmdArgs, new FileStreamConverter(tarfile.getParentFile(), tarfile.getName() + ".tmp"), repositoryDirectory);

        if (result.returnCodes() != 0)
        {
            throw new RuntimeException("git process failed: " + StringUtils.join(result.getErrors(), '\n'));
        }
        final File tf = result.getOutput();
        if (!tf.renameTo(tarfile))
        {
            throw new RuntimeException("Failed to move temporary file to " + tarfile);
        }
        return StringUtils.join(result.getErrors(), '\n');
    }

    static int slashCount(String path)
    {
        int slashes = 0;

        int i = -1;

        while ((i = path.indexOf('/', i + 1)) >= 0)
        {
            slashes++;
        }

        return slashes;
    }

    /**
     * Extract a subdirectory from a tar file. If the path is the empty string it will be
     * treated as the root of the archive. Otherwise, it will be the name of a subdirectory.
     * The contents will be extracted to the directory 'target', which will be created.
     */
    public static ExportScmResult extractFromTarFile(File tarfile, String path, File target)
            throws IOException, InterruptedException
    {
        /*
         * We want to end up with either the empty string (for the root) or a directory
         * name with a slash on the end.
         */
        if (!"".equals(path))
        {
            path = stripLeadingAndEnsureTrailingSlashes(path);
        }

        int slashes = slashCount(path);

        if (!target.mkdirs())
        {
            throw new IOException("Failed to create directory: " + target);
        }

        List<String> cmd =  new ArrayList<String>();
        cmd.add("tar");
        cmd.add("xvf");
        cmd.add(tarfile.getPath());
        cmd.add("-C");
        cmd.add(target.getPath());
        cmd.add("--strip-components");
        cmd.add(Integer.toString(slashes));

        if (!"".equals(path))
        {
            cmd.add(path); // tar on Mac OS X doesn't like an empty '' argument here
        }
        final ExecutionResult<List<String>> result = new ProcessExecution().executeCommand(cmd, new ListStreamConverter());
        if (result.returnCodes() != 0)
        {
            throw new RuntimeException("Untarring failed: " + StringUtils.join(result.getErrors(), '\n'));
        }
        return new ExportScmResult(StringUtils.join(cmd, ' '), result.getOutput());
    }

    public static String refForProject(MavenProject p, String productBranchName, CheckoutFromTagFilter filter)
    {
        final String version = p.getVersion();
        Validate.notEmpty(version, "Invalid project definition. <version/> is missing.");

        // filter each artifact to determine if it should be exported from the branch or from a tag with the same name
        if (filter != null && filter.checkoutBranch(p))
        {
            Validate.notEmpty(productBranchName, String.format("Unable to determine the branch for %s as 'productBranchName' is empty", version));
            return Ref.head(productBranchName);
        }

        return Ref.tag(tagForProject(p));
    }

    public static String tagForProject(MavenProject p)
    {
        return p.getArtifactId() + "-" + p.getVersion();
    }

    static String repoIfGit(String mavenScmUrl)
    {
        if (mavenScmUrl.startsWith("scm:git:"))
        {
            String url = mavenScmUrl.substring(4);
            return splitUrlIntoRepositoryAndPath(url).getRepositoryUrl();
        }
        else
        {
            return null;
        }
    }

    /**
     * Find the most senior parent project with an SCM connection that's in the same Git
     * repository as this one.
     */
    public static MavenProject findRootProjectInSameRepository(MavenProject subproject)
    {
        MavenProject p = subproject;

        MavenProject parent = p.getParent();

        while (parent != null)
        {
            String repo = repoIfGit(p.getScm().getConnection());
            String parentRepo = repoIfGit(parent.getScm().getConnection());

            if (repo != null && repo.equals(parentRepo))
            {
                p = parent;
            }
            else
            {
                return p;
            }

            parent = p.getParent();
        }

        return p;
    }

    public static ExportScmResult export(File temporaryDirectory, String providerRepositoryUrl,
            String ref, File targetDirectory)
        throws ScmException
    {
        RepositoryAndPath repoAndSubdir = splitUrlIntoRepositoryAndPath(providerRepositoryUrl);
        return export(temporaryDirectory, repoAndSubdir, ref, targetDirectory);
    }

    public static ExportScmResult export(File temporaryDirectory, RepositoryAndPath repoAndSubdir,
                                         String ref, File targetDirectory) throws ScmException
    {
        RepositoryAndRef repoAndTag = new RepositoryAndRef(repoAndSubdir.getRepositoryUrl(), ref);

        File tarfile = new File(temporaryDirectory, repoAndTag.getExportFilename() + ".tar");

        try
        {
            if (!tarfile.isFile())
            {
                makeLocalCopy(repoAndTag, tarfile, temporaryDirectory);
            }

            return extractFromTarFile(tarfile, repoAndSubdir.getPath(), targetDirectory);
        }
        catch (IOException e)
        {
            throw new ScmException("Failed to export from Git", e);
        }
        catch (InterruptedException e)
        {
            Thread.currentThread().interrupt();
            throw new ScmException("Failed to export from Git", e);
        }
    }
}
