package com.atlassian.maven.plugins.sourcerelease.util;

import org.apache.maven.plugin.logging.Log;

import java.util.concurrent.Callable;


/**
 * Helper class that allows you to easily run a task that is likely to fail multiple times. It will attempt to rerun the
 * task using a backing off algorithm or constant time. By default backing off is used with 1000ms initial delay and 10
 * max retries. Delays are capped at 60s per run. The last attempt will be at least 4 minutes after the first call.
 */
public class RetryingTaskExecutor<T>
{
    private final Log log;
    // ------------------------------------------------------------------------------------------------------- Constants
    public static final long DEFAULT_MAX_RETRY_DELAY = 20000L;
    public static final int DEFAULT_MAX_RETRIES = 5;
    public static final int DEFAULT_INITIAL_RETRY_DELAY = 1000;
    public static final int DEFAULT_BACK_OFF_MULTIPLIER = 2;

    // ------------------------------------------------------------------------------------------------- Type Properties
    private final long initialRetryDelay;
    private final long maxRetryDelay;
    private final int maxRetries;
    private final boolean useExponentialBackOff;
    private final long backOffMultiplier;
    private final boolean rethrowLastException;

    protected volatile T objectToReturn;
    private int failures;

    // ---------------------------------------------------------------------------------------------------- Dependencies
    // ---------------------------------------------------------------------------------------------------- Constructors

    public RetryingTaskExecutor(long initialRetryDelay, long maxRetryDelay, int maxRetries, long backOffMultiplier, boolean useExponentialBackOff, boolean rethrowLastException, final Log log)
    {
        this.initialRetryDelay = initialRetryDelay;
        this.maxRetryDelay = maxRetryDelay;
        this.maxRetries = maxRetries;
        this.useExponentialBackOff = useExponentialBackOff;
        this.backOffMultiplier = backOffMultiplier;
        this.rethrowLastException = rethrowLastException;
        this.log = log;
    }

    public RetryingTaskExecutor(Log log, final int maxRetries)
    {
        this(DEFAULT_INITIAL_RETRY_DELAY, DEFAULT_MAX_RETRY_DELAY, maxRetries, DEFAULT_BACK_OFF_MULTIPLIER, true, true, log);
    }

    // ----------------------------------------------------------------------------------------------- Interface Methods
    // -------------------------------------------------------------------------------------------------- Public Methods

    public void runTask(Callable<T> task) throws Exception
    {
        runTask(task.getClass().getName(), task);
    }

    public void runTask(final String taskName, Callable<T> task) throws Exception
    {
        boolean run = true;
        long retryDelay = initialRetryDelay;
        while (run)
        {
            run = rerun(task, taskName, retryDelay);
            if (useExponentialBackOff)
            {
                retryDelay *= backOffMultiplier;
                if (retryDelay > maxRetryDelay)
                {
                    retryDelay = maxRetryDelay;
                }
            }
            else
            {
                retryDelay = initialRetryDelay;
            }
        }
    }

    private boolean rerun(Callable<T> task, String taskName, long retryDelay) throws Exception
    {

        try
        {
            objectToReturn = task.call();
            if (failures > 0)
            {
                log.info("Task '" + taskName + "' finished successfully during retry number " + failures);
            }
            return false;
        }
        catch (Exception e)
        {
            failures++;
            log.info("Task '" + taskName + "' was unsuccessful. Run " + failures + " / " + maxRetries + ". Exception thrown when running task '" + taskName + "', with message: " + e.getMessage());
            log.debug("Full stack trace: ", e);

            if (failures >= maxRetries)
            {
                log.error("Failed to run task '" + taskName + "' after " + failures + " attempts. Task was not executed.");
                if (rethrowLastException)
                {
                    log.error("Exception being rethrown");
                    throw e;
                }
                else
                {
                    log.error("Final exception was ", e);
                }

                return false;
            }


            log.info("Waiting " + retryDelay + "ms before retrying...");
            try
            {
                Thread.sleep(retryDelay);
                return true;
            }
            catch (InterruptedException inter)
            {
                log.warn("Retry was interrupted! Task was not run.");
                Thread.currentThread().interrupt();
                return false;
            }
        }


    }

    public T getObjectToReturn()
    {
        return objectToReturn;
    }

    // -------------------------------------------------------------------------------------- Basic Accessors / Mutators
}
