package com.atlassian.maven.plugins.sourcerelease.mojos;

import com.atlassian.maven.plugins.sourcerelease.configurations.BuildConfiguration;
import com.atlassian.maven.plugins.sourcerelease.configurations.ReleaseArtifactMapping;
import com.atlassian.maven.plugins.sourcerelease.configurations.RepositoryMapping;
import com.atlassian.maven.plugins.sourcerelease.configurations.SourceDependency;
import com.atlassian.maven.plugins.sourcerelease.util.FinalMavenBuildCommand;
import com.atlassian.maven.plugins.sourcerelease.util.MavenVersion;
import com.atlassian.maven.plugins.sourcerelease.util.SCMPathType;
import com.atlassian.maven.plugins.sourcerelease.util.filters.OrArtifactFilter;

import org.apache.commons.lang.StringUtils;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.DefaultArtifact;
import org.apache.maven.artifact.factory.ArtifactFactory;
import org.apache.maven.artifact.metadata.ArtifactMetadataSource;
import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.artifact.resolver.ArtifactNotFoundException;
import org.apache.maven.artifact.resolver.ArtifactResolutionException;
import org.apache.maven.artifact.resolver.ArtifactResolutionResult;
import org.apache.maven.artifact.resolver.ArtifactResolver;
import org.apache.maven.artifact.resolver.filter.ArtifactFilter;
import org.apache.maven.artifact.resolver.filter.ScopeArtifactFilter;
import org.apache.maven.model.Scm;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.InvalidProjectModelException;
import org.apache.maven.project.MavenProject;
import org.apache.maven.project.MavenProjectBuilder;
import org.apache.maven.project.ProjectBuildingException;
import org.apache.maven.project.artifact.InvalidDependencyVersionException;
import org.apache.maven.settings.Settings;

import java.io.File;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

/**
 * Does most of the heavy lifting for resolving what projects will be checked out
 *
 * @since v1.4
 */
public abstract class AbstractSourceDistributionMojo extends AbstractMojo
{
    /**
     * Location of the file.
     * @parameter expression="${project.build.directory}"
     * @required
     */
    protected File outputDirectory;

    /**
     * Upload a POM for this artifact.  Will generate a default POM if none is
     * supplied with the pomFile argument.
     *
     * @parameter expression="${sourcedistribution.generatePom}" default-value="false"
     */
    protected boolean generatePom;

    /**
     * Generate a build script that builds each dependent project seperately.
     * @parameter default-value="true"
     */
    protected boolean buildScript;

    /**
     * Allows renaming of the outputted build script prefix
     * @parameter default-value="build"
     */
    protected String buildScriptNamePrefix;

    /**
     * Generate a build script that builds each dependent project seperately, and runs all tests
     * @parameter default-value="false"
     */
    protected boolean buildScriptWithTests;

    /**
     * Is useLocalRepo is set, this is the directory that will be used
     * @parameter
     */
    protected String localRepoDirectory;

    /**
     * GroupId of the generated pom
     * @parameter
     * @required
     */
    protected String groupId;

    /**
     * ArtifactId of the generated source release pom
     * @parameter
     * @required
     */
    protected String artifactId;

    /**
     * Version of the generated pom
     * @parameter
     * @required
     */
    protected String version;

    /**
     * Product Name of the generated pom (eg. Atlassian Confluence)
     * @parameter
     * @required
     */
    protected String productName;

    /**
     * Directory name of the location in the output directory where the source release project will be generated
     * @parameter default-value="checkouts"
     */
    protected String checkoutDirectoryName;

    /**
     * Mask of groupId to include in source release (eg. "com.atlassian." or "org.apache.maven.plugins." )
     * @parameter
     * @required
     */
    protected String groupIdMask;

    /**
     * List of artifacts to exclude from the source release. Format "groupId:artifactId"
     * @parameter
     */
    protected List exclusions;

    /**
     * List of artifacts to exclude from the source release including all their dependencies. Format "groupId:artifactId"
     *
     * @parameter
     */
    protected List dependencyExclusions;

    /**
     * Skips the execution
     * @parameter expression="${sourcedistribution.skip}" default-value="false"
     */
    protected boolean skip;

    /**
     * Whether to resolve and check out the parent project for modules of a multi-module projects
     * @parameter expression="${sourcedistribution.resolveRoot}" default-value="false"
     */
    protected boolean resolveRoot;

    /**
     * Whether to ignore export failures or not. All failures will be batched up and logged at the end.
     * @parameter expression="${ignoreFailures}" default-value="false"
     */
    protected boolean ignoreFailures;

    /**
     * Specific source dependency configurations
     * @deprecated Use {@link RepositoryMapping} instead. Since v1.4.4.
     * @parameter expression="${sourceDepConfiguration}"
     */
    protected SourceDependency[] sourceDependencies;

    /**
     * Parameter that tells the plugin to traverse down the submodules of the base project collecting dependencies
     * @parameter default-value="fase"
     */
    protected boolean traverseBaseProjectModules;

    /**
     * List of exclusions based on scm prefix
     * @parameter
     */
    protected List scmPrefixExclusions;

    /**
     * Whether or not to mimic SVN checkouts with hg, i.e. only keep relevant
     * subfolders.
     * @parameter default-value="true"
     */
    protected boolean hgSubFolders;


    /**
     * Specific source dependency configurations
     * @parameter expression="${repositoryMappingConfiguration}"
     */
    protected RepositoryMapping[] repositoryMappings;

    /**
     * Allows overriding of artifacts tag prefixes.
     * @parameter
     */
    protected ReleaseArtifactMapping[] releaseArtifactMappings;

    /**
     * Allows custom build configuration for each dependency
     * @parameter
     */
    protected BuildConfiguration[] buildConfigurations;

    /**
     * Allows the specification of maven arguments that will be added to all maven commands
     * @parameter
     */
    protected String addToAllMaven;

    /**
     * Allows changing the settings file name to be shipped with the source distro.
     * It should include Atlassian Public and Atlassian Config repos.
     * The plugin will attempt to copy this file into the source distribution into the filename
     * specified with settingsFileName
     * @parameter
     */
    protected String includeSettingsFile;

    /**
     * The path to the settings file to be used with the build script
     * @parameter default-value="settings.xml"
     */
    protected String settingsFileName;

    /**
     * The commands that should be run after all the dependencies are built. For example this may be the maven involation
     * to actually build and package the WAR.
     * @parameter
     */
    protected FinalMavenBuildCommand[] finalMavenBuildCommands;

    /**
     * Whether to use the reactor artifacts for dependency resolution or the artifacts in the
     * source distribution pom. If false, the dependencies of the project are resolved
     * transitively and filtered to include only runtime and compile scopes.
     * @parameter expression="${useReactor}" default-value="true"
     *
     */
    protected boolean useReactor;

    /**
     * The source distribution plugin will not add the root project, i.e. the project calling the plugin,
     * into the distribution, since usually customers shouldn't get the it.
     * Set this parameter to true if that behaviour is not desired.
     * @parameter default-value="false"
     */
    protected boolean includeRootProject;

    /**
     * @parameter expression="${project}"
     * @required
     */
    protected MavenProject project;

    /**
     * The maven project representing the actual product this is being built for
     * @parameter
     * @required
     */
    protected String baseProject;

    /**
     * @component
     */
    protected ArtifactFactory artifactFactory;

    /**
     * All projects listed in the reactor
     * @parameter expression="${reactorProjects}"
     * @required
     */
    protected List<MavenProject> reactorProjects;

    /**
     * @component
     */
    protected MavenProjectBuilder projectBuilder;

    /**
     * @component
     */
    protected ArtifactResolver resolver;

    /**
     * @parameter expression="${settings}"
     * @required
     * @readonly
     */
    protected Settings settings;

    /**
     * @parameter default-value="${localRepository}"
     * @required
     * @readonly
     */
    protected ArtifactRepository localRepository;

    /**
     * @parameter default-value="${project.remoteArtifactRepositories}"
     * @required
     * @readonly
     */
    protected List remoteRepositories;

    /**
     * @component
     */
    protected ArtifactMetadataSource metadataSource;

    protected Map<MavenProject, MavenVersion> mavenVersionMap = new HashMap<MavenProject, MavenVersion>();

    /**
     * Set to true to skip time consuming checkouts for debugging purposes
     * @parameter default-value="false"
     */
    protected boolean debugSkipCheckout;

    /**
     * Resolves the set of projects which will be checked out. If {@link #useReactor} is true,
     * will resolve the initial set of projects based on those in the reactor. Otherwise, projects
     * will be resolved based on the transitive dependencies of the module invoking this plugin.
     *
     * This initial set of projects will then undergo a set of transformations if they match any
     * {@link RepositoryMapping} configurations specified.
     *
     * @return the set of maven projects that will be checked out
     * @throws MojoExecutionException
     */
    protected Set<MavenProject> getResolvedProjects() throws MojoExecutionException
    {
        Set<Artifact> dependencies = useReactor ? getReactorProjectDependencies() : getProjectDependencies();

        removeExcludedDependencies(dependencies);

        final Set<MavenProject> projects = getProjectsForArtifacts(dependencies);

        if (projects == null)
        {
            throw new MojoExecutionException("Could not resolve dependent projects");
        }

        //Remove undesirables
        if (useReactor)
        {
            removeReactorProjects(projects);
        }

        removeExcludedArtifacts(projects);

        Set<MavenProject> resolvedProjects = processProjects(projects);
        resolvedProjects = removeScmPrefixExclusions(resolvedProjects);
        resolvedProjects = removeChildProjects(resolvedProjects);

        handleSnapshots(resolvedProjects);

        if (reactorProjects == null || reactorProjects.size() < 1)
        {
            throw new MojoExecutionException("There are no dependent projects left to process.");
        }

        return resolvedProjects;
    }

    static Set<MavenProject> removeChildProjects(final Set<MavenProject> resolvedProjects)
    {
        final Map<String, MavenProject> scmToProjectMapping = new TreeMap<String, MavenProject>();
        for (final MavenProject resolvedProject : resolvedProjects)
        {
            scmToProjectMapping.put(resolvedProject.getScm().getConnection(), resolvedProject);
        }

        final Set<String> uniqueScms = new HashSet<String>();
        String currentScm = null;
        for (final String scm : scmToProjectMapping.keySet())
        {
            if(currentScm == null || !scm.startsWith(currentScm))
            {
                currentScm = scm;
                uniqueScms.add(currentScm);
            }
        }
        scmToProjectMapping.keySet().retainAll(uniqueScms);
        return new HashSet<MavenProject>(scmToProjectMapping.values());
    }

    public static <T> List<T> checkedCopy(Collection<?> c, Class<T> type)
    {
        List<T> coll = Collections.checkedList(new ArrayList<T>(), type);
        coll.addAll((Collection<T>) c);
        return coll;
    }

    private Set<MavenProject> removeScmPrefixExclusions(final Set<MavenProject> resolvedProjects)
    {
        if(scmPrefixExclusions == null)
        {
            return resolvedProjects;
        }
        final Set<MavenProject> includedProjects = new HashSet<MavenProject>(resolvedProjects);
        for(final Object scmExclusion : scmPrefixExclusions)
        {
            final Set<MavenProject> excludedProjects = new HashSet<MavenProject>();
            for(final MavenProject project : includedProjects)
            {
                if(project.getScm().getConnection().startsWith((String) scmExclusion))
                {
                    excludedProjects.add(project);
                }
            }
            includedProjects.removeAll(excludedProjects);
        }
        return includedProjects;
    }

    private Set<Artifact> getProjectDependencies() throws MojoExecutionException
    {
        final Set dependencies = new HashSet();
        try
        {
            final Set<Artifact> artifacts = project.createArtifacts(artifactFactory, null, null);
            dependencies.addAll(artifacts);
        }
        catch (InvalidDependencyVersionException e)
        {
            throw new RuntimeException(e);
        }
        final OrArtifactFilter dependencyFilter = new OrArtifactFilter();
        dependencyFilter.add(new ScopeArtifactFilter(DefaultArtifact.SCOPE_COMPILE));
        dependencyFilter.add(new ScopeArtifactFilter(DefaultArtifact.SCOPE_RUNTIME));
        if (traverseBaseProjectModules)
        {
            dependencies.addAll(getBaseProjectModules(dependencies));
        }
        return resolveTransitiveArtifacts(dependencies, dependencyFilter);

    }

    /**
     * Processes the filtered set of dependent projects to resolve parent projects in a multi module
     * set up as well as doing any scm path manipulation to fix incorrect scm information in the pom
     *
     * @param projects the projects to process
     * @return
     */
    private Set<MavenProject> processProjects(final Set<MavenProject> projects)
    {
        final Set<MavenProject> resolvedList = new HashSet<MavenProject>();
        final Set<String> seenSCMUrls = new HashSet<String>();
        for(MavenProject mavenProject : projects)
        {

            final MavenProject projectWithRepositoryMappingsApplied = findAndApplyRepositoryMappingConfig(mavenProject);
            final MavenProject projectWithSourceDepConfigApplied = findAndApplySourceDepConfig(projectWithRepositoryMappingsApplied);
            if (projectWithSourceDepConfigApplied != null)
            {
                final Scm scm = projectWithSourceDepConfigApplied.getScm();
                if (scm != null && scm.getConnection() != null)
                {
                    if (!seenSCMUrls.contains(scm.getConnection()))
                    {
                        getLog().debug("Scm is " + scm.getConnection());
                        resolvedList.add(projectWithSourceDepConfigApplied);
                        seenSCMUrls.add(scm.getConnection());
                    }
                }
                else
                {
                    getLog().info("No SCM details for " + mavenProject.getArtifact());
                }
            }
        }
        return resolvedList;
    }

    private MavenProject findAndApplyRepositoryMappingConfig(final MavenProject mavenProject)
    {
        if(repositoryMappings != null)
        {
            for (final RepositoryMapping config : repositoryMappings)
            {
                final Map<String,String> moduleMappings = config.getModuleMapping();
                final List<String> repositoryModules = config.getRepositoryModules();
                if((moduleMappings.containsKey(mavenProject.getArtifactId()) ||
                    repositoryModules.contains(mavenProject.getArtifactId())))
                {
                    return applyRepositoryMappingConfig(config, mavenProject);
                }
            }
        }
        return mavenProject;
    }

    private MavenProject applyRepositoryMappingConfig(final RepositoryMapping config, final MavenProject mavenProject)
    {
        MavenProject processedMavenProject = mavenProject;
        if(config.isResolveRoot())
        {
            processedMavenProject = resolveRootProject(mavenProject);
        }
        if(config.getBaseRepository() != null && !config.getBaseRepository().equals(""))
        {
            modifyBaseRepository(config.getBaseRepository(), processedMavenProject);
        }
        if(config.getModuleMapping() != null && !config.getModuleMapping().isEmpty())
        {
            applyModuleMapping(config.getModuleMapping(), processedMavenProject);
        }
        return processedMavenProject;

    }

    /**
     * Resolves the root, parent project in a multi module project set up. It tries to figure this out based on the
     * current SCM resolved for the project. e.g. for project <pre>foobar</pre> with resolved scm of
     * <pre>scm:svn:http://svn.atlassian.com/foo/bar/foobar/tags/foo-1.0</pre> , will determine the number of path elements
     * to foobar (foo/bar) and traverse up the hierarchy twice to return the MavenProject corresponding to <pre>foo</pre>.
     *
     * @param mavenProject the module to start upward resolution from
     * @return the parent maven project
     */
    protected MavenProject resolveRootProject(final MavenProject mavenProject)
    {
        getLog().debug("Resolving root project for project" + mavenProject.getArtifactId());


        final String connection = mavenProject.getScm().getConnection();
        final SCMPathType scmPathType = SCMPathType.determinePathType(connection);

        MavenProject parentProject = mavenProject;

        if(scmPathType.equals(SCMPathType.UNKNOWN))
        {
            getLog().warn(MessageFormat.format("Unable to find trunk or tag markers on scm path during root project resolution for {0}, original scm information was {1}",
                    mavenProject.getArtifactId(), connection));
        }
        else
        {
            final List<String> pathElements = scmPathType.determinePathElements(connection);


            for (String pathElement : pathElements)
            {
                parentProject = parentProject.getParent();
            }

        }

        if(!mavenProject.getArtifactId().equals(parentProject.getArtifactId()))
        {
            getLog().info(MessageFormat.format("Resolved root project for {0} to {1}", mavenProject.getArtifactId(),
                    parentProject.getArtifactId()));
        }
        else
        {
            getLog().info(MessageFormat.format("Resolved root project for {0}: it was itself", mavenProject.getArtifactId()));
        }

        return parentProject;
    }

    /**
     * Determines if a project has a sourceDependency configuration specified and
     * applies the configuration if so. {@link SourceDependency} should no longer be used, use
     * {@link RepositoryMapping} instead.
     *
     * @param mavenProject
     * @return
     */
    private MavenProject findAndApplySourceDepConfig(final MavenProject mavenProject)
    {
        if(sourceDependencies != null)
        {
            for (final SourceDependency config : sourceDependencies)
            {
                if((config.getArtifactId().equals(mavenProject.getArtifactId())))
                {
                    final RepositoryMapping repositoryMapping = new RepositoryMapping(config.getBaseRepository(),
                            config.getModuleMapping(), config.isResolveRoot(), new String[]{ config.getArtifactId() });
                    return applyRepositoryMappingConfig(repositoryMapping, mavenProject);
                }
            }
        }

        final MavenProject rootProject;
        if(resolveRoot)
        {
            rootProject = resolveRootProject(mavenProject);
        }
        else
        {
            rootProject = mavenProject;
        }

        return rootProject;
    }

    /**
     * Applies a module mapping for sub modules with paths different from their artifactId. When
     * maven does scm path resolution for sub modules, it implicitly assumes the sub module's scm
     * path follows its artifactId. Specify moduleMappings in a sourceDependency configuration should
     * this be the case.
     *
     * @param moduleMapping
     * @param mavenProject
     */
    protected void applyModuleMapping(final Map<String, String> moduleMapping, final MavenProject mavenProject)
    {
        final String connection = mavenProject.getScm().getConnection();
        String newConnection = connection;
        for(Map.Entry<String, String> mapping : moduleMapping.entrySet())
        {
            //newConnection = newConnection.replace("/"+ mapping.getKey(), "/" + mapping.getValue());
            try
            {
                StringBuilder b = new StringBuilder(newConnection);
                String key = "/"+ mapping.getKey();
                String value = "/" + mapping.getValue();
                b.replace(newConnection.lastIndexOf(key), newConnection.lastIndexOf(key) + key.length(), value );
                newConnection = b.toString();
            }
            catch(StringIndexOutOfBoundsException e)
            {
                getLog().warn("Bad substitution: could not find string " + "/"+ mapping.getKey() + " in connection url: " + newConnection);
            }
        }
        getLog().info(MessageFormat.format("Module mapping found for {0}, replacing {1} with {2}",
                mavenProject.getArtifactId(), connection, newConnection));
        mavenProject.getScm().setConnection(newConnection);
    }

    /**
     * find if there is a build config for a particular artifact. If not, return a default maven 2 config
     * @param artifactId
     * @return
     */
    protected BuildConfiguration findBuildConfiguration(final String groupId, final String artifactId)
    {
        BuildConfiguration defaultBuildConfiguration = null;
        if (buildConfigurations != null  && buildConfigurations.length > 0)
        {
            for (BuildConfiguration buildConfiguration : buildConfigurations)
            {
                if (buildConfiguration.getArtifactId().equals(artifactId) && buildConfiguration.getGroupId().equals(groupId))
                {
                    defaultBuildConfiguration = buildConfiguration;
                }
            }
        }
        return defaultBuildConfiguration;
    }

    /**
     * Modifies the base repository from which scm information is calculated relative to.
     *
     * @param baseRepository
     * @param mavenProject
     */
    private void modifyBaseRepository(final String baseRepository, final MavenProject mavenProject)
    {
        final String connection = mavenProject.getScm().getConnection();

        final SCMPathType scmPathType = SCMPathType.determinePathType(connection);

        if(scmPathType.equals(SCMPathType.UNKNOWN))
        {
            getLog().warn(MessageFormat.format("Unable to find trunk or tag markers on scm path during baseRepository replacement for {0}, using project as is",
                    mavenProject.getArtifactId()));
        }
        else
        {
            final String pathToSubmodule = scmPathType.getPathToSubModule(connection);
            mavenProject.getScm().setConnection(baseRepository + scmPathType.getMarker().substring(1) + "/" + pathToSubmodule);
            getLog().info(MessageFormat.format("Modified base reposiory for {0} from {1} into {2}", mavenProject.getArtifactId(), baseRepository, baseRepository + scmPathType.getMarker().substring(1) + "/" + pathToSubmodule));
        }
    }

    protected Set<MavenProject> getModuleProjects(MavenProject parentProject)
    {
        return new HashSet(parentProject.getCollectedProjects());
    }

    protected Set<Artifact> getBaseProjectModules(Set<Artifact> artifacts)
            throws MojoExecutionException
    {
        String baseGroup = baseProject.split(":")[0];
        String baseArtifact = baseProject.split(":")[1];
        Set modules = new HashSet();
        Set<String> excludedClassifiers = new HashSet<String>();
        excludedClassifiers.add("bundle");
        excludedClassifiers.add("atlassian-plugin");
        for (Artifact artifact : artifacts)
        {
            if (artifact.getArtifactId().equals(baseArtifact) && artifact.getGroupId().equals(baseGroup))
            {
                MavenProject parentProject = getProjectForArtifact(artifact);
                for (MavenProject mavenProject : getModuleProjects(parentProject))
                {
                    if(!excludedClassifiers.contains(mavenProject.getArtifact().getType()))
                    {
                        modules.add(mavenProject.getArtifact());
                    }
                }
            }
        }

        return modules;
    }

    private Set<Artifact> getReactorProjectDependencies() throws MojoExecutionException
    {
        Set<Artifact> dependencies = new HashSet<Artifact>();
        for (MavenProject mavenProject : reactorProjects)
        {
            Set<?> projectDependencies = mavenProject.getDependencyArtifacts();
            if (projectDependencies != null && !projectDependencies.isEmpty())
            {
                dependencies.addAll(checkedCopy(projectDependencies, Artifact.class));
            }
        }
        if (traverseBaseProjectModules)
        {
            dependencies.addAll(getBaseProjectModules(dependencies));
        }
        return resolveTransitiveArtifacts(dependencies);
    }

    private Set<MavenProject> getProjectsForArtifacts(Set<Artifact> artifacts) throws MojoExecutionException
    {
        Set<MavenProject> projects = new HashSet<MavenProject>();
        for (Artifact artifact : artifacts)
        {
            MavenProject currentProject = getProjectForArtifact(artifact);
            if (artifact.getGroupId().startsWith(groupIdMask))
            {
                projects.add(currentProject);
            }
        }
        return projects;
    }

    /* Checks for SNAPSHOT dependecies among the project list and fails fast */
    private void handleSnapshots(Set<MavenProject> projects)
            throws MojoExecutionException
    {
        List<MavenProject> projectList = new ArrayList<MavenProject>();
        for (MavenProject currentProject : projects)
        {
            if (currentProject.getVersion().contains("SNAPSHOT") && !currentProject.getGroupId().startsWith(baseProject.split(":")[0]))
            {
                projectList.add(currentProject);
            }
        }

        if(!projectList.isEmpty())
        {
            getLog().error("Found SNAPSHOT dependencies of different groupId than base project (listed below), not continuing build");
            StringBuffer message = new StringBuffer("");
            for (MavenProject snapshotProject : projectList)
            {
                message.append("\t* Dependency " + snapshotProject.getGroupId() + ":" + snapshotProject.getArtifactId() + ":" + snapshotProject.getVersion() + " is a SNAPSHOT\n");
            }
            getLog().error(message);

            throw new MojoExecutionException("Found SNAPSHOT dependencies, could not build source distribution. Please change SNAPSHOT dependencies to stable versions");
        }
    }

    private void removeExcludedArtifacts(Set<MavenProject> projects)
    	throws MojoExecutionException
    {
        List<String> exclusionPairs = getExclusions();
        List<MavenProject> projectList = new ArrayList<MavenProject>(projects);
        for (MavenProject currentProject : projectList)
        {
            if (currentProject == null)
            {
                getLog().info("Project in project list is null");
                continue;
            }

            if (matchesExclusion(currentProject, exclusionPairs))
            {
                projects.remove(currentProject);
            }
        }
    }

    private void removeExcludedDependencies(Set<Artifact> artifacts)
            throws MojoExecutionException
    {
        Set<String> exclusions = new HashSet<String>(getDependencyExclusions());

        List<Artifact> artifactList = new ArrayList(artifacts);
        for (Artifact currentArtifact : artifactList)
        {
            if (currentArtifact == null)
            {
                getLog().info("Project in project list is null");
                continue;
            }

            final String dependencyId = currentArtifact.getGroupId() + ":" + currentArtifact.getArtifactId();
            if (exclusions.contains(dependencyId))
            {
                getLog().info("Excluding " + dependencyId);
                artifacts.remove(currentArtifact);
            }
        }
    }

    private void removeReactorProjects(Set<MavenProject> projects)
    {
        for (MavenProject project : reactorProjects.subList(1, reactorProjects.size()))
        {
            if (projects.contains(project))
            {
                if (!projects.remove(project))
                {
                    getLog().warn("Could not remove reactor project from checkout list for '" + project.getGroupId() + ":" + project.getArtifactId() + "'");
                }
            }
        }
    }

    private List<String> getExclusions()
    {
        List<String> excludes = new ArrayList<String>();
        if (exclusions != null)
        {
            excludes.addAll(exclusions);
        }
        List<MavenProject> reactorExclusions = reactorProjects.subList(1, reactorProjects.size());
        for (MavenProject mavenProject : reactorExclusions)
        {
            excludes.add(mavenProject.getGroupId() + ":" + mavenProject.getArtifactId());
        }
        return excludes;
    }

    private List<String> getDependencyExclusions()
    {
        if (dependencyExclusions == null)
        {
            dependencyExclusions = new ArrayList<String>();
        }
        return dependencyExclusions;
    }

    private boolean matchesExclusion(MavenProject project, List<String> exclusionPairs)
        throws MojoExecutionException
    {
        for (String exclusion : exclusionPairs)
        {
            String[] parts = StringUtils.split(exclusion, ':');
            if (parts.length == 2)
            {
                String currentGroupId = project.getGroupId();
                String currentArtifactId = project.getArtifactId();

                if (parts[0].equals(currentGroupId) && parts[1].equals(currentArtifactId))
                {
                    return true;
                }
            }
            else
            {
                throw new MojoExecutionException("Exclusion is not in the format of groupId:artifactId. Value:" + exclusion);
            }
        }
    	return false;
    }

    /*
     * Resolves transitive dependencies for a set of given artifacts
     * Returns a Set of all artifacts resolved
     */
    private Set<Artifact> resolveTransitiveArtifacts(Set<Artifact> artifacts, ArtifactFilter filter)
    	throws MojoExecutionException
    {
        try
        {
            ArtifactResolutionResult result = resolver.resolveTransitively(artifacts, project.getArtifact(),
                localRepository, remoteRepositories, metadataSource, filter);
            return new HashSet<Artifact>(checkedCopy(result.getArtifacts(), Artifact.class));
        }
        catch (ArtifactResolutionException e)
        {
            throw new MojoExecutionException(e.getMessage(), e);
        }
        catch (ArtifactNotFoundException e)
        {
            throw new MojoExecutionException(e.getMessage(), e);
        }
    }

    private Set<Artifact> resolveTransitiveArtifacts(Set<Artifact> artifacts) throws MojoExecutionException
    {
        return resolveTransitiveArtifacts(artifacts, null);
    }

    /*
     * Returns the MavenProject for a given Artifact
     */
    private MavenProject getProjectForArtifact(Artifact artifact)
    	throws MojoExecutionException
    {
        try
        {
            return projectBuilder.buildFromRepository(artifact, remoteRepositories, localRepository);
        }
        catch (InvalidProjectModelException e)
        {
            if (e.getValidationResult() != null)
            {
                getLog().error("Validation Errors: " + e.getValidationResult().getMessages());
            }
            throw new MojoExecutionException(e.getMessage(), e);
        }
        catch (ProjectBuildingException e)
        {
            throw new MojoExecutionException(e.getMessage(), e);
        }
    }

    /**
     * @param project the maven project to look up in the mapping.
     * @return the release artifact mapping if one is found otherwise null
     */
    protected ReleaseArtifactMapping determineReleaseArtifactMappingForProject(MavenProject project)
    {
        if (releaseArtifactMappings != null)
        {
            for (ReleaseArtifactMapping mapping : releaseArtifactMappings)
            {
                if (mapping.getGroupId().equals(project.getGroupId()) &&
                        mapping.getArtifactId().equals((project.getArtifactId())))
                {
                    return mapping;
                }
            }
        }

        return null;
    }
}
