package com.atlassian.maven.plugins.sourcerelease.mojos;

//Portions Copyright Apache Software Foundation

import com.atlassian.maven.plugins.scm.ScmInfo;
import com.atlassian.maven.plugins.sourcerelease.configurations.BuildConfiguration;
import com.atlassian.maven.plugins.sourcerelease.configurations.ReleaseArtifactMapping;
import com.atlassian.maven.plugins.sourcerelease.mojos.git.GitScmProviderUtils;
import com.atlassian.maven.plugins.sourcerelease.mojos.git.Ref;
import com.atlassian.maven.plugins.sourcerelease.mojos.hg.HgScmProviderUtils;
import com.atlassian.maven.plugins.sourcerelease.util.*;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.io.IOUtils;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.factory.ArtifactFactory;
import org.apache.maven.artifact.resolver.ArtifactNotFoundException;
import org.apache.maven.artifact.resolver.ArtifactResolutionException;
import org.apache.maven.artifact.resolver.filter.ArtifactFilter;
import org.apache.maven.execution.MavenSession;
import org.apache.maven.model.Model;
import org.apache.maven.model.Scm;
import org.apache.maven.model.io.xpp3.MavenXpp3Writer;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.DuplicateProjectException;
import org.apache.maven.project.MavenProject;
import org.apache.maven.project.ProjectSorter;
import org.apache.maven.scm.NoSuchCommandScmException;
import org.apache.maven.scm.ScmException;
import org.apache.maven.scm.ScmFileSet;
import org.apache.maven.scm.ScmResult;
import org.apache.maven.scm.ScmTag;
import org.apache.maven.scm.ScmVersion;
import org.apache.maven.scm.command.export.ExportScmResult;
import org.apache.maven.scm.manager.ScmManager;
import org.apache.maven.scm.provider.ScmProviderRepositoryWithHost;
import org.apache.maven.scm.provider.git.repository.GitScmProviderRepository;
import org.apache.maven.scm.provider.hg.HgScmProvider;
import org.apache.maven.scm.provider.hg.repository.HgScmProviderRepository;
import org.apache.maven.scm.repository.ScmRepository;
import org.apache.maven.scm.repository.ScmRepositoryException;
import org.apache.maven.settings.Server;
import org.apache.maven.shared.artifact.filter.StrictPatternIncludesArtifactFilter;
import org.codehaus.plexus.PlexusContainer;
import org.codehaus.plexus.component.repository.exception.ComponentLookupException;
import org.codehaus.plexus.util.FileUtils;
import org.codehaus.plexus.util.dag.CycleDetectedException;
import org.sonatype.plexus.components.sec.dispatcher.SecDispatcher;
import org.sonatype.plexus.components.sec.dispatcher.SecDispatcherException;

import java.io.*;
import java.lang.reflect.MalformedParameterizedTypeException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.zip.GZIPInputStream;

/**
 * Goal exports dependencies from source control for a given groupId mask.
 *
 * @goal source
 * @phase package
 */
@SuppressWarnings({"ThrowableResultOfMethodCallIgnored"})
public class SourceMojo
    extends AbstractSourceDistributionMojo
{
    public static final String SNAPSHOT_FILTER = ":::*-SNAPSHOT";

    /**
     * @component
     */
    private ScmManager manager;

    /**
     * @parameter expression="${sourcedistribution.delay}" default-value="0"
     */
    private int delay;

    /**
     * @parameter default-value="0"
     */
    private int retries;

    /**
     * The Maven Session Object
     *
     * @parameter expression="${session}"
     * @required
     * @readonly
     */
    private MavenSession session;

    /**
     * @parameter expression="${project.build.directory}/source-dvcs-checkout"
     */
    private File sourceDvcsCheckoutDirectory;

    /**
     * The name of the currentBranch of the product the sources are built for.
     *
     * Can be used to make the source distribution build work for SNAPSHOT versions on the main (non-tag) branch/heads.
     *
     * @parameter
     */
    private String productBranchName;

    /**
     * A comma-separated list of artifacts for which we will <b>not</b> try to check out from a tag. By default, any
     * artifact that matches {@value #SNAPSHOT_FILTER} will not be exported from a tag, but will be exported from the
     * current branch instead.
     * <p/>
     * The artifact syntax is defined by <code>StrictPatternIncludesArtifactFilter</code>.
     *
     * @see org.apache.maven.shared.artifact.filter.StrictPatternIncludesArtifactFilter
     * @since 1.6.6
     *
     * @parameter
     */
    private List<String> checkoutHeadIncludes;

    private final Map<String, Exception> ignoredFailures = new HashMap();


    /**
     * If a sandbox key has been specified, then use a specific branch to check out of
     *
     * @parameter expression="${sandbox.key}"
     */
    private String sandboxKey;


    public void execute()
        throws MojoExecutionException
    {
        final Log log = getLog();
        if (skip)
        {
            log.info("Skipping source distribution execution");
            return;
        }

    	final File checkoutDirectory = createCheckoutDirectory();

        final Set<MavenProject> resolvedProjects = getResolvedProjects();

    	final List<MavenProject> modules = checkoutProjects(resolvedProjects, checkoutDirectory);

        MavenProject rootProject = reactorProjects.get(0);
        log.info("Checking out root project " + rootProject.getGroupId() + ":" + rootProject.getArtifactId());

        //Checkout root project last if it's not already in the module list
        if (includeRootProject && !resolvedProjects.contains(rootProject))
        {
            try
            {
                doCheckout(rootProject, checkoutDirectory, sandboxKey);
                modules.add(rootProject);
            }
            catch (Exception e)
            {
                String message = "Error: could not check out root project: " + rootProject.getGroupId() + ":" + rootProject.getArtifactId() + ":" + rootProject.getVersion();
                if(!ignoreFailures)
                {
                    log.error(message);
                    throw new MojoExecutionException(message, e);
                }
                else
                {
                    getLog().info(message + ". Ignoring due to 'ignoreFailures' flag: " + e.getMessage());
                }
            }
            if (ignoreFailures && !ignoredFailures.isEmpty())
            {
                log.warn("**** Failed to export " + ignoredFailures.size() + " dependencies, but will be ignored due to ignoreFailures flag.");
                log.warn("**** We recommend that you exclude the dependencies explicitly rather than use the ignoreFailures flag... Example follows");
                for (String s : ignoredFailures.keySet())
                {
                    log.warn("        <exclusion>" + s + "</exclusion>");
                }
                log.warn("**** Exceptions are: ");
                for (Map.Entry<String, Exception> entry : ignoredFailures.entrySet())
                {
                    log.warn(entry.getKey(), entry.getValue());
                }
            }
        }

        //automatically compute and store the correct maven version for each project
        computeMavenVersions(modules);
        String buildFilename = "build";
        if (!StringUtils.isBlank(buildScriptNamePrefix))
        {
            buildFilename = buildScriptNamePrefix;
        }

    	if (generatePom && !modules.isEmpty())
    	{
            writeSourceDistributionPom(modules, checkoutDirectory);
    	}
        
        if (buildScript && !modules.isEmpty())
        {
            writeSourceDistributionBuildScript(modules, new PosixScriptHelper(checkoutDirectory, buildFilename + ".sh", false), checkoutDirectory);
            writeSourceDistributionBuildScript(modules, new WindowsScriptHelper(checkoutDirectory, buildFilename + ".bat", false), checkoutDirectory);
        }
        
        if (buildScriptWithTests && !modules.isEmpty())
        {
            writeSourceDistributionBuildScript(modules, new PosixScriptHelper(checkoutDirectory, buildFilename + "-with-tests.sh", true), checkoutDirectory);
            writeSourceDistributionBuildScript(modules, new WindowsScriptHelper(checkoutDirectory, buildFilename +  "-with-tests.bat", true), checkoutDirectory);
        }

        //Add maven2 and maven3 to the distribution
        packageMaven(checkoutDirectory, "2.1.0", "3.0.3", "maven2", "maven3");
        
        //Add mvn2 and mvn3 alias scripts
        writeMavenAliasScripts(checkoutDirectory, "mvn2.sh", "mvn3.sh");
        writeMavenAliasScripts(checkoutDirectory, "mvn2.bat", "mvn3.bat");
    }


    /*
     * Checks out artifacts and returns a list of their artifactIds
     */
    private List<MavenProject> checkoutProjects(Set<MavenProject> projects, File checkoutDirectory)
    	throws MojoExecutionException
    {
        getLog().info("Will checkout the following artifacts:");
        for (MavenProject currentProject : projects)
        {
            getLog().info(currentProject.getGroupId() + ":" + currentProject.getArtifactId());
        }

        getLog().info("A delay of " + delay + " ms has been specified between checkouts.");
    	final List<MavenProject> modules = new ArrayList();
    	for (MavenProject currentProject : projects)
        {
            final String projectArtifactId = currentProject.getArtifactId();
            if (modules.contains(projectArtifactId))
            {
                throw new MojoExecutionException("Cannot have two modules checked out with the same directory name " + projectArtifactId);
            }
            getLog().info("----------------------");
            getLog().info("Processing " + currentProject.getGroupId() + ":" + currentProject.getArtifactId() + ":" + currentProject.getVersion());
            modules.add(currentProject);
            if (!debugSkipCheckout)
            {
                doCheckout(currentProject, checkoutDirectory);
            }
        }

    	return modules;
    }

    /*
     * Writes out the generated project pom
     */
    private void writeSourceDistributionPom(List<MavenProject> modules, File checkoutDirectory)
    	throws MojoExecutionException
    {
    	Exception ex = null;
        List<String> toBuildArtifactIds = new ArrayList<String>();
        for (MavenProject mavenProject : modules)
        {
            toBuildArtifactIds.add(mavenProject.getArtifactId());
        }

        Model model = new Model();
        model.setModelVersion( "4.0.0" );
        model.setGroupId( groupId );
        model.setArtifactId( artifactId );
        model.setVersion( version );
        model.setName( productName + " Source Release"  );
        model.setPackaging( "pom" );
        model.setDescription( "Source for " + productName );
        model.setModules(toBuildArtifactIds);

        FileWriter writer;
        try
        {
            writer = new FileWriter(new File(checkoutDirectory, "pom.xml"));
            new MavenXpp3Writer().write(writer, model);
            writer.flush();
            writer.close();
        }
        catch (IOException e)
        {
            throw new MojoExecutionException("Could not write source release pom", e);
        }
    }

    /**
     * Write out a build script that builds each project individually, using different build
     * commands and settings for each artifact
     */
    private void writeSourceDistributionBuildScript(List<MavenProject> modules, ScriptHelper scriptHelper, File checkoutDirectory)
            throws MojoExecutionException
    {
        //compute the correct build order
        List<MavenProject> toBuildProjects = resolveBuildOrder(modules);
        scriptHelper.useErrorMode();
        scriptHelper.addCurrentDirToPath();

        if(StringUtils.isNotBlank(this.settingsFileName))
        {
            scriptHelper.addSettingsLine(this.settingsFileName);
            if (StringUtils.isNotBlank(this.includeSettingsFile))
            {
                try
                {
                    scriptHelper.includeSettingsFile(this.includeSettingsFile, this.settingsFileName);
                }
                catch(IOException e)
                {
                    throw new MojoExecutionException("Error creating build script: could not copy settings file " + this.includeSettingsFile, e);
                }
            }
            else 
            {
                try
                {
                    InputStream settingsFileStream = this.getClass().getResourceAsStream("/settings.xml");

                    File settingsFile = new File(checkoutDirectory, this.settingsFileName);
                    FileWriter settingsWriter = new FileWriter(settingsFile);

                    IOUtils.copy(settingsFileStream, settingsWriter);

                    settingsFileStream.close();
                    settingsWriter.flush();
                    settingsWriter.close();
                }
                catch(IOException e)
                {
                    throw new MojoExecutionException("Couldn't write mvn alias scripts: ", e);
                }
            }

        }

        if(StringUtils.isNotBlank(localRepoDirectory))
        {
            scriptHelper.addLocalRepoLine(localRepoDirectory);
        }

        for (MavenProject module : toBuildProjects)
        {
            //default config
            boolean useMaven = true;
            String buildCmd = MavenVersion.MAVEN2.getCmd();
            String cmdArgs = "";
            boolean takesCliArgs = true;
            boolean overrideAutoMavenVersion = false;
            
            //Now check if there are custom config options for this artifact
            BuildConfiguration buildConfiguration = findBuildConfiguration(module.getGroupId(), module.getArtifactId());
            if (buildConfiguration != null)
            {
                getLog().info("Found custom build configuration for " + module.getArtifactId());
                buildCmd = buildConfiguration.getBuildCmd();
                cmdArgs = " " + buildConfiguration.getCmdArgs();
                useMaven = buildConfiguration.isMaven();
                takesCliArgs = buildConfiguration.takesCliArgs();
                overrideAutoMavenVersion = buildConfiguration.overrideMavenVersion();
            }
            if (mavenVersionMap.get(module) != null && !overrideAutoMavenVersion)
            {
                buildCmd = mavenVersionMap.get(module).getCmd();
            }
            
            if(useMaven){
                if(!StringUtils.isBlank(addToAllMaven))
                {
                    cmdArgs = " " + addToAllMaven + cmdArgs;
                }

                cmdArgs = "clean install -f \"" + module.getArtifactId() + "/pom.xml\"" + cmdArgs;
            }

            scriptHelper.addCommand(buildCmd, cmdArgs, useMaven, takesCliArgs);
        }

        //Process final build commands
        if (finalMavenBuildCommands != null && finalMavenBuildCommands.length > 0)
        {
            for(FinalMavenBuildCommand finalMavenBuildCommand : finalMavenBuildCommands)
            {
                scriptHelper.addCommand(finalMavenBuildCommand.getMavenVersion().getCmd(), finalMavenBuildCommand.getCmdArgs(), true, true);
            }
        }

        try
        {
            scriptHelper.writeScript();
        }
        catch (IOException e)
        {
            throw new MojoExecutionException("Could not write source release build script: ", e);
        }
    }

    /**
     * Sorts a list of maven projects in topograhic order, i.e. by dependency order
     */
    private List<MavenProject> resolveBuildOrder (List<MavenProject> modules)
            throws MojoExecutionException
    {
        List<MavenProject> toBuildProjects = new ArrayList<MavenProject>();
        try
        {
            ProjectSorter projectSorter = new ProjectSorter(modules);
            List results = projectSorter.getSortedProjects();
            for (Object module : results)
            {
                toBuildProjects.add((MavenProject) module);
            }
        }
        catch(CycleDetectedException e)
        {
            throw new MojoExecutionException("Error resolving build order: Found a cyclic dependency", e);
        }
        catch(DuplicateProjectException e)
        {
            throw new MojoExecutionException("Error resolving build order: Found a duplicate project", e);
        }
        return toBuildProjects;
    }

    /**
     * Downloads maven 2 and maven 3, uncompresses them and adds them to the source distribution.
     */
    private void packageMaven(File checkoutDirectory, String maven2Version, String maven3Version, String maven2DirectoryName, String maven3DirectoryName)
            throws MojoExecutionException
    {
        Artifact maven2Artifact = this.artifactFactory.createArtifactWithClassifier("org.apache.maven", "apache-maven", maven2Version, "zip", "bin");
        Artifact maven3Artifact = this.artifactFactory.createArtifactWithClassifier("org.apache.maven", "apache-maven", maven3Version, "zip", "bin");
        try
        {
            this.resolver.resolve(maven2Artifact, this.remoteRepositories, this.localRepository);
            File maven2Binary = maven2Artifact.getFile();
            File maven2Dir = new File(checkoutDirectory, maven2DirectoryName);
            maven2Dir.mkdirs();
            ZipHelper zipHelper = new ZipHelper(getLog());
            zipHelper.unzipArchive(maven2Binary, maven2Dir, true);

            this.resolver.resolve(maven3Artifact, this.remoteRepositories, this.localRepository);
            File maven3Binary = maven3Artifact.getFile();
            File maven3Dir = new File(checkoutDirectory, maven3DirectoryName);
            maven3Dir.mkdirs();
            zipHelper.unzipArchive(maven3Binary, maven3Dir, true);

        }
        catch(ArtifactResolutionException e)
        {
            throw new MojoExecutionException("Failed to resolve maven binary artifact: ", e);
        }
        catch(ArtifactNotFoundException e)
        {
            throw new MojoExecutionException("Failed to find maven binary artifact: ", e);
        }

    }

    /**
     * Writes out alias scripts mvn2 and mvn3 that invoke maven from the bundled versions that come with the source
     * distribution
     */
    private void writeMavenAliasScripts(File checkoutDirectory, String maven2script, String maven3script)
            throws MojoExecutionException
    {
        try
        {
            InputStream maven2InputStream = this.getClass().getResourceAsStream("/" + maven2script);
            InputStream maven3InputStream = this.getClass().getResourceAsStream("/" + maven3script);

            File maven2File = new File(checkoutDirectory, maven2script);
            FileWriter maven2Writer = new FileWriter(maven2File);

            File maven3File = new File(checkoutDirectory, maven3script);
            FileWriter maven3Writer = new FileWriter(maven3File);

            IOUtils.copy(maven2InputStream, maven2Writer);
            IOUtils.copy(maven3InputStream, maven3Writer);

            maven2File.setExecutable(true, false);
            maven3File.setExecutable(true, false);
            
            maven2InputStream.close();
            maven3InputStream.close();
            maven2Writer.flush();
            maven3Writer.flush();
            maven2Writer.close();
            maven3Writer.close();
        }
        catch(IOException e)
        {
            throw new MojoExecutionException("Couldn't write mvn alias scripts: ", e);
        }

    }

    /**
     * Automatically detects the maven version for an atlassian project based on which version of the base pom it inherits from.
     * @param projects
     */
    public void computeMavenVersions (List<MavenProject> projects)
    {
        for (MavenProject mavenProject : projects)
        {
            if (mavenProject != null)
            {
                //Find the base pom version for this project
                MavenProject parent = mavenProject;
                MavenVersion mavenVersion = MavenVersion.MAVEN2;
                while (parent != null && !parent.getArtifactId().equals("atlassian-base-pom"))
                {
                    parent = parent.getParent();
                }

                if (parent != null && parent.getArtifactId() != null && parent.getArtifactId().equals("atlassian-base-pom"))
                {
                    try
                    {
                        Double version;
                        if (parent.getVersion().length() >= 2)
                        {
                            if(parent.getVersion().contains("."))
                            {
                                version = Double.parseDouble(parent.getVersion().substring(0, 3));
                            }
                            else{
                                version = Double.parseDouble(parent.getVersion().substring(0, 2));
                            }
                            if (version >= 37.0)
                            {
                                mavenVersion = MavenVersion.MAVEN3;
                            }
                        }
                    }
                    catch(NumberFormatException e)
                    {

                    }

                }
                mavenVersionMap.put(mavenProject, mavenVersion);
            }
        }
    }
    
    /*
     * Checks out a MavenProject to the given checkoutDirectory location
     */
    private void doCheckout(final MavenProject project, final File checkoutDirectory)
    	throws MojoExecutionException
    {
        doCheckout(project, checkoutDirectory, null);
    }

    /*
    * Checks out a MavenProject to the given checkoutDirectory location
    */
    private void doCheckout(final MavenProject project, final File checkoutDirectory, final String tagName)
    	throws MojoExecutionException
    {
        final Scm scm = project.getScm();
        if (scm != null && scm.getConnection() != null)
        {
            try
            {
                final RetryingTaskExecutor<Void> executor = new RetryingTaskExecutor(getLog(), retries);
                executor.runTask(new Callable<Void>()
                {
                    public Void call() throws Exception
                    {
                        ScmRepository scmRepository = getScmRepository(scm.getConnection());
                        ScmResult result = export(scmRepository, checkoutDirectory, project, tagName);
                        if (delay > 0)
                        {
                            try
                            {
                                Thread.sleep(delay);
                            }
                            catch (InterruptedException e)
                            {
                                throw new MojoExecutionException("Failed to delay checkout, root cause attached", e);
                            }
                        }

                        checkResult(result);

                        return null;
                    }
                });
            }
            catch (Exception e)
            {
                if (!ignoreFailures)
                {
                    throw new MojoExecutionException(e.getMessage(), e);
                }
                else
                {
                    //noinspection ThrowableResultOfMethodCallIgnored
                    final String groupId = project.getGroupId() + ":" + project.getArtifactId();
                    getLog().info("Error found when exporting " + groupId + ":" + project.getVersion() + ". Ignoring due to 'ignoreFailures' flag: " + e.getMessage());
                    ignoredFailures.put(groupId, e);
                }
            }
        }
        else
        {
            getLog().warn("Skipping " + project.getArtifactId() + " as it did not have a valid <scm> tag");
        }
    }


    /**
     * This method does most fo the hard work exporting the projects from SCM. The code is vastly different for each provider.
     * It should be able to work out maven submodules in svn, git and hg repos, provided that the directory structure acutally
     * reflects the maven project heirarchy. If not, repositoryMappings must be used in the configuration that calls this plugin.
     */
    private ScmResult export(ScmRepository scmRepository, final File checkoutDirectory, final MavenProject project, final String tagOverride) throws ScmException, MojoExecutionException
    {
        getLog().info("Exporting " + project.getGroupId() + ":" + project.getArtifactId() + ":" + project.getVersion());

        if (getScmManager().getProviderByRepository(scmRepository).getScmType().equals("git"))
        {
            getLog().debug("Git SCM provider does not support 'export', using workaround");

            GitScmProviderRepository repo = (GitScmProviderRepository) scmRepository.getProviderRepository();
            String fetchUrl = repo.getFetchUrl();
            MavenProject tagProject = GitScmProviderUtils.findRootProjectInSameRepository(project);
            String ref;

            ReleaseArtifactMapping mapping = determineReleaseArtifactMappingForProject(project);
            String tagPrefix = tagProject.getProperties().getProperty("atlassian.release.scm.tag.prefix");
            // NOTE(jwilson): Do a look up for the property in the project to see if it
            // specifies the tag prefix to use.
            if (tagPrefix != null)
            {
                ref = Ref.tag(tagPrefix + "-" + project.getVersion());
            }
            else if (mapping != null)
            {
                // NOTE(jwilson): Checks if there is a mapping for the project artifact to a different
                // tag if the tag name is changed during the release process.
                ref = Ref.tag(mapping.getScmTagPrefix() + "-" + project.getVersion());
            }
            else
            {
                ref = GitScmProviderUtils.refForProject(tagProject, this.productBranchName, new BranchVsTagFilter());
            }

            File dest = new File(checkoutDirectory, project.getArtifactId());

            getLog().info("Exporting ref '" + ref + "' from '" + repo + "' to: " + dest);

            if (!sourceDvcsCheckoutDirectory.isDirectory() && !sourceDvcsCheckoutDirectory.mkdir())
            {
                throw new ScmException("Unable to create temporary directory for DVCS export: " + sourceDvcsCheckoutDirectory);
            }

            ExportScmResult result = GitScmProviderUtils.export(sourceDvcsCheckoutDirectory, fetchUrl, ref, dest);
            if (new File(dest, ".git").exists())
            {
                throw new ScmException("Export failed. Git export included .git directory.");
            }
            return result;
        }
        else if (scmRepository.getProviderRepository() instanceof HgScmProviderRepository)
        {
            getLog().debug("Hg SCM provider does not support 'export', falling back to workaround");
            String subFolder = "";

            //Change the connection to the parent project if it's a submodule stored in bitbucket and in hg
            String repositoryUrl = ((HgScmProviderRepository) scmRepository.getProviderRepository()).getURI();

            //Since hg doesn't have nice url delimiters like ".git", we have to make assumptions about the url.
            getLog().info("Attempting to determine the base url of BitBucket hg repository: " + repositoryUrl);
            String[] urlComponents = repositoryUrl.split("/");
            String parentConnection = "scm:hg:" + join(Arrays.copyOfRange(urlComponents, 0, 5), "/");
            if (urlComponents.length > 5)
            {
                subFolder = join(Arrays.copyOfRange(urlComponents, 5, urlComponents.length), "/");
            }
            getLog().info("Determined base repository to be: " + parentConnection);
            scmRepository = getScmRepository(parentConnection);

            final ScmVersion hgTag;
            if (StringUtils.isBlank(tagOverride))
            {
                hgTag = findHgTag(project);
            }
            else
            {
                getLog().info("Checking out sandbox on a particular tag " + tagOverride);
                hgTag = new ScmTag("sandbox-" + tagOverride);
            }
            ExportScmResult result=  HgScmProviderUtils.export((HgScmProvider) getScmManager().getProviderByRepository(scmRepository),
                    scmRepository,
                    new ScmFileSet(checkoutDirectory),
                    hgTag,
                    project.getArtifactId());

            //Experimental - keep only the folder of the submodule, not the whole parent project
            if (result.isSuccess() && hgSubFolders && !StringUtils.isBlank(subFolder))
            {
                DvcsReplaceParentWithSubfolder(checkoutDirectory, project.getArtifactId(), subFolder);
            }
            return result;
        }

        return getScmManager().export(scmRepository, new ScmFileSet(checkoutDirectory), project.getArtifactId());
    }

    /**
     * Joins an array of strings, putting sep between each string.
     */
    private String join(String[] components, String sep)
    {
        String result = "";
        for (String component : components)
        {
            result = result + sep + component;
        }
        return result.substring(1);
    }

    /*
     * Method that allows the use of DVCS submodules. Copies a subfolder on top of the parent folder,
     * effectively just mv ./base/path/to/subdir/ ./subdir, rm -r ./base
     */
    private void DvcsReplaceParentWithSubfolder(File checkoutDirectory, String parentDirectory, String subFolder)
        throws MojoExecutionException
    {
        getLog().info("Hg: Clearing parent directory and keeping only subfolder " + subFolder);
        try
        {
            File outputDirectory = new File(checkoutDirectory, parentDirectory);
            File subDirectory = new File(outputDirectory, subFolder);
            File tempDir = new File(System.getProperty("java.io.tmpdir"), outputDirectory.getName());

            //copy the subdir out
            FileUtils.copyDirectoryStructure(subDirectory, tempDir);

            FileUtils.cleanDirectory(outputDirectory);

            //copy the subdir back
            FileUtils.copyDirectoryStructure(tempDir, outputDirectory);

        }
        catch(IOException e)
        {
            throw new MojoExecutionException("Unable to complete file operations: ", e);
        }
    }

    private ScmTag findHgTag(MavenProject project)
    {
        // go through parents files to find the pom.xml file that introduces the scm info we are using in this module
        // use the artifactId and versionId of that pom.xml to construct the tag

        getLog().info("Trying to find the first parent POM file that defines an SCM for this project...");

        MavenProject currentProject = project;
        boolean found = false;

        while (!found)
        {
            Model currentModel = currentProject.getOriginalModel();

            if (currentModel.getScm() != null)
            {
                getLog().info("SCM defined in: " + currentModel.getArtifactId() + "-" + currentModel.getVersion());
                found = true;
            }
            else
            {
                currentProject = currentProject.getParent();
            }
        }

        return new ScmTag(currentProject.getArtifactId() + "-" + currentProject.getVersion());
    }

    private void checkResult( ScmResult result )
    	throws MojoExecutionException
    {
        if ( !result.isSuccess() )
        {
            getLog().error( "Provider message:" );
            getLog().error( result.getProviderMessage() == null ? "" : result.getProviderMessage() );
            getLog().error( "Command output:" );
            getLog().error( result.getCommandOutput() == null ? "" : result.getCommandOutput() );
            throw new MojoExecutionException(
                "Command failed. " + StringUtils.defaultString( result.getProviderMessage() ) );
        }
    }

    private File createCheckoutDirectory()
    	throws MojoExecutionException
    {
    	String dirName = checkoutDirectoryName;

    	File checkoutDirectory = new File(outputDirectory, dirName);
        if (!checkoutDirectory.exists() && !checkoutDirectory.mkdirs())
        {
            throw new MojoExecutionException("Could not create directory " + checkoutDirectory.getAbsolutePath());
        }
    	return checkoutDirectory;
    }

    private ScmManager getScmManager()
    {
        return manager;
    }

    private ScmRepository getScmRepository(String connectionUrl)
    	throws ScmException
    {
        ScmRepository repository;
        getLog().info("Getting repository and provider details for " + connectionUrl);

        try
        {
            repository = getScmManager().makeScmRepository( connectionUrl );

            if ( repository.getProviderRepository() instanceof ScmProviderRepositoryWithHost )
            {
                ScmProviderRepositoryWithHost repo = (ScmProviderRepositoryWithHost) repository.getProviderRepository();
                ScmInfo info = loadInfosFromSettings( repo );

                if ( !StringUtils.isEmpty( info.getUsername() ) )
                {
                    repo.setUser( info.getUsername() );
                }

                if ( !StringUtils.isEmpty( info.getPassword() ) )
                {
                    repo.setPassword( decryptIfNecessary(info.getPassword()) );
                }

                if ( !StringUtils.isEmpty( info.getPrivateKey()) )
                {
                    repo.setPrivateKey( info.getPrivateKey() );
                }

                if ( !StringUtils.isEmpty( info.getPassphrase() ) )
                {
                    repo.setPassphrase( decryptIfNecessary(info.getPassphrase()) );
                }
            }
        }
        catch ( ScmRepositoryException e )
        {
            if ( !e.getValidationMessages().isEmpty() )
            {
                for ( Iterator i = e.getValidationMessages().iterator(); i.hasNext(); )
                {
                    String message = (String) i.next();
                    getLog().error( message );
                }
            }
            throw new ScmException( "Can't load the scm provider.", e );
        }
        catch ( Exception e )
        {
            throw new ScmException( "Can't load the scm provider.", e );
        }

        return repository;
    }

    private ScmInfo loadInfosFromSettings( ScmProviderRepositoryWithHost repo )
    {
    	ScmInfo info = new ScmInfo();
        String host = repo.getHost();
        int port = repo.getPort();

        if ( port > 0 )
        {
            host += ":" + port;
        }

        Server server = this.settings.getServer( host );
        if (server != null)
        {
            info.setUsername(this.settings.getServer( host ).getUsername());
            info.setPassword(this.settings.getServer( host ).getPassword());
            info.setPrivateKey(this.settings.getServer( host ).getPrivateKey());
            info.setPassphrase(this.settings.getServer( host ).getPassphrase());
        }

        return info;
    }

    private String decryptIfNecessary(String password)
    {
        PlexusContainer container = session.getContainer();
        SecDispatcher secDispatcher = null;

        try
        {
            secDispatcher = (SecDispatcher) container.lookup( SecDispatcher.ROLE, "maven" );
        }
        catch (ComponentLookupException e)
        {
            getLog().info("Could not find SecDispatcher, won't be able to decrypt passwords.");
        }

        if (secDispatcher != null)
        {
            try
            {
                password = secDispatcher.decrypt(password);
            }
            catch (SecDispatcherException e)
            {
                getLog().error("SecDispatcher error decrypting password.");
            }
        }

        return password;
    }

    private ArtifactFilter checkoutFromBranchFilter()
    {
        List<String> patterns = checkoutHeadIncludes != null ? checkoutHeadIncludes : Arrays.asList(SNAPSHOT_FILTER);

        return new StrictPatternIncludesArtifactFilter(patterns);
    }

    private class BranchVsTagFilter implements CheckoutFromTagFilter
    {
        public boolean checkoutBranch(MavenProject project)
        {
            // resolve the artifact and then see if it matches any of the exclusions
            Artifact artifact = artifactFactory.createArtifact(project.getGroupId(), project.getArtifactId(), project.getVersion(), "compile", "");

            return checkoutFromBranchFilter().include(artifact);
        }
    }
}