package com.atlassian.maven.plugins.sourcerelease.mojos;

//Portions Copyright Apache Software Foundation

import com.atlassian.maven.plugins.scm.ScmInfo;
import com.atlassian.maven.plugins.sourcerelease.configurations.ReleaseArtifactMapping;
import com.atlassian.maven.plugins.sourcerelease.mojos.git.GitScmProviderUtils;
import com.atlassian.maven.plugins.sourcerelease.mojos.git.Ref;
import com.atlassian.maven.plugins.sourcerelease.mojos.hg.HgScmProviderUtils;
import com.atlassian.maven.plugins.sourcerelease.util.RetryingTaskExecutor;
import org.apache.commons.lang.StringUtils;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.resolver.filter.ArtifactFilter;
import org.apache.maven.execution.MavenSession;
import org.apache.maven.model.Model;
import org.apache.maven.model.Scm;
import org.apache.maven.model.io.xpp3.MavenXpp3Writer;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;
import org.apache.maven.scm.NoSuchCommandScmException;
import org.apache.maven.scm.ScmException;
import org.apache.maven.scm.ScmFileSet;
import org.apache.maven.scm.ScmResult;
import org.apache.maven.scm.ScmTag;
import org.apache.maven.scm.ScmVersion;
import org.apache.maven.scm.command.export.ExportScmResult;
import org.apache.maven.scm.manager.ScmManager;
import org.apache.maven.scm.provider.ScmProviderRepositoryWithHost;
import org.apache.maven.scm.provider.git.repository.GitScmProviderRepository;
import org.apache.maven.scm.provider.hg.HgScmProvider;
import org.apache.maven.scm.provider.hg.repository.HgScmProviderRepository;
import org.apache.maven.scm.repository.ScmRepository;
import org.apache.maven.scm.repository.ScmRepositoryException;
import org.apache.maven.settings.Server;
import org.apache.maven.shared.artifact.filter.StrictPatternIncludesArtifactFilter;
import org.codehaus.plexus.PlexusContainer;
import org.codehaus.plexus.component.repository.exception.ComponentLookupException;
import org.sonatype.plexus.components.sec.dispatcher.SecDispatcher;
import org.sonatype.plexus.components.sec.dispatcher.SecDispatcherException;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;

/**
 * Goal exports dependencies from source control for a given groupId mask.
 *
 * @goal source
 * @phase package
 */
@SuppressWarnings({"ThrowableResultOfMethodCallIgnored"})
public class SourceMojo
    extends AbstractSourceDistributionMojo
{
    public static final String SNAPSHOT_FILTER = ":::*-SNAPSHOT";

    /**
     * @component
     */
    private ScmManager manager;

    /**
     * @parameter expression="${sourcedistribution.delay}" default-value="0"
     */
    private int delay;
    
    /**
     * @parameter default-value="0"
     */
    private int retries;

    /**
     * The Maven Session Object
     *
     * @parameter expression="${session}"
     * @required
     * @readonly
     */
    private MavenSession session;

    /**
     * @parameter expression="${project.build.directory}/source-dvcs-checkout"
     */
    private File sourceDvcsCheckoutDirectory;

    /**
     * The name of the currentBranch of the product the sources are built for.
     *
     * Can be used to make the source distribution build work for SNAPSHOT versions on the main (non-tag) branch/heads.
     *
     * @parameter
     */
    private String productBranchName;

    /**
     * A comma-separated list of artifacts for which we will <b>not</b> try to check out from a tag. By default, any
     * artifact that matches {@value #SNAPSHOT_FILTER} will not be exported from a tag, but will be exported from the
     * current branch instead.
     * <p/>
     * The artifact syntax is defined by <code>StrictPatternIncludesArtifactFilter</code>.
     *
     * @see org.apache.maven.shared.artifact.filter.StrictPatternIncludesArtifactFilter
     * @since 1.6.6
     *
     * @parameter
     */
    private List<String> checkoutHeadIncludes;

    private final Map<String, Exception> ignoredFailures = new HashMap();


    /**
     * If a sandbox key has been specified, then use a specific branch to check out of
     *
     * @parameter expression="${sandbox.key}"
     */
    private String sandboxKey;


    public void execute()
        throws MojoExecutionException
    {
        final Log log = getLog();
        if (skip)
        {
            log.info("Skipping source distribution execution");
            return;
        }

    	final File checkoutDirectory = createCheckoutDirectory();

        final Set<MavenProject> resolvedProjects = getResolvedProjects();

    	final List<String> modules = checkoutProjects(resolvedProjects, checkoutDirectory);

        MavenProject rootProject = reactorProjects.get(0);
        log.info("Checking out root project " + rootProject.getGroupId() + ":" + rootProject.getArtifactId());

        //Checkout root project last
        try
        {
            doCheckout(rootProject, checkoutDirectory, sandboxKey);
        }
        catch (Exception e)
        {
            log.warn("Tag does not exist for project " + project.getGroupId() + ":" + project.getArtifactId());
            log.debug(e);
        }

        if (ignoreFailures && !ignoredFailures.isEmpty())
        {
            log.warn("**** Failed to export " + ignoredFailures.size() + " dependencies, but will be ignored due to ignoreFailures flag.");
            log.warn("**** We recommend that you exclude the dependencies explicitly rather than use the ignoreFailures flag... Example follows");
            for (String s : ignoredFailures.keySet())
            {
                log.warn("        <exclusion>" + s + "</exclusion>");
            }
            log.warn("**** Exceptions are: ");
            for (Map.Entry<String, Exception> entry : ignoredFailures.entrySet())
            {
                log.warn(entry.getKey(), entry.getValue());
            }
        }

    	if (generatePom && !modules.isEmpty())
    	{
            writeSourceDistributionPom(modules, checkoutDirectory);
    	}
    }

    
    /*
     * Checks out artifacts and returns a list of their artifactIds
     */
    private List<String> checkoutProjects(Set<MavenProject> projects, File checkoutDirectory)
    	throws MojoExecutionException
    {
        getLog().info("Will checkout the following artifacts:");
        for (MavenProject currentProject : projects)
        {
            getLog().info(currentProject.getGroupId() + ":" + currentProject.getArtifactId());
        }

        getLog().info("A delay of " + delay + " ms has been specified between checkouts.");
    	final List<String> modules = new ArrayList();
    	for (MavenProject currentProject : projects)
        {
            final String projectArtifactId = currentProject.getArtifactId();
            if (modules.contains(projectArtifactId))
            {
                throw new MojoExecutionException("Cannot have two modules checked out with the same directory name " + projectArtifactId);
            }
            modules.add(currentProject.getArtifactId());
            doCheckout(currentProject, checkoutDirectory);
        }

    	return modules;
    }

    /*
     * Writes out the generated project pom
     */
    private void writeSourceDistributionPom(List<String> modules, File checkoutDirectory)
    	throws MojoExecutionException
    {
    	Exception ex = null;

        Model model = new Model();
        model.setModelVersion( "4.0.0" );
        model.setGroupId( groupId );
        model.setArtifactId( artifactId );
        model.setVersion( version );
        model.setName( productName + " Source Release"  );
        model.setPackaging( "pom" );
        model.setDescription( "Source for " + productName );
        model.setModules(modules);

        FileWriter writer;
        try
        {
            writer = new FileWriter(new File(checkoutDirectory, "pom.xml"));
            new MavenXpp3Writer().write(writer, model);
            writer.flush();
            writer.close();
        }
        catch (IOException e)
        {
            throw new MojoExecutionException("Could not write source release pom", ex);
        }
    }

    /*
     * Checks out a MavenProject to the given checkoutDirectory location
     */
    private void doCheckout(final MavenProject project, final File checkoutDirectory)
    	throws MojoExecutionException
    {
        doCheckout(project, checkoutDirectory, null);
    }

    /*
    * Checks out a MavenProject to the given checkoutDirectory location
    */
    private void doCheckout(final MavenProject project, final File checkoutDirectory, final String tagName)
    	throws MojoExecutionException
    {
        final Scm scm = project.getScm();
        if (scm != null && scm.getConnection() != null)
        {
            try
            {
                final RetryingTaskExecutor<Void> executor = new RetryingTaskExecutor(getLog(), retries);
                executor.runTask(new Callable<Void>()
                {
                    public Void call() throws Exception
                    {
                        ScmRepository scmRepository = getScmRepository(scm.getConnection());
                        ScmResult result = export(scmRepository, checkoutDirectory, project, tagName);
                        if (delay > 0)
                        {
                            try
                            {
                                Thread.sleep(delay);
                            }
                            catch (InterruptedException e)
                            {
                                throw new MojoExecutionException("Failed to delay checkout, root cause attached", e);
                            }
                        }

                        checkResult(result);

                        return null;
                    }
                });
            }
            catch (Exception e)
            {
                if (!ignoreFailures)
                {
                    throw new MojoExecutionException(e.getMessage(), e);
                }
                else
                {
                    //noinspection ThrowableResultOfMethodCallIgnored
                    final String groupId = project.getGroupId() + ":" + project.getArtifactId();
                    getLog().info("Error found when exporting " + groupId + ":" + project.getVersion() + ". Ignoring due to 'ignoreFailures' flag: " + e.getMessage());
                    ignoredFailures.put(groupId, e);
                }
            }
        }
        else
        {
            getLog().warn("Skipping " + project.getArtifactId() + " as it did not have a valid <scm> tag");
        }
    }


    private ScmResult export(final ScmRepository scmRepository, final File checkoutDirectory, final MavenProject project, final String tagOverride) throws ScmException
    {
        getLog().info("Exporting " + project.getGroupId() + ":" + project.getArtifactId() + ":" + project.getVersion());

        if (getScmManager().getProviderByRepository(scmRepository).getScmType().equals("git"))
        {
            getLog().debug("Git SCM provider does not support 'export', using workaround");

            GitScmProviderRepository repo = (GitScmProviderRepository) scmRepository.getProviderRepository();
            String fetchUrl = repo.getFetchUrl();
            MavenProject tagProject = GitScmProviderUtils.findRootProjectInSameRepository(project);
            Ref ref;

            ReleaseArtifactMapping mapping = determineReleaseArtifactMappingForProject(project);
            String tagPrefix = tagProject.getProperties().getProperty("atlassian.release.scm.tag.prefix");
            // NOTE(jwilson): Do a look up for the property in the project to see if it
            // specifies the tag prefix to use.
            if (tagPrefix != null)
            {
                ref = Ref.tag(tagPrefix + "-" + project.getVersion());
            }
            else if (mapping != null)
            {
                // NOTE(jwilson): Checks if there is a mapping for the project artifact to a different
                // tag if the tag name is changed during the release process.
                ref = Ref.tag(mapping.getScmTagPrefix() + "-" + project.getVersion());
            }
            else
            {
                ref = GitScmProviderUtils.refForProject(tagProject, this.productBranchName, new BranchVsTagFilter());
            }

            File dest = new File(checkoutDirectory, project.getArtifactId());
            
            getLog().info("Exporting ref '" + ref + "' from '" + repo + "' to: " + dest);
            
            if (!sourceDvcsCheckoutDirectory.isDirectory() && !sourceDvcsCheckoutDirectory.mkdir())
            {
                throw new ScmException("Unable to create temporary directory for DVCS export: " + sourceDvcsCheckoutDirectory);
            }
            
            ExportScmResult result = GitScmProviderUtils.export(sourceDvcsCheckoutDirectory, fetchUrl, ref, dest);
            if (new File(dest, ".git").exists())
            {
                throw new ScmException("Export failed. Git export included .git directory.");
            }
            return result;
        }
        
        try
        {
            return getScmManager().export(scmRepository, new ScmFileSet(checkoutDirectory), project.getArtifactId());
        }
        catch (NoSuchCommandScmException e)
        {
            if (scmRepository.getProviderRepository() instanceof HgScmProviderRepository)
            {
                getLog().debug("Hg SCM provider does not support 'export', falling back to workaround");
                final ScmVersion hgTag;
                if (StringUtils.isBlank(tagOverride))
                {
                    hgTag = findHgTag(project);
                }
                else
                {
                    getLog().info("Checking out sandbox on a particular tag " + tagOverride);
                    hgTag = new ScmTag("sandbox-" + tagOverride);
                }
                return HgScmProviderUtils.export((HgScmProvider) getScmManager().getProviderByRepository(scmRepository),
                                                 scmRepository,
                                                 new ScmFileSet(checkoutDirectory),
                                                 hgTag,
                                                 project.getArtifactId());
            }
            else
            {
                throw e;
            }
        }
    }

    private ScmTag findHgTag(MavenProject project)
    {
        // go through parents files to find the pom.xml file that introduces the scm info we are using in this module
        // use the artifactId and versionId of that pom.xml to construct the tag

        getLog().info("Trying to find the first parent POM file that defines an SCM for this project...");

        MavenProject currentProject = project;
        boolean found = false;
        
        while (!found)
        {
            Model currentModel = currentProject.getOriginalModel();

            if (currentModel.getScm() != null)
            {
                getLog().info("SCM defined in: " + currentModel.getArtifactId() + "-" + currentModel.getVersion());
                found = true;
            }
            else
            {
                currentProject = currentProject.getParent();
            }
        }

        return new ScmTag(currentProject.getArtifactId() + "-" + currentProject.getVersion());
    }

    private void checkResult( ScmResult result )
    	throws MojoExecutionException
    {
        if ( !result.isSuccess() )
        {
            getLog().error( "Provider message:" );
            getLog().error( result.getProviderMessage() == null ? "" : result.getProviderMessage() );
            getLog().error( "Command output:" );
            getLog().error( result.getCommandOutput() == null ? "" : result.getCommandOutput() );
            throw new MojoExecutionException(
                "Command failed. " + StringUtils.defaultString( result.getProviderMessage() ) );
        }
    }

    private File createCheckoutDirectory()
    	throws MojoExecutionException
    {
    	String dirName = checkoutDirectoryName;

    	File checkoutDirectory = new File(outputDirectory, dirName);
        if (!checkoutDirectory.exists() && !checkoutDirectory.mkdirs())
        {
            throw new MojoExecutionException("Could not create directory " + checkoutDirectory.getAbsolutePath());
        }
    	return checkoutDirectory;
    }

    private ScmManager getScmManager()
    {
        return manager;
    }

    private ScmRepository getScmRepository(String connectionUrl)
    	throws ScmException
    {
        ScmRepository repository;

        try
        {
            repository = getScmManager().makeScmRepository( connectionUrl );

            if ( repository.getProviderRepository() instanceof ScmProviderRepositoryWithHost )
            {
                ScmProviderRepositoryWithHost repo = (ScmProviderRepositoryWithHost) repository.getProviderRepository();
                ScmInfo info = loadInfosFromSettings( repo );

                if ( !StringUtils.isEmpty( info.getUsername() ) )
                {
                    repo.setUser( info.getUsername() );
                }

                if ( !StringUtils.isEmpty( info.getPassword() ) )
                {
                    repo.setPassword( decryptIfNecessary(info.getPassword()) );
                }

                if ( !StringUtils.isEmpty( info.getPrivateKey()) )
                {
                    repo.setPrivateKey( info.getPrivateKey() );
                }

                if ( !StringUtils.isEmpty( info.getPassphrase() ) )
                {
                    repo.setPassphrase( decryptIfNecessary(info.getPassphrase()) );
                }
            }
        }
        catch ( ScmRepositoryException e )
        {
            if ( !e.getValidationMessages().isEmpty() )
            {
                for ( Iterator i = e.getValidationMessages().iterator(); i.hasNext(); )
                {
                    String message = (String) i.next();
                    getLog().error( message );
                }
            }
            throw new ScmException( "Can't load the scm provider.", e );
        }
        catch ( Exception e )
        {
            throw new ScmException( "Can't load the scm provider.", e );
        }

        return repository;
    }

    private ScmInfo loadInfosFromSettings( ScmProviderRepositoryWithHost repo )
    {
    	ScmInfo info = new ScmInfo();
        String host = repo.getHost();
        int port = repo.getPort();

        if ( port > 0 )
        {
            host += ":" + port;
        }

        Server server = this.settings.getServer( host );
        if (server != null)
        {
            info.setUsername(this.settings.getServer( host ).getUsername());
            info.setPassword(this.settings.getServer( host ).getPassword());
            info.setPrivateKey(this.settings.getServer( host ).getPrivateKey());
            info.setPassphrase(this.settings.getServer( host ).getPassphrase());
        }

        return info;
    }

    private String decryptIfNecessary(String password)
    {
        PlexusContainer container = session.getContainer();
        SecDispatcher secDispatcher = null;

        try
        {
            secDispatcher = (SecDispatcher) container.lookup( SecDispatcher.ROLE, "maven" );
        }
        catch (ComponentLookupException e)
        {
            getLog().info("Could not find SecDispatcher, won't be able to decrypt passwords.");
        }

        if (secDispatcher != null)
        {
            try
            {
                password = secDispatcher.decrypt(password);
            }
            catch (SecDispatcherException e)
            {
                getLog().error("SecDispatcher error decrypting password.");
            }
        }

        return password;
    }

    private ArtifactFilter checkoutFromBranchFilter()
    {
        List<String> patterns = checkoutHeadIncludes != null ? checkoutHeadIncludes : Arrays.asList(SNAPSHOT_FILTER);

        return new StrictPatternIncludesArtifactFilter(patterns);
    }

    private class BranchVsTagFilter implements CheckoutFromTagFilter
    {
        public boolean checkoutBranch(MavenProject project)
        {
            // resolve the artifact and then see if it matches any of the exclusions
            Artifact artifact = artifactFactory.createArtifact(project.getGroupId(), project.getArtifactId(), project.getVersion(), "compile", "");

            return checkoutFromBranchFilter().include(artifact);
        }
    }
}
