package com.atlassian.maven.plugins.sourcerelease.mojos.hg;

import org.apache.commons.lang.StringUtils;
import org.apache.maven.scm.CommandParameter;
import org.apache.maven.scm.CommandParameters;
import org.apache.maven.scm.ScmException;
import org.apache.maven.scm.ScmFileSet;
import org.apache.maven.scm.ScmResult;
import org.apache.maven.scm.ScmVersion;
import org.apache.maven.scm.command.Command;
import org.apache.maven.scm.command.export.AbstractExportCommand;
import org.apache.maven.scm.command.export.ExportScmResult;
import org.apache.maven.scm.command.export.ExportScmResultWithRevision;
import org.apache.maven.scm.provider.ScmProviderRepository;
import org.apache.maven.scm.provider.hg.HgUtils;
import org.apache.maven.scm.provider.hg.command.HgCommandConstants;
import org.apache.maven.scm.provider.hg.command.HgConsumer;
import org.apache.maven.scm.provider.hg.command.checkout.HgCheckOutCommand;
import org.apache.maven.scm.provider.hg.repository.HgScmProviderRepository;

import java.io.File;

public class HgExportCommand extends AbstractExportCommand
    implements Command
{

    @Override
    protected ExportScmResult executeExportCommand(final ScmProviderRepository repository, final ScmFileSet fileSet, final ScmVersion scmVersion, final String outputDirectory) throws ScmException
    {
        return executeExportCommand((HgScmProviderRepository) repository, fileSet, scmVersion, outputDirectory);
    }

    private ExportScmResult executeExportCommand(HgScmProviderRepository repository, ScmFileSet fileSet, ScmVersion scmVersion, String outputDirectory) throws ScmException {
        ScmFileSet checkoutDirectory = new ScmFileSet(new File(fileSet.getBasedir(), outputDirectory + "-clone"));

        ScmFileSet repositoryLocation;

        if (isRepositoryAvailable(repository, fileSet)) //Are we within a Mercurial checkout from a correct repository?
        {
            getLogger().info("Detected that we are running from within a valid Mercurial repository clone, we will now update it and use as a source of export");
            repositoryLocation = fileSet;
        }
        else if (isRepositoryAvailable(repository, checkoutDirectory)) //maybe we've checked out the repository already?
        {
            getLogger().info("Detected a valid Mercurial repository clone in " + checkoutDirectory.getBasedir() + ", we will now update it and use as a source of export");
            repositoryLocation = checkoutDirectory;
        }
        else
        {
            getLogger().warn("No valid checked-out Mercurial repositories found, cloning remote repository " + repository.getURI());
            //we have to do a full checkout
            ExportScmResult result = checkoutRepository(repository, checkoutDirectory, scmVersion);
            if (!result.isSuccess())
            {
                return result;
            }
            repositoryLocation = checkoutDirectory;
        }

        return exportLocal(repositoryLocation, scmVersion, outputDirectory);
    }

    // Executes a hg clone
    private ExportScmResult checkoutRepository(final ScmProviderRepository repository, final ScmFileSet checkoutDirectory, final ScmVersion scmVersion) throws ScmException
    {
        CommandParameters parameters = new CommandParameters();
        parameters.setScmVersion(CommandParameter.SCM_VERSION, scmVersion);

        //We could use the --noupdate switch here but we would have to clone HgCheckOutCommand for that
        final HgCheckOutCommand command = new HgCheckOutCommand();
        command.setLogger(getLogger());
        ScmResult result = command.execute(repository, checkoutDirectory, parameters);
        
        return newExportScmResult(result);
    }

    //export a locally available repository into unversioned files
    private ExportScmResult exportLocal(final ScmFileSet fileSet, final ScmVersion scmVersion, final String outputDirectory) throws ScmException
    {
        ScmResult result = updateLocalRepository(fileSet, scmVersion);

        if (!result.isSuccess())
        {
            return newExportScmResult(result);
        }

        result = archiveLocalRepository(fileSet, scmVersion, outputDirectory);

        if (!result.isSuccess())
        {
            return newExportScmResult(result);
        }

        return new ExportScmResultWithRevision(result.getCommandLine(), result.getProviderMessage(), result.getCommandOutput(), toString(scmVersion), result.isSuccess());
    }

    //checks if a given location is
    private boolean isRepositoryAvailable(final HgScmProviderRepository repository, final ScmFileSet fileSet)
    {
        File workingDir = fileSet.getBasedir();
        if (!workingDir.isDirectory())
        {
            getLogger().info("Directory " + workingDir + " does not exist");
            return false;
        }
        try
        {
            String[] archiveCmd = new String[]{
                    HgCommandConstantsExtended.PATH_CMD
            };
            HgPathConsumer consumer = new HgPathConsumer(getLogger());
            ScmResult result = HgUtils.execute(consumer, getLogger(), workingDir, archiveCmd);
            return result.isSuccess() && pathMatches(repository, consumer.getPathForAlias("default"));
        }
        catch (ScmException e)
        {
            return false;
        }
    }

    private boolean pathMatches(HgScmProviderRepository repository, String path) {
        getLogger().debug("Checking if [" + repository.getURI() + "] is [" + path + "]");
        return repository.getURI().equals(path);
    }

    private ScmResult updateLocalRepository(final ScmFileSet fileSet, final ScmVersion tag) throws ScmException
    {
        File workingDir = fileSet.getBasedir();
                                                                                                                                                                                                                                                                               
        String[] updateCmd = new String[] {
                HgCommandConstants.PULL_CMD,
                HgCommandConstants.REVISION_OPTION,
                toString(tag)};
        ScmResult updateResult = HgUtils.execute(new HgConsumer(getLogger()), getLogger(), workingDir, updateCmd);

        return updateResult;
    }

    private ScmResult archiveLocalRepository(final ScmFileSet fileSet, final ScmVersion tag, final String outputDirectory) throws ScmException
    {
        File workingDir = fileSet.getBasedir();

        String[] archiveCmd = new String[]{
                HgCommandConstantsExtended.ARCHIVE_CMD,
                HgCommandConstants.REVISION_OPTION,
                toString(tag),
                outputDirectory};
        ScmResult archiveResult = HgUtils.execute(new HgConsumer(getLogger()), getLogger(), workingDir, archiveCmd);

        return archiveResult;
    }

    private ExportScmResult newExportScmResult(final ScmResult updateResult)
    {
        return new ExportScmResult(updateResult.getCommandLine(), updateResult.getProviderMessage(), updateResult.getCommandOutput(), updateResult.isSuccess());
    }

    private String toString(final ScmVersion tag)
    {
        return tag != null && !StringUtils.isEmpty(tag.getName()) ? tag.getName() : "tip";
    }
}
