package com.atlassian.maven.plugins.sourcerelease.mojos;

//Portions Copyright Apache Software Foundation

import com.atlassian.maven.plugins.scm.ScmInfo;
import com.atlassian.maven.plugins.sourcerelease.mojos.hg.HgScmProviderUtils;
import com.atlassian.maven.plugins.sourcerelease.util.RetryingTaskExecutor;
import org.apache.commons.lang.StringUtils;
import org.apache.maven.model.Model;
import org.apache.maven.model.Scm;
import org.apache.maven.model.io.xpp3.MavenXpp3Writer;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.MavenProject;
import org.apache.maven.scm.NoSuchCommandScmException;
import org.apache.maven.scm.ScmException;
import org.apache.maven.scm.ScmFileSet;
import org.apache.maven.scm.ScmResult;
import org.apache.maven.scm.manager.ScmManager;
import org.apache.maven.scm.provider.ScmProviderRepositoryWithHost;
import org.apache.maven.scm.provider.hg.HgScmProvider;
import org.apache.maven.scm.provider.hg.repository.HgScmProviderRepository;
import org.apache.maven.scm.repository.ScmRepository;
import org.apache.maven.scm.repository.ScmRepositoryException;
import org.apache.maven.settings.Server;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Callable;

/**
 * Goal checkes out dependencies from source control for a given groupId mask
 *
 * @goal source
 *
 * @phase package
 */
public class SourceMojo
    extends AbstractSourceDistributionMojo
{
    /**
     * @component
     */
    private ScmManager manager;

    /**
     * @parameter expression="${sourcedistribution.delay}" default-value="0"
     */
    private int delay;
    
    /**
     * @parameter default-value="0"
     */
    private int retries;


    public void execute()
        throws MojoExecutionException
    {
        if (skip)
        {
            getLog().info("Skipping source distribution execution");
            return;
        }

    	final File checkoutDirectory = createCheckoutDirectory();
        final Set<MavenProject> projects;

        final Set<MavenProject> resolvedProjects = getResolvedProjects();

    	final List<String> modules = checkoutProjects(resolvedProjects, checkoutDirectory);

        MavenProject rootProject = reactorProjects.get(0);
        getLog().info("Checking out root project " + rootProject.getGroupId() + ":" + rootProject.getArtifactId());

        //Checkout root project last
        try
        {
            doCheckout(rootProject, checkoutDirectory);
        }
        catch (Exception e)
        {
            getLog().warn("Tag does not exist for project " + project.getGroupId() + ":" + project.getArtifactId());
            getLog().debug(e);
        }
    	
    	if (generatePom && !modules.isEmpty())
    	{
            writeSourceDistributionPom(modules, checkoutDirectory);
    	}
    }

    
    /*
     * Checks out artifacts and returns a list of their artifactIds
     */
    private List<String> checkoutProjects(Set<MavenProject> projects, File checkoutDirectory)
    	throws MojoExecutionException
    {
        getLog().info("Will checkout the following artifacts:");
        for (MavenProject currentProject : projects)
        {
            getLog().info(currentProject.getGroupId() + ":" + currentProject.getArtifactId());
        }

        getLog().info("A delay of " + delay + " ms has been specified between checkouts.");
    	final List<String> modules = new ArrayList();
    	for (MavenProject currentProject : projects)
        {
            final String projectArtifactId = currentProject.getArtifactId();
            if (modules.contains(projectArtifactId))
            {
                throw new MojoExecutionException("Cannot have two modules checked out with the same directory name " + projectArtifactId);
            }
            modules.add(currentProject.getArtifactId());
            doCheckout(currentProject, checkoutDirectory);
        }

    	return modules;
    }

    /*
     * Writes out the generated project pom
     */
    private void writeSourceDistributionPom(List<String> modules, File checkoutDirectory)
    	throws MojoExecutionException
    {
    	Exception ex = null;

        Model model = new Model();
        model.setModelVersion( "4.0.0" );
        model.setGroupId( groupId );
        model.setArtifactId( artifactId );
        model.setVersion( version );
        model.setName( productName + " Source Release"  );
        model.setPackaging( "pom" );
        model.setDescription( "Source for " + productName );
        model.setModules(modules);

        FileWriter writer;
        try
        {
            writer = new FileWriter(new File(checkoutDirectory, "pom.xml"));
            new MavenXpp3Writer().write(writer, model);
            writer.flush();
            writer.close();
        }
        catch (IOException e)
        {
            throw new MojoExecutionException("Could not write source release pom", ex);
        }
    }

    /*
     * Checks out a MavenProject to the given checkoutDirectory location
     */
    private void doCheckout(final MavenProject project, final File checkoutDirectory)
    	throws MojoExecutionException
    {
        final Scm scm = project.getScm();
        if (scm != null && scm.getConnection() != null)
        {
            try
            {
                final RetryingTaskExecutor<Void> executor = new RetryingTaskExecutor(getLog(), retries);
                executor.runTask(new Callable<Void>()
                {
                    public Void call() throws Exception
                    {
                        ScmRepository scmRepository = getScmRepository(scm.getConnection());
                        ScmResult result = export(scmRepository, checkoutDirectory, project);
                        if (delay > 0)
                        {
                            try
                            {
                                Thread.sleep(delay);
                            }
                            catch (InterruptedException e)
                            {
                                throw new MojoExecutionException("Failed to delay checkout, root cause attached", e);
                            }
                        }

                        checkResult(result);

                        return null;
                    }
                });
            }
            catch (Exception e)
            {
                throw new MojoExecutionException(e.getMessage(), e);
            }
        }
        else
        {
            getLog().warn("Skipping " + project.getArtifactId() + " as it did not have a valid <scm> tag");
        }
    }

    private ScmResult export(final ScmRepository scmRepository, final File checkoutDirectory, final MavenProject project) throws ScmException
    {
        getLog().info("Exporting " + project.getGroupId() + ":" + project.getArtifactId() + ":" + project.getVersion());
        try
        {
            return getScmManager().export(scmRepository, new ScmFileSet(checkoutDirectory), project.getArtifactId());
        }
        catch (NoSuchCommandScmException e)
        {
            if (scmRepository.getProviderRepository() instanceof HgScmProviderRepository)
            {
                getLog().debug("Hg SCM provider does not support 'export', falling back to workaround");
                return HgScmProviderUtils.export((HgScmProvider) getScmManager().getProviderByRepository(scmRepository),
                                                 scmRepository,
                                                 new ScmFileSet(checkoutDirectory),
                                                 project.getArtifactId());
            }
            else
            {
                throw e;
            }
        }
    }

    private void checkResult( ScmResult result )
    	throws MojoExecutionException
    {
        if ( !result.isSuccess() )
        {
            getLog().error( "Provider message:" );
            getLog().error( result.getProviderMessage() == null ? "" : result.getProviderMessage() );
            getLog().error( "Command output:" );
            getLog().error( result.getCommandOutput() == null ? "" : result.getCommandOutput() );
            throw new MojoExecutionException(
                "Command failed. " + StringUtils.defaultString( result.getProviderMessage() ) );
        }
    }

    private File createCheckoutDirectory()
    	throws MojoExecutionException
    {
    	String dirName = checkoutDirectoryName;

    	File checkoutDirectory = new File(outputDirectory, dirName);
        if (!checkoutDirectory.exists() && !checkoutDirectory.mkdirs())
        {
            throw new MojoExecutionException("Could not create directory " + checkoutDirectory.getAbsolutePath());
        }
    	return checkoutDirectory;
    }

    private ScmManager getScmManager()
    {
        return manager;
    }

    private ScmRepository getScmRepository(String connectionUrl)
    	throws ScmException
    {
        ScmRepository repository;

        try
        {
            repository = getScmManager().makeScmRepository( connectionUrl );

            if ( repository.getProviderRepository() instanceof ScmProviderRepositoryWithHost )
            {
                ScmProviderRepositoryWithHost repo = (ScmProviderRepositoryWithHost) repository.getProviderRepository();
                ScmInfo info = loadInfosFromSettings( repo );

                if ( !StringUtils.isEmpty( info.getUsername() ) )
                {
                    repo.setUser( info.getUsername() );
                }

                if ( !StringUtils.isEmpty( info.getPassword() ) )
                {
                    repo.setPassword( info.getPassword() );
                }

                if ( !StringUtils.isEmpty( info.getPrivateKey()) )
                {
                    repo.setPrivateKey( info.getPrivateKey() );
                }

                if ( !StringUtils.isEmpty( info.getPassphrase() ) )
                {
                    repo.setPassphrase( info.getPassphrase() );
                }
            }
        }
        catch ( ScmRepositoryException e )
        {
            if ( !e.getValidationMessages().isEmpty() )
            {
                for ( Iterator i = e.getValidationMessages().iterator(); i.hasNext(); )
                {
                    String message = (String) i.next();
                    getLog().error( message );
                }
            }
            throw new ScmException( "Can't load the scm provider.", e );
        }
        catch ( Exception e )
        {
            throw new ScmException( "Can't load the scm provider.", e );
        }

        return repository;
    }

    private ScmInfo loadInfosFromSettings( ScmProviderRepositoryWithHost repo )
    {
    	ScmInfo info = new ScmInfo();
        String host = repo.getHost();
        int port = repo.getPort();

        if ( port > 0 )
        {
            host += ":" + port;
        }

        Server server = this.settings.getServer( host );
        if (server != null)
        {
            info.setUsername(this.settings.getServer( host ).getUsername());
            info.setPassword(this.settings.getServer( host ).getPassword());
            info.setPrivateKey(this.settings.getServer( host ).getPrivateKey());
            info.setPassphrase(this.settings.getServer( host ).getPassphrase());
        }

        return info;
    }
}
