package com.atlassian.maven.plugins.sourcerelease.mojos;

import com.atlassian.maven.plugins.sourcerelease.configurations.RepositoryMapping;
import com.atlassian.maven.plugins.sourcerelease.configurations.SourceDependency;
import com.atlassian.maven.plugins.sourcerelease.util.SCMPathType;
import com.atlassian.maven.plugins.sourcerelease.util.filters.OrArtifactFilter;
import org.apache.commons.lang.StringUtils;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.DefaultArtifact;
import org.apache.maven.artifact.factory.ArtifactFactory;
import org.apache.maven.artifact.metadata.ArtifactMetadataSource;
import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.artifact.resolver.ArtifactNotFoundException;
import org.apache.maven.artifact.resolver.ArtifactResolutionException;
import org.apache.maven.artifact.resolver.ArtifactResolutionResult;
import org.apache.maven.artifact.resolver.ArtifactResolver;
import org.apache.maven.artifact.resolver.filter.ArtifactFilter;
import org.apache.maven.artifact.resolver.filter.ScopeArtifactFilter;
import org.apache.maven.model.Scm;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.InvalidProjectModelException;
import org.apache.maven.project.MavenProject;
import org.apache.maven.project.MavenProjectBuilder;
import org.apache.maven.project.ProjectBuildingException;
import org.apache.maven.project.artifact.InvalidDependencyVersionException;
import org.apache.maven.settings.Settings;

import java.io.File;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * TODO: Document this class / interface here
 *
 * @since v4.1
 */
public abstract class AbstractSourceDistributionMojo extends AbstractMojo
{
    /**
     * Location of the file.
     * @parameter expression="${project.build.directory}"
     * @required
     */
    protected File outputDirectory;

    /**
     * Upload a POM for this artifact.  Will generate a default POM if none is
     * supplied with the pomFile argument.
     *
     * @parameter expression="${sourcedistribution.generatePom}" default-value="true"
     */
    protected boolean generatePom;

    /**
     * GroupId of the generated pom
     * @parameter
     * @required
     */
    protected String groupId;

    /**
     * ArtifactId of the generated source release pom
     * @parameter
     * @required
     */
    protected String artifactId;

    /**
     * Version of the generated pom
     * @parameter
     * @required
     */
    protected String version;

    /**
     * Product Name of the generated pom (eg. Atlassian Confluence)
     * @parameter
     * @required
     */
    protected String productName;

    /**
     * Directory name of the location in the output directory where the source release project will be generated
     * @parameter default-value="checkouts"
     */
    protected String checkoutDirectoryName;

    /**
     * Mask of groupId to include in source release (eg. "com.atlassian." or "org.apache.maven.plugins." )
     * @parameter
     * @required
     */
    protected String groupIdMask;

    /**
     * List of artifacts to exclude from the source release. Format "groupId:artifactId"
     * @parameter
     */
    protected List exclusions;

    /**
     * List of artifacts to exclude from the source release including all their dependencies. Format "groupId:artifactId"
     *
     * @parameter
     */
    protected List dependencyExclusions;

    /**
     * Skips the execution
     * @parameter expression="${sourcedistribution.skip}" default-value="false"
     */
    protected boolean skip;

    /**
     * Whether to resolve and check out the parent project for modules of a multi-module projects
     * @parameter expression="${sourcedistribution.resolveRoot}" default-value="false"
     */
    protected boolean resolveRoot;

    /**
     * Specific source dependency configurations
     * @parameter expression="${sourceDepConfiguration}"
     */
    protected SourceDependency[] sourceDependencies;

    /**
     * List of exclusions based on scm prefix
     * @parameter
     */
    protected List scmPrefixExclusions;


    /**
     * Specific source dependency configurations
     * @parameter expression="${repositoryMappingConfiguration}"
     */
    protected RepositoryMapping[] repositoryMappings;

    /**
     * Whether to use the reactor artifacts for dependency resolution or the artifacts in the
     * source distribution pom. If false, the dependencies of the project are resolved
     * transitively and filtered to include only runtime and compile scopes.
     * @parameter expression="${useReactor}" default-value="true"
     *
     */
    protected boolean useReactor;

    /**
     * @parameter expression="${project}"
     * @required
     */
    protected MavenProject project;

    /**
     * @component
     */
    protected ArtifactFactory artifactFactory;

    /**
     * All projects listed in the reactor
     * @parameter expression="${reactorProjects}"
     * @required
     */
    protected List<MavenProject> reactorProjects;

    /**
     * @component
     */
    protected MavenProjectBuilder projectBuilder;

    /**
     * @component
     */
    protected ArtifactResolver resolver;

    /**
     * @parameter expression="${settings}"
     * @required
     * @readonly
     */
    protected Settings settings;

    /**
     * @parameter default-value="${localRepository}"
     * @required
     * @readonly
     */
    protected ArtifactRepository localRepository;

    /**
     * @parameter default-value="${project.remoteArtifactRepositories}"
     * @required
     * @readonly
     */
    protected List remoteRepositories;

    /**
     * @component
     */
    protected ArtifactMetadataSource metadataSource;

    
    protected Set<MavenProject> getResolvedProjects() throws MojoExecutionException
    {
        Set<Artifact> dependencies = useReactor ? getReactorProjectDependencies() : getProjectDependencies();

        removeExcludedDependencies(dependencies);

        final Set<MavenProject> projects = getProjectsForArtifacts(dependencies);

        if (projects == null)
        {
            throw new MojoExecutionException("Could not resolve dependent projects");
        }

        //Remove undesirables
        if (useReactor)
        {
            removeReactorProjects(projects);
        }

        removeExcludedArtifacts(projects);

        //Resolve root projects, if needed
        final Set<MavenProject> resolvedProjects =
                removeScmPrefixExclusions(
                        processProjects(projects)
                );

        if (reactorProjects == null || reactorProjects.size() < 1)
        {
            throw new MojoExecutionException("There are no dependent projects left to process.");
        }

        return resolvedProjects;
    }

    private Set<MavenProject> removeScmPrefixExclusions(final Set<MavenProject> resolvedProjects)
    {
        if(scmPrefixExclusions == null)
        {
            return resolvedProjects;
        }
        final Set<MavenProject> includedProjects = new HashSet<MavenProject>(resolvedProjects);
        for(final Object scmExclusion : scmPrefixExclusions)
        {
            final Set<MavenProject> excludedProjects = new HashSet<MavenProject>();
            for(final MavenProject project : includedProjects)
            {
                if(project.getScm().getConnection().startsWith((String) scmExclusion))
                {
                    excludedProjects.add(project);
                }
            }
            includedProjects.removeAll(excludedProjects);
        }
        return includedProjects;
    }

    private Set<Artifact> getProjectDependencies() throws MojoExecutionException
    {
        final Set dependencies = new HashSet();
        try
        {
            final Set<Artifact> artifacts = project.createArtifacts(artifactFactory, null, null);
            dependencies.addAll(artifacts);
        }
        catch (InvalidDependencyVersionException e)
        {
            throw new RuntimeException(e);
        }
        final OrArtifactFilter dependencyFilter = new OrArtifactFilter();
        dependencyFilter.add(new ScopeArtifactFilter(DefaultArtifact.SCOPE_COMPILE));
        dependencyFilter.add(new ScopeArtifactFilter(DefaultArtifact.SCOPE_RUNTIME));
        return resolveTransitiveArtifacts(dependencies, dependencyFilter);

    }

    /**
     * Processes the filtered set of dependent projects to resolve parent projects in a multi module
     * set up as well as doing any scm path manipulation to fix incorrect scm information in the pom
     *
     * @param projects the projects to process
     * @return
     */
    private Set<MavenProject> processProjects(final Set<MavenProject> projects)
    {
        final Set<MavenProject> resolvedList = new HashSet<MavenProject>();
        final Set<String> seenSCMUrls = new HashSet<String>();
        for(MavenProject mavenProject : projects)
        {

            final MavenProject projectWithRepositoryMappingsApplied = findAndApplyRepositoryMappingConfig(mavenProject);
            final MavenProject projectWithSourceDepConfigApplied = findAndApplySourceDepConfig(projectWithRepositoryMappingsApplied);
            if (projectWithSourceDepConfigApplied != null)
            {
                final Scm scm = projectWithSourceDepConfigApplied.getScm();
                if (scm != null && scm.getConnection() != null)
                {
                    if (!seenSCMUrls.contains(scm.getConnection()))
                    {
                        getLog().debug("Scm is " + scm.getConnection());
                        resolvedList.add(projectWithSourceDepConfigApplied);
                        seenSCMUrls.add(scm.getConnection());
                    }
                }
                else
                {
                    getLog().info("No SCM details for " + mavenProject.getArtifact());
                }
            }
        }
        return resolvedList;
    }

    private MavenProject findAndApplyRepositoryMappingConfig(final MavenProject mavenProject)
    {
        if(repositoryMappings != null)
        {
            for (final RepositoryMapping config : repositoryMappings)
            {
                final Map<String,String> moduleMappings = config.getModuleMapping();
                final List<String> repositoryModules = config.getRepositoryModules();
                if((moduleMappings.containsKey(mavenProject.getArtifactId()) ||
                    repositoryModules.contains(mavenProject.getArtifactId())))
                {
                    return applyRepositoryMappingConfig(config, mavenProject);
                }
            }
        }
        return mavenProject;
    }

    private MavenProject applyRepositoryMappingConfig(final RepositoryMapping config, final MavenProject mavenProject)
    {
        MavenProject processedMavenProject = mavenProject;
        if(config.isResolveRoot())
        {
            processedMavenProject = resolveRootProject(mavenProject);
        }
        if(config.getBaseRepository() != null && !config.getBaseRepository().equals(""))
        {
            modifyBaseRepository(config.getBaseRepository(), processedMavenProject);
        }
        if(config.getModuleMapping() != null && !config.getModuleMapping().isEmpty())
        {
            applyModuleMapping(config.getModuleMapping(), processedMavenProject);
        }
        return processedMavenProject;

    }

    /**
     * Resolves the root, parent project in a multi module project set up
     *
     * @param mavenProject the module to start upward resolution from
     * @return
     */
    protected MavenProject resolveRootProject(final MavenProject mavenProject)
    {
        getLog().debug("Resolving root project for project" + mavenProject.getArtifactId());


        final String connection = mavenProject.getScm().getConnection();
        final SCMPathType scmPathType = SCMPathType.determinePathType(connection);

        MavenProject parentProject = mavenProject;

        if(scmPathType.equals(SCMPathType.UNKNOWN))
        {
            getLog().warn(MessageFormat.format("Unable to find trunk or tag markers on scm path during root project resolution for {0}, original scm information was {1}",
                    mavenProject.getArtifactId(), connection));
        }
        else
        {
            final List<String> pathElements = scmPathType.determinePathElements(connection);


            for (String pathElement : pathElements)
            {
                parentProject = parentProject.getParent();
            }

        }

        if(!mavenProject.getArtifactId().equals(parentProject.getArtifactId()))
        {
            getLog().info(MessageFormat.format("Resolved root project for {0} to {1}", mavenProject.getArtifactId(),
                    parentProject.getArtifactId()));
        }

        return parentProject;
    }

    /**
     * Determines if a project has a sourceDependency configuration specified and
     * applies the configuration if so
     *
     * @param mavenProject
     * @return
     */
    private MavenProject findAndApplySourceDepConfig(final MavenProject mavenProject)
    {
        if(sourceDependencies != null)
        {
            for (final SourceDependency config : sourceDependencies)
            {
                if((config.getArtifactId().equals(mavenProject.getArtifactId())))
                {
                    return applySourceDepConfig(config, mavenProject);
                }
            }
        }

        final MavenProject rootProject;
        if(resolveRoot)
        {
            rootProject = resolveRootProject(mavenProject);
        }
        else
        {
            rootProject = mavenProject;
        }

        return rootProject;
    }

    /**
     * Applies a source dependency configuration
     *
     * @param config
     * @param mavenProject
     * @return
     */
    private MavenProject applySourceDepConfig(final SourceDependency config, final MavenProject mavenProject)
    {
        MavenProject processedMavenProject = mavenProject;
        if(config.isResolveRoot())
        {
            processedMavenProject = resolveRootProject(mavenProject);
        }
        if(config.getBaseRepository() != null && !config.getBaseRepository().equals(""))
        {
            modifyBaseRepository(config.getBaseRepository(), processedMavenProject);
        }
        if(config.getModuleMapping() != null && !config.getModuleMapping().isEmpty())
        {
            applyModuleMapping(config.getModuleMapping(), processedMavenProject);
        }
        return processedMavenProject;
    }

    /**
     * Applies a module mapping for sub modules with paths different from their artifactId. When
     * maven does scm path resolution for sub modules, it implicitly assumes the sub module's scm
     * path follows its artifactId. Specify moduleMappings in a sourceDependency configuration should
     * this be the case.
     *
     * @param moduleMapping
     * @param mavenProject
     */
    protected void applyModuleMapping(final Map<String, String> moduleMapping, final MavenProject mavenProject)
    {
        final String connection = mavenProject.getScm().getConnection();
        String newConnection = connection;
        for(Map.Entry<String, String> mapping : moduleMapping.entrySet())
        {
            newConnection = newConnection.replace("/"+ mapping.getKey(), "/" + mapping.getValue());
        }
        getLog().info(MessageFormat.format("Module mapping found for {0}, replacing {1} with {2}",
                mavenProject.getArtifactId(), connection, newConnection));
        mavenProject.getScm().setConnection(newConnection);
    }

    /**
     * Modifies the base repository from which scm information is calculated relative to.
     *
     * @param baseRepository
     * @param mavenProject
     */
    private void modifyBaseRepository(final String baseRepository, final MavenProject mavenProject)
    {
        final String connection = mavenProject.getScm().getConnection();

        final SCMPathType scmPathType = SCMPathType.determinePathType(connection);

        if(scmPathType.equals(SCMPathType.UNKNOWN))
        {
            getLog().warn(MessageFormat.format("Unable to find trunk or tag markers on scm path during baseRepository replacement for {0}, using project as is",
                    mavenProject.getArtifactId()));
        }
        else
        {
            final String pathToSubmodule = scmPathType.getPathToSubModule(connection);
            mavenProject.getScm().setConnection(baseRepository + scmPathType.getMarker().substring(1) + "/" + pathToSubmodule);
        }
    }


    private Set getReactorProjectDependencies() throws MojoExecutionException
    {
        Set dependencies = new HashSet();
        for (MavenProject mavenProject : reactorProjects)
        {
            Set projectDependencies = mavenProject.getDependencyArtifacts();
            if (projectDependencies != null && !projectDependencies.isEmpty())
            {
                dependencies.addAll(projectDependencies);
            }
        }
        return resolveTransitiveArtifacts(dependencies);
    }

    private Set<MavenProject> getProjectsForArtifacts(Set<Artifact> artifacts) throws MojoExecutionException
    {
        Set<MavenProject> projects = new HashSet<MavenProject>();
        for (Artifact artifact : artifacts)
        {
            MavenProject currentProject = getProjectForArtifact(artifact);
            if (artifact.getGroupId().startsWith(groupIdMask))
            {
                projects.add(currentProject);
            }
        }
        return projects;
    }

    private void removeExcludedArtifacts(Set<MavenProject> projects)
    	throws MojoExecutionException
    {
        List<String> exclusionPairs = getExclusions();
        List<MavenProject> projectList = new ArrayList(projects);
        for (MavenProject currentProject : projectList)
        {
            if (currentProject == null)
            {
                getLog().info("Project in project list is null");
                continue;
            }

            if (matchesExclusion(currentProject, exclusionPairs))
            {
                projects.remove(currentProject);
            }
        }
    }

    private void removeExcludedDependencies(Set<Artifact> artifacts)
            throws MojoExecutionException
    {
        Set<String> exclusions = new HashSet<String>(getDependencyExclusions());

        List<Artifact> artifactList = new ArrayList(artifacts);
        for (Artifact currentArtifact : artifactList)
        {
            if (currentArtifact == null)
            {
                getLog().info("Project in project list is null");
                continue;
            }

            final String dependencyId = currentArtifact.getGroupId() + ":" + currentArtifact.getArtifactId();
            if (exclusions.contains(dependencyId))
            {
                getLog().info("Excluding " + dependencyId);
                artifacts.remove(currentArtifact);
            }
        }
    }

    private void removeReactorProjects(Set<MavenProject> projects)
    {
        for (MavenProject project : reactorProjects.subList(1, reactorProjects.size()))
        {
            if (projects.contains(project))
            {
                if (!projects.remove(project))
                {
                    getLog().warn("Could not remove reactor project from checkout list for '" + project.getGroupId() + ":" + project.getArtifactId() + "'");
                }
            }
        }
    }

    private List<String> getExclusions()
    {
        List<String> excludes = new ArrayList<String>();
        if (exclusions != null)
        {
            excludes.addAll(exclusions);
        }
        List<MavenProject> reactorExclusions = reactorProjects.subList(1, reactorProjects.size());
        for (MavenProject mavenProject : reactorExclusions)
        {
            excludes.add(mavenProject.getGroupId() + ":" + mavenProject.getArtifactId());
        }
        return excludes;
    }

    private List<String> getDependencyExclusions()
    {
        return dependencyExclusions;
    }

    private boolean matchesExclusion(MavenProject project, List<String> exclusionPairs)
        throws MojoExecutionException
    {
        for (String exclusion : exclusionPairs)
        {
            String[] parts = StringUtils.split(exclusion, ':');
            if (parts.length == 2)
            {
                String currentGroupId = project.getGroupId();
                String currentArtifactId = project.getArtifactId();

                if (parts[0].equals(currentGroupId) && parts[1].equals(currentArtifactId))
                {
                    return true;
                }
            }
            else
            {
                throw new MojoExecutionException("Exclusion is not in the format of groupId:artifactId. Value:" + exclusion);
            }
        }
    	return false;
    }

    /*
     * Resolves transitive dependencies for a set of given artifacts
     * Returns a Set of all artifacts resolved
     */
    private Set resolveTransitiveArtifacts(Set artifacts, ArtifactFilter filter)
    	throws MojoExecutionException
    {
        try
        {
            ArtifactResolutionResult result = resolver.resolveTransitively(artifacts, project.getArtifact(),
                localRepository, remoteRepositories, metadataSource, filter);
            return result.getArtifacts();
        }
        catch (ArtifactResolutionException e)
        {
            throw new MojoExecutionException(e.getMessage(), e);
        }
        catch (ArtifactNotFoundException e)
        {
            throw new MojoExecutionException(e.getMessage(), e);
        }
    }

    private Set resolveTransitiveArtifacts(Set artifacts) throws MojoExecutionException
    {
        return resolveTransitiveArtifacts(artifacts, null);
    }


    /*
     * Returns the MavenProject for a given Artifact
     */
    private MavenProject getProjectForArtifact(Artifact artifact)
    	throws MojoExecutionException
    {
        try
        {
            return projectBuilder.buildFromRepository(artifact, remoteRepositories, localRepository);
        }
        catch (InvalidProjectModelException e)
        {
            getLog().error("Validation Errors: " + e.getValidationResult().getMessages());
            throw new MojoExecutionException(e.getMessage(), e);
        }
        catch (ProjectBuildingException e)
        {
            throw new MojoExecutionException(e.getMessage(), e);
        }
    }
    
}
