package com.atlassian.maven.plugins;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.dom4j.Document;
import org.dom4j.Element;
import org.codehaus.plexus.util.FileUtils;

import java.io.File;
import java.io.IOException;

/**
 * @author Atlassian
 * @goal orion
 * @execute phase="generate-sources"
 */
public class Orion
        extends AbstractIdeaMojo
{

    public void execute() throws MojoExecutionException, MojoFailureException
    {
        try
        {
            doDependencyResolution(executedProject, localRepo);
        }
        catch (Exception e)
        {
            throw new MojoExecutionException("Unable to build project dependencies.", e);
        }

        build();
    }

    public void build()
            throws MojoExecutionException
    {
        initAtlassianVariables();

        if (executedProject.isExecutionRoot())
        {

            String orionLocation = atlassianProperties.getProperty("atlassian.idea.orion.location");
            if (orionLocation != null)
            {
                File orion = new File(orionLocation + "/orion.jar");

                if (orion.exists())
                {
                    modifyIpr("$PROJECT_DIR$/modules/orion_env/orion_env.iml");
                    modifyIws();
                    generateModulesForApps();
                    createOrionXml();
                } else
                {
                    getLog().error("Unable to find orion jar in " + orion.getAbsolutePath());
                }
            }
        }
    }


    public void modifyIws() throws MojoExecutionException
    {
        File workspaceFile = new File(executedProject.getBasedir(), executedProject.getArtifactId() + ".iws");
        Document document = readXmlDocument(workspaceFile);

        Element module = document.getRootElement();
        Element runManager = findComponent(module, "RunManager");
        Element conf = findElement(runManager, "configuration", "Orion")
                .addAttribute("type", "Application")
                .addAttribute("default", "false")
                .addAttribute("selected", "false")
                .addAttribute("factoryName", "Application");
        String mainClassName = "com.evermind.server.ApplicationServer";
        String vmParameters = "-Xdebug -Xnoagent -Djava.compiler=NONE -Xrunjdwp:transport=dt_socket,server=y,suspend=n,address=5005 " + appParams;
        String programParameters = "-userThreads";
        String workingDirectory = "file://$PROJECT_DIR$/orion";
        String moduleName = "orion_env";
        createOptions(conf, mainClassName, vmParameters, programParameters, workingDirectory, moduleName);

        writeXmlDocument(workspaceFile, document);


    }


    void generateModulesForApps() throws MojoExecutionException
    {
        getLog().info("Creating Orion Environment...");
        File modulePath = new File(executedProject.getBasedir(), "/modules/orion_env");
        File resourceFile = new File(modulePath, "orion_env-module.xml");
        File imlFile = new File(modulePath, "orion_env.iml");
        modulePath.mkdirs();
        modifyEnvModule("atlassian.idea.orion.location", resourceFile, imlFile);
    }


    private void createOrionXml()
    {
        File baseDir = executedProject.getBasedir();
        File orionDir = new File(baseDir, "/orion/persistence/");
        orionDir.mkdirs();
        orionDir = new File(baseDir, "/orion/config/");
        orionDir.mkdirs();
        //getLog().info("Creating Orion xml files ...");

        try
        {
            /* Resource "orion-data-sources.xml"*/
            String resourceName = "orion-data-sources.xml";
            File orionDataSourceXml = new File(orionDir, "/data-sources.xml");

            Document document = readXmlDocument(resourceName);

            Element root = document.getRootElement();
            dataSource(root);
            writeXmlDocument(orionDataSourceXml, document);

            /* Resource "orion-default-web-site.xml"*/
            resourceName = "orion-default-web-site.xml";
            orionDataSourceXml = new File(orionDir, "/default-web-site.xml");

            document = readXmlDocument(resourceName);

            root = document.getRootElement();
            root.addAttribute("port", appPort);
            if (atlassianProperties.containsKey("atlassian.idea.application.webapp.contextpath"))
            {
                String contextPath = "/" + atlassianProperties.getProperty("atlassian.idea.application.webapp.contextpath");
                findElement(root, "web-app", "application", "default")
                        .addAttribute("name", "webapp")
                        .addAttribute("root", contextPath)
                        .addAttribute("load-on-startup", "true");

                findElement(root, "default-web-app", "application", "default")
                        .addAttribute("name", "");
            } else
            {
                findElement(root, "default-web-app", "application", "default")
                        .addAttribute("name", "webapp");
            }

            writeXmlDocument(orionDataSourceXml, document);

            /* Resource "orion-application.xml"*/
            resourceName = "orion-application.xml";
            orionDataSourceXml = new File(orionDir, "/application.xml");

            document = readXmlDocument(resourceName);

            root = document.getRootElement();
            String mavenWarSrc = atlassianProperties.getProperty("maven.war.src");
            if (atlassianProperties.getProperty("maven.war.src") != null)
            {
                root.element("web-module").addAttribute("path", mavenWarSrc);
            } else
            {
                throw new MojoExecutionException("maven.war.src variable not set\n"
                    + "Please set maven.war.src variable your webapp directory (relative to the project root)");
            }
            writeXmlDocument(orionDataSourceXml, document);

            /*Copy orion-server.xml file*/
            String name = "/default/orion-server.xml";
            orionDataSourceXml = new File(orionDir, "/server.xml");
            FileUtils.copyURLToFile(getClass().getResource(name), orionDataSourceXml);

            /*Copy orion-global-web-application.xml file*/
            name = "/default/orion-global-web-application.xml";
            orionDataSourceXml = new File(orionDir, "/global-web-application.xml");
            FileUtils.copyURLToFile(getClass().getResource(name), orionDataSourceXml);

            /*Copy orion-mime.types file*/
            name = "/default/orion-mime.types";
            orionDataSourceXml = new File(orionDir, "/mime.types");
            FileUtils.copyURLToFile(getClass().getResource(name), orionDataSourceXml);

        } catch (IOException e)
        {
            e.printStackTrace();  //To change body of catch statement use File | Settings | File Templates.
        } catch (MojoExecutionException e)
        {
            e.printStackTrace();  //To change body of catch statement use File | Settings | File Templates.
        }

    }

    private void dataSource(Element root)
    {
        if (atlassianProperties.containsKey("atlassian.idea.application.db.jndiname"))
        {
            String jndiName = atlassianProperties.getProperty("atlassian.idea.application.db.jndiname");
            String driverName = atlassianProperties.getProperty("atlassian.idea.application.db.drivername");
            String url = atlassianProperties.getProperty("atlassian.idea.application.db.url");
            String username = atlassianProperties.getProperty("atlassian.idea.application.db.username");
            String password = atlassianProperties.getProperty("atlassian.idea.application.db.password");

            findElement(root, "data-source")
                    .addAttribute("location", jndiName)
                    .addAttribute("username", username)
                    .addAttribute("password", password)
                    .addAttribute("connection-driver", driverName)
                    .addAttribute("url", url)
                    .addAttribute("inactivity-timeout", "30")
                    .addAttribute("class", "com.evermind.sql.DriverManagerDataSource")
                    .addAttribute("name", "database");

        }
    }


}
