package com.atlassian.maven.plugins;

/*
 * Copyright 2005-2006 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import org.apache.maven.model.Resource;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;
import org.apache.maven.artifact.factory.ArtifactFactory;
import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.artifact.resolver.ArtifactResolver;
import org.apache.maven.artifact.metadata.ArtifactMetadataSource;
import org.dom4j.Document;
import org.dom4j.Element;

import java.io.File;
import java.io.IOException;
import java.util.*;


/**
 * @author Edwin Punzalan
 * @goal module
 * @execute phase="generate-sources"
 */
public class IdeaModuleMojo
        extends AbstractIdeaMojo
{

    /**
     * @parameter expression="${atlassian.idea.includeAcceptanceTests}"
     */
    private String includeAcceptanceTests;

    /**
     * Create IDEA (.iml) project files.
     *
     * @throws org.apache.maven.plugin.MojoExecutionException
     *
     */
    public void execute()
            throws MojoExecutionException
    {
        try
        {
            doDependencyResolution(executedProject, localRepo);
        }
        catch (Exception e)
        {
            throw new MojoExecutionException("Unable to build project dependencies.", e);
        }
        rewriteModule();
    }

    public void initParam(MavenProject project, ArtifactFactory artifactFactory, ArtifactRepository localRepo, ArtifactResolver artifactResolver, ArtifactMetadataSource artifactMetadataSource, Log log, boolean overwrite, String includeAcceptanceTests)
    {
        super.initParam(project, artifactFactory, localRepo, artifactResolver, artifactMetadataSource, log, overwrite);    //To change body of overridden methods use File | Settings | File Templates.
        this.includeAcceptanceTests = includeAcceptanceTests;
    }

    public void rewriteModule()
            throws MojoExecutionException
    {
        initAtlassianVariables();
        File moduleFile = new File(executedProject.getBasedir(), executedProject.getArtifactId() + ".iml");

        Document document = readXmlDocument(moduleFile);

        Element module = document.getRootElement();


        Element component = findComponent(module, "NewModuleRootManager");

        Element content = findElement(component, "content");

        addWebAppModules(module);
        removeResourcesFromSources(content);

        /** add acceptance test as test **/
        if (executedProject.getArtifactId().equals("confluence"))
        {
            if ("true".equals(includeAcceptanceTests))
            {
                Element acceptance = findElement(content, "sourceFolder", "url", "file://$MODULE_DIR$/src/test/acceptance");
                acceptance.addAttribute("isTestSource", "true");
            }
        }

        /** Add the sources of the Atlassian Dependencies if they exist **/
        if (executedProject.getArtifactId().equals("confluence-project"))
        {
            setupClassLibrarySources(component);
        }

        /** set deploymentDescriptor optional=true **/
        String appserverModule = atlassianProperties.getProperty("appserver.module.dependency");
        if (executedProject.getArtifactId().equals(appserverModule))
        {
            Element webModule = findComponent(module, "WebModuleProperties");
            if (webModule != null)
            {
                Element deploymentDesc = findElement(webModule, "deploymentDescriptor");
                deploymentDesc.addAttribute("optional", "true");
            }
        }

        createBuildResource(component);
        createTestResource(component);
        createServerExcludes(content);
        setupGlobalLibraries(component);
        moveSourceFolderToLast(component);


        writeXmlDocument(moduleFile, document);

    }

    private void moveSourceFolderToLast(Element component)
    {
        for (Iterator children = component.elementIterator("orderEntry"); children.hasNext();)
        {
            Element orderEntry = (Element) children.next();

            String type = orderEntry.attributeValue("type");
            if ("sourceFolder".equals(type))
            {
                /* This is not the best way to move an element to the end... but it works*/
                Element newOrderEntry = (Element) orderEntry.clone();
                component.add(newOrderEntry); // adds it to the end
                component.remove(orderEntry); // removes it from the start
            }
        }
    }

    private void removeResourcesFromSources(Element content)
    {
        for (Iterator i = executedProject.getBuild().getResources().iterator(); i.hasNext();)
        {
            Resource resource = (Resource) i.next();
            String directory = resource.getDirectory();
            if (resource.getTargetPath() == null && !resource.isFiltering())
            {
                String relativeDir = getModuleFileUrl(directory);
                deleteSourceFolder(content, relativeDir, false);
            } else
            {
                getLog().info(
                        "Not adding resource directory as it has an incompatible target path or filtering: " +
                                directory);
            }
        }

        for (Iterator i = executedProject.getBuild().getTestResources().iterator(); i.hasNext();)
        {
            Resource resource = (Resource) i.next();
            String directory = resource.getDirectory();
            if (resource.getTargetPath() == null && !resource.isFiltering())
            {
                String relativeDir = getModuleFileUrl(directory);
                deleteSourceFolder(content, relativeDir, true);
            } else
            {
                getLog().info(
                        "Not adding test resource directory as it has an incompatible target path or filtering: " +
                                directory);
            }
        }
    }

    private void setupClassLibrarySources(Element root) throws MojoExecutionException
    {
        getLog().debug("**Adding Sources** ");
        String source = atlassianProperties.getProperty("sourceXML.location");
        String moduleRelativeBaseDir = atlassianProperties.getProperty("module.relative.base.dir");
        if (source != null && moduleRelativeBaseDir != null)
        {
            File sourcePath = new File(executedProject.getBasedir(), source);
            SourceIncludeConfiguration sic = new SourceIncludeConfiguration(log, sourcePath);
            LinkedList sources = sic.getSourceInclude();


            for (int i = 0; i < sources.size(); i++)
            {
                File path = null;
                SourceInclude sourceObject = (SourceInclude) sources.get(i);
                if (sourceObject.getSource() != null)
                {
                    String moduleDir = sourceObject.getSource();
                    String modulePath = executedProject.getBasedir() + "/"
                            + moduleRelativeBaseDir + "/"
                            + moduleDir;
                    String pathNameM1 = modulePath + "/src/java";
                    String pathNameM2 = modulePath + "/src/main/java";
                    File pathM1 = new File(pathNameM1);
                    getLog().debug("Path to Dependency Source Module is: " + modulePath);
                    File pathM2 = new File(pathNameM2);
                    if (pathM1.exists())
                    {
                        path = pathM1;
                    } else if (pathM2.exists())
                    {
                        path = pathM2;
                    }
                    if (path != null)
                    {
                        getLog().debug("Path to Source of Dependency is:" + path.getAbsolutePath());
                        for (Iterator children = root.elementIterator("orderEntry"); children.hasNext();)
                        {
                            Element orderEntry = (Element) children.next();
                            String type = orderEntry.attributeValue("type");
                            if ("module-library".equals(type))
                            {
                                Element lib = orderEntry.element("library");
                                String name = lib.attributeValue("name");
                                if (sourceObject.getArtifactId().equals(name))
                                {
                                    Element sourceTag = lib.element("SOURCES");
                                    if (sourceTag != null)
                                    {
                                        lib.remove(sourceTag);
                                    }
                                    sourceTag = findElement(lib, "SOURCES");

                                    try
                                    {
                                        findElement(sourceTag, "root", "url", invertSlashes("file://" + path.getCanonicalPath()));
                                        getLog().debug("Adding Source of Dependency: " + path.getCanonicalPath());
                                    } catch (IOException e)
                                    {
                                        throw new MojoExecutionException("Path is malformed", e);
                                    }

                                }
                            }

                        }
                    }
                }
            }
        }
    }

    private void setupGlobalLibraries(Element node) throws MojoExecutionException
    {
        if (atlassianProperties.getProperty("atlassian.idea.global.libraries.to.enable") != null)
        {
            String globalLibraries = atlassianProperties.getProperty("atlassian.idea.global.libraries.to.enable");
            try
            {
                String[] globalLibrariesArray = globalLibraries.split(",");
                for (int i = 0; i < globalLibrariesArray.length; i++)
                {
                    Element orderEntry = findElement(node, "orderEntry", "name", globalLibrariesArray[i]);
                    orderEntry.addAttribute("type", "library");
                    orderEntry.addAttribute("level", "application");
                }
            } catch (Exception e)
            {
                throw new MojoExecutionException("Cannot add libraries", e);  //To change body of catch statement use File | Settings | File Templates.
            }
        }
    }

    private void createServerExcludes(Element content)
    {
        String baseDir = executedProject.getBasedir().toString();
        if (executedProject.isExecutionRoot())
        {
            createExclude(content, "/release");
            createExclude(content, "/resin/cache");
            createExclude(content, "/resin/database");
            createExclude(content, "/resin/logs");
            createExclude(content, "/resin/tmp");
            createExclude(content, "/resin/work");
            createExclude(content, "/resin3/logs");
            createExclude(content, "/resin3/tmp");
            createExclude(content, "/resin3/webapps");
            createExclude(content, "/resin3/WEB-INF");
            createExclude(content, "/resin3/work");
            createExclude(content, "/tomcat/logs");
            createExclude(content, "/tomcat/work");
            createExclude(content, "/orion/application-deployments");
            createExclude(content, "/orion/log");
            createExclude(content, "/orion/persistence");
            if (executedProject.getArtifactId().indexOf("confluence") >= 0)
            {
                String relPath;
                relPath = getModuleFileUrl(baseDir + "/conf-sourcerelease/target");
                findElement(content, "excludeFolder", "url", relPath);
                relPath = getModuleFileUrl(baseDir + "/conf-sourcerelease/release");
                findElement(content, "excludeFolder", "url", relPath);
                relPath = getModuleFileUrl(baseDir + "/conf-acceptance-test/target");
                findElement(content, "excludeFolder", "url", relPath);
                relPath = getModuleFileUrl(baseDir + "/conf-standalone/target");
                findElement(content, "excludeFolder", "url", relPath);
                relPath = getModuleFileUrl(baseDir + "/conf-standalone/release");
                findElement(content, "excludeFolder", "url", relPath);
                relPath = getModuleFileUrl(baseDir + "/conf-war/release");
                findElement(content, "excludeFolder", "url", relPath);
                relPath = getModuleFileUrl(baseDir + "/conf-war/target");
                findElement(content, "excludeFolder", "url", relPath);
                relPath = getModuleFileUrl(baseDir + "/conf-webapp/target");
                findElement(content, "excludeFolder", "url", relPath);
            }
        }
        if (executedProject.getArtifactId().equals("confluence-acceptance-test"))
        {
            createExclude(content, "/target/confluence-home");
            createExclude(content, "/target/confluence-home-node1");
            createExclude(content, "/target/confluence-home-node2");
            createExclude(content, "/target/resin2x");
            createExclude(content, "/target/tomcat5x");
            createExclude(content, "/target/tomcat5x-node1");
            createExclude(content, "/target/tomcat5x-node2");
        }
    }

    private void createExclude(Element content, String dir)
    {
        findElement(content, "excludeFolder", "url", "file://$MODULE_DIR$" + dir);
    }

    private void createTestResource(Element component)
    {
        Element lib = createResource(component, "Test Resources");
        Element classesChild = lib.element("CLASSES");
        if (classesChild == null)
        {
            classesChild = createElement(lib, "CLASSES");
        }

        /* remove all root url elements */
        removeOldElements(classesChild, "root");

        /* This section generates the source URL from the test resources in POM*/
        for (Iterator i = executedProject.getBuild().getTestResources().iterator(); i.hasNext();)
        {
            Resource resource = (Resource) i.next();
            String directory = resource.getDirectory();
            if (resource.getTargetPath() == null && !resource.isFiltering())
            {
                String relativeDir = getModuleFileUrl(directory);
                findElement(classesChild, "root", "url", relativeDir);
            }
        }
    }

    private Element createResource(Element component, String resourceName)
    {

        boolean resourceFound = false;
        Element lib = null;
        for (Iterator children = component.elementIterator("orderEntry"); children.hasNext();)
        {
            Element orderEntry = (Element) children.next();

            String type = orderEntry.attributeValue("type");
            if ("module-library".equals(type))
            {
                lib = orderEntry.element("library");
                String name = lib.attributeValue("name");
                if (resourceName.equals(name))
                {
                    resourceFound = true;
                    break;
                }
            }
        }
        if (!resourceFound)
        {
            Element orderEntry = createElement(component, "orderEntry")
                    .addAttribute("type", "module-library")
                    .addAttribute("exported", "");

            lib = createElement(orderEntry, "library")
                    .addAttribute("name", resourceName);

        }
        return lib;
    }

    private void createBuildResource(Element component)
    {
        Element lib = createResource(component, "Build Resources");
        Element classesChild = lib.element("CLASSES");
        if (classesChild == null)
        {
            classesChild = createElement(lib, "CLASSES");
        }

        /* remove all root url elements */
        removeOldElements(classesChild, "root");

        /* This section generates the source URL from the resources in POM*/
        for (Iterator i = executedProject.getBuild().getResources().iterator(); i.hasNext();)
        {
            Resource resource = (Resource) i.next();
            String directory = resource.getDirectory();
            if (resource.getTargetPath() == null && !resource.isFiltering())
            {
                String relativeDir = getModuleFileUrl(directory);
                findElement(classesChild, "root", "url", relativeDir);
            }
        }

        /* This section generates the source URL from the resources in the extraResourcesVariable*/
        String extraResources = atlassianProperties.getProperty("atlassian.idea.extra.resources");
        if (extraResources != null)
        {
            StringTokenizer extraResourceTokens = new StringTokenizer(extraResources, ",");
            while (extraResourceTokens.hasMoreTokens())
            {
                String extraResourceDir = extraResourceTokens.nextToken();
                findElement(classesChild, "root", "url", extraResourceDir);
            }
        }
    }

    private void deleteSourceFolder
            (Element
                    content, String
                    directory, boolean b)
    {
        for (Iterator children = content.elementIterator(); children.hasNext();)
        {
            Element child = (Element) children.next();
            if ("sourceFolder".equals(child.getName()))
            {
                Boolean bool = new Boolean(b);
                String test = (bool).toString();
                if (directory.equals(child.attributeValue("url")) && test.equals(child.attributeValue("isTestSource")))
                {
                    content.remove(child);
                }

            }
        }
    }


    /**
     * Translate the relative path of the file into module path
     *
     * @param basedir File to use as basedir
     * @param path    Absolute path string to translate to ModuleFileUrl
     * @return moduleFileUrl Translated Module File URL
     */
    private String getModuleFileUrl
            (File
                    basedir, String
                    path)
    {
        return "file://$MODULE_DIR$/" + toRelative(basedir, path);
    }

    private String getModuleFileUrl
            (String
                    file)
    {
        return getModuleFileUrl(executedProject.getBasedir(), file);
    }


}