package com.atlassian.maven.plugins;

/*
 * Copyright 2005-2006 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.factory.ArtifactFactory;
import org.apache.maven.artifact.metadata.ArtifactMetadataSource;
import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.artifact.resolver.ArtifactNotFoundException;
import org.apache.maven.artifact.resolver.ArtifactResolutionException;
import org.apache.maven.artifact.resolver.ArtifactResolutionResult;
import org.apache.maven.artifact.resolver.ArtifactResolver;
import org.apache.maven.artifact.resolver.filter.ArtifactFilter;
import org.apache.maven.artifact.resolver.filter.ExcludesArtifactFilter;
import org.apache.maven.artifact.versioning.InvalidVersionSpecificationException;
import org.apache.maven.artifact.versioning.VersionRange;
import org.apache.maven.model.Dependency;
import org.apache.maven.model.DependencyManagement;
import org.apache.maven.model.Exclusion;
import org.apache.maven.model.Plugin;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.idea.IdeaXmlWriter;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;
import org.apache.maven.project.ProjectBuildingException;
import org.apache.maven.project.artifact.InvalidDependencyVersionException;
import org.codehaus.plexus.util.StringUtils;
import org.codehaus.plexus.util.cli.Commandline;
import org.codehaus.plexus.util.cli.StreamConsumer;
import org.codehaus.plexus.util.cli.CommandLineUtils;
import org.codehaus.plexus.util.cli.CommandLineException;
import org.codehaus.plexus.util.xml.Xpp3Dom;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.Attribute;
import org.dom4j.io.SAXReader;
import org.dom4j.io.XMLWriter;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.io.*;
import java.util.*;

/**
 * @author Edwin Punzalan
 */
public abstract class AbstractIdeaMojo
        extends AbstractMojo
{
    /**
     * The Maven Project.
     *
     * @parameter expression="${executedProject}"
     * @required
     * @readonly
     */
    protected MavenProject executedProject;

    /* holder for the log object only */
    protected Log log;

    /**
     * Whether to update the existing project files or overwrite them.
     *
     * @parameter expression="${overwrite}" default-value="false"
     */
    protected boolean overwrite;

    /**
     * @parameter expression="${component.org.apache.maven.artifact.factory.ArtifactFactory}"
     * @required
     * @readonly
     */
    protected ArtifactFactory artifactFactory;


    /**
     * @parameter expression="${localRepository}"
     * @required
     * @readonly
     */
    protected ArtifactRepository localRepo;

    /**
     * @parameter expression="${component.org.apache.maven.artifact.resolver.ArtifactResolver}"
     */
    protected ArtifactResolver artifactResolver;

    /**
     * @component role="org.apache.maven.artifact.metadata.ArtifactMetadataSource" hint="maven"
     */
    protected ArtifactMetadataSource artifactMetadataSource;
    protected String appParams;
    protected String appPort;
    protected Properties atlassianProperties;
    private String CONFLUENCE_APPSERVER_MODULE = "confluence-webapp";
    protected static boolean ideaHasBeenRun = false;

    public void initParam(MavenProject project, ArtifactFactory artifactFactory, ArtifactRepository localRepo,
                          ArtifactResolver artifactResolver, ArtifactMetadataSource artifactMetadataSource, Log log,
                          boolean overwrite)
    {
        this.executedProject = project;

        this.log = log;

        this.artifactFactory = artifactFactory;

        this.localRepo = localRepo;

        this.artifactResolver = artifactResolver;

        this.artifactMetadataSource = artifactMetadataSource;

        this.overwrite = overwrite;


    }

    protected Document readXmlDocument(File file) throws MojoExecutionException
    {
        if (!ideaHasBeenRun)
        {
            callMavenIdea();
        }
        try
        {
            SAXReader reader = new SAXReader();
            return reader.read(file);
        } catch (DocumentException e)
        {
            throw new MojoExecutionException("Please Run   $ mvn idea:idea    before this.", e);  //To change body of catch statement use File | Settings | File Templates.
        }
    }


    protected Document readXmlDocument(String fileName) throws MojoExecutionException
    {
        try
        {
            SAXReader reader = new SAXReader();
            String name = "/default/" + fileName;
            InputStream resourceAsStream = getClass().getResourceAsStream(name);
            return reader.read(resourceAsStream);
        } catch (DocumentException e)
        {
            throw new MojoExecutionException("Unable to read file", e);  //To change body of catch statement use File | Settings | File Templates.
        }
    }

    protected void writeXmlDocument(File file, Document document) throws MojoExecutionException
    {
        try
        {
            XMLWriter writer = new IdeaXmlWriter(file);
            writer.write(document);
            writer.close();
        } catch (IOException e)
        {
            throw new MojoExecutionException("Unable to write file", e);  //To change body of catch statement use File | Settings | File Templates.
        }
    }

    /**
     * Finds element from the module element.
     *
     * @param module Xpp3Dom element
     * @param name   Name attribute to find
     * @return component  Returns the Xpp3Dom element found.
     */
    protected Element findComponent(Element module, String name)
    {
        return findElement(module, "component", name);
    }

    /**
     * Find a module with the path parameter passed to it
     * (used for *.ipr file)
     */
    protected Element findModule(Element module, String path)
    {
        return findElement(module, "module", "filepath", path);
    }

    /**
     * Find an element with the "elementName" and "attributeName", "attributeValue"
     * if it doesnt exist create it.
     */
    protected Element findElement(Element element, String elementName, String attributeName, String attributeValue)
    {
        for (Iterator children = element.elementIterator(elementName); children.hasNext();)
        {
            Element child = (Element) children.next();
            if (attributeValue.equals(child.attributeValue(attributeName)))
            {
                return child;
            }
        }
        return createElement(element, elementName).addAttribute(attributeName, attributeValue);
    }

    protected Element findElement(Element element, String elementName, String attributeName)
    {
        for (Iterator children = element.elementIterator(elementName); children.hasNext();)
        {
            Element child = (Element) children.next();
            if (attributeName.equals(child.attributeValue("name")))
            {
                return child;
            }
        }
        return createElement(element, elementName).addAttribute("name", attributeName);
    }

    protected void removeElement(Element element, String elementName, String attributeName)
    {
        for (Iterator children = element.elementIterator(elementName); children.hasNext();)
        {
            Element child = (Element) children.next();
            if (attributeName.equals(child.attributeValue("name")))
            {
                element.remove(child);
                return;
            }
        }
    }

    protected void removeElement(Element element, String elementName, String attributeName, String attributeValue)
    {
        for (Iterator children = element.elementIterator(elementName); children.hasNext();)
        {
            Element child = (Element) children.next();
            for (Iterator attributes = child.attributeIterator(); attributes.hasNext();)
            {
                Attribute attribute = (Attribute) attributes.next();
                if (attributeName.equals(attribute.getName()) && attributeValue.equals(attribute.getValue()))
                {
                    element.remove(child);
                }
            }
        }
    }

    protected Element findElement(Element component, String name)
    {
        Element element = component.element(name);
        if (element == null)
        {
            element = createElement(component, name);
        }
        return element;
    }

    protected Element findElementWithAttribute(Element element, String elementName, String attributeName)
    {
        for (Iterator children = element.elementIterator(elementName); children.hasNext();)
        {
            Element child = (Element) children.next();
            Iterator attributes = child.attributeIterator();
            while (attributes.hasNext())
            {
                Attribute attribute = (Attribute) attributes.next();
                if (attributeName.equals(attribute.getName()))
                {
                    return child;
                }

            }

        }
        return createElement(element, elementName);
    }

    /**
     * Creates an Xpp3Dom element.
     *
     * @param module Xpp3Dom element
     * @param name   Name of the element
     * @return component Xpp3Dom element
     */
    protected Element createElement(Element module, String name)
    {
        return module.addElement(name);
    }

    /**
     * Translate the absolutePath into its relative path.
     *
     * @param basedir      The basedir of the project.
     * @param absolutePath The absolute path that must be translated to relative path.
     * @return relative  Relative path of the parameter absolute path.
     */
    protected String toRelative(File basedir, String absolutePath)
    {
        String relative;

        if (absolutePath.startsWith(basedir.getAbsolutePath()))
        {
            relative = absolutePath.substring(basedir.getAbsolutePath().length() + 1);
        } else
        {
            relative = absolutePath;
        }

        relative = StringUtils.replace(relative, "\\", "/");

        return relative;
    }

    /**
     * Remove elements from content (Xpp3Dom).
     *
     * @param content Xpp3Dom element
     * @param name    Name of the element to be removed
     */
    protected void removeOldElements(Element content, String name)
    {
        for (Iterator children = content.elementIterator(); children.hasNext();)
        {
            Element child = (Element) children.next();
            if (name.equals(child.getName()))
            {
                content.remove(child);
            }
        }
    }

    /**
     * modifies backslashes to forwardslashes (may be an inherited bug from Java.lang.String.replaceFirst())
     */
    protected String invertSlashes(String s)
    {
        if (s != null)
        {
            return s.replace('\\', '/');
        } else
        {
            return null;
        }
    }

    protected void doDependencyResolution(MavenProject project, ArtifactRepository localRepo)
            throws InvalidDependencyVersionException, ProjectBuildingException
    {
        Map managedVersions =
                createManagedVersionMap(artifactFactory, project.getId(), project.getDependencyManagement());

        try
        {
            ArtifactResolutionResult result = artifactResolver.resolveTransitively(getProjectArtifacts(),
                    project.getArtifact(),
                    managedVersions, localRepo,
                    project.getRemoteArtifactRepositories(),
                    artifactMetadataSource);

            project.setArtifacts(result.getArtifacts());
        }
        catch (ArtifactNotFoundException e)
        {
            getLog().debug(e.getMessage(), e);

            StringBuffer msg = new StringBuffer();
            msg.append("An error occurred during dependency resolution.\n\n");
            msg.append("    Failed to retrieve " + e.getDownloadUrl() + "\n");
            msg.append("from the following repositories:");
            for (Iterator repositories = e.getRemoteRepositories().iterator(); repositories.hasNext();)
            {
                ArtifactRepository repository = (ArtifactRepository) repositories.next();
                msg.append("\n    " + repository.getId() + "(" + repository.getUrl() + ")");
            }
            msg.append("\nCaused by: " + e.getMessage());

            getLog().warn(msg);
        }
        catch (ArtifactResolutionException e)
        {
            getLog().debug(e.getMessage(), e);

            StringBuffer msg = new StringBuffer();
            msg.append("An error occurred during dependency resolution of the following artifact:\n\n");
            msg.append("    " + e.getGroupId() + ":" + e.getArtifactId() + e.getVersion() + "\n\n");
            msg.append("Caused by: " + e.getMessage());

            getLog().warn(msg);
        }
    }

    /*
    * @todo we need a more permanent feature that does this properly
    */
    protected String getPluginSetting(String artifactId, String optionName, String defaultValue)
    {
        for (Iterator it = executedProject.getBuildPlugins().iterator(); it.hasNext();)
        {
            Plugin plugin = (Plugin) it.next();
            if (plugin.getArtifactId().equals(artifactId))
            {
                Xpp3Dom o = (Xpp3Dom) plugin.getConfiguration();
                if (o != null && o.getChild(optionName) != null)
                {
                    return o.getChild(optionName).getValue();
                }
            }
        }
        return defaultValue;
    }

    protected Set getProjectArtifacts()
    {
        Set artifacts = new HashSet();

        for (Iterator dependencies = executedProject.getDependencies().iterator(); dependencies.hasNext();)
        {
            Dependency dep = (Dependency) dependencies.next();

            String groupId = dep.getGroupId();
            String artifactId = dep.getArtifactId();
            VersionRange versionRange = VersionRange.createFromVersion(dep.getVersion());
            String type = dep.getType();
            if (type == null)
            {
                type = "jar";
            }
            String classifier = dep.getClassifier();
            boolean optional = dep.isOptional();
            String scope = dep.getScope();
            if (scope == null)
            {
                scope = Artifact.SCOPE_COMPILE;
            }

            Artifact artifact = artifactFactory.createDependencyArtifact(groupId, artifactId, versionRange, type,
                    classifier, scope, optional);

            if (scope.equalsIgnoreCase(Artifact.SCOPE_SYSTEM))
            {
                artifact.setFile(new File(dep.getSystemPath()));
            }

            List exclusions = new ArrayList();
            for (Iterator j = dep.getExclusions().iterator(); j.hasNext();)
            {
                Exclusion e = (Exclusion) j.next();
                exclusions.add(e.getGroupId() + ":" + e.getArtifactId());
            }

            ArtifactFilter newFilter = new ExcludesArtifactFilter(exclusions);

            artifact.setDependencyFilter(newFilter);

            artifacts.add(artifact);
        }

        return artifacts;
    }

    private Map createManagedVersionMap(ArtifactFactory artifactFactory, String projectId,
                                        DependencyManagement dependencyManagement)
            throws ProjectBuildingException
    {
        Map map;
        if (dependencyManagement != null && dependencyManagement.getDependencies() != null)
        {
            map = new HashMap();
            for (Iterator i = dependencyManagement.getDependencies().iterator(); i.hasNext();)
            {
                Dependency d = (Dependency) i.next();

                try
                {
                    VersionRange versionRange = VersionRange.createFromVersionSpec(d.getVersion());
                    Artifact artifact = artifactFactory.createDependencyArtifact(d.getGroupId(), d.getArtifactId(),
                            versionRange, d.getType(),
                            d.getClassifier(), d.getScope(),
                            d.isOptional());
                    map.put(d.getManagementKey(), artifact);
                }
                catch (InvalidVersionSpecificationException e)
                {
                    throw new ProjectBuildingException(projectId, "Unable to parse version '" + d.getVersion() +
                            "' for dependency '" + d.getManagementKey() + "': " + e.getMessage(), e);
                }
            }
        } else
        {
            map = Collections.EMPTY_MAP;
        }
        return map;
    }

    public Log getLog()
    {
        if (log == null)
        {
            log = super.getLog();
        }

        return log;
    }

    protected void initAtlassianVariables()
    {
        atlassianProperties = executedProject.getProperties();
        appParams = atlassianProperties.getProperty("atlassian.idea.application.params");
        appParams = (appParams == null) ? "-Xms128m -Xmx256m" : appParams;
        appPort = atlassianProperties.getProperty("atlassian.idea.application.port");
        appPort = (appPort == null) ? "8080" : appPort;

    }

    protected void replace(File file, String regex, String replacement) throws MojoExecutionException
    {

        try
        {
            if (replacement == null)
            {
                replacement = "";
            }
            File tempFile = File.createTempFile("temp", "temp");
            BufferedReader reader = new BufferedReader(new FileReader(file));
            BufferedWriter writer = new BufferedWriter(new FileWriter(tempFile));

            while (true)
            {
                String line = reader.readLine();

                if (line == null)
                {
                    break;
                }

                line = line.replaceAll(regex, replacement);
                writer.write(line);
                writer.newLine();
            }

            writer.close();
            reader.close();
            file.delete();
            tempFile.renameTo(file);
        } catch (IOException e)
        {
            throw new MojoExecutionException("failed to read/write to file");
        }
    }

    protected void modifyResinEnvModule(File dir, boolean isResin3) throws MojoExecutionException
    {
        String imlFile;
        String regex;
        String property;
        String moduleXml;
        String resinLocation;
        if (isResin3)
        {
            getLog().info("Creating Resin 3 Environment...");
            regex = "\\$\\{atlassian.idea.resin3.location\\}";
            imlFile = "resin3_env.iml";
            resinLocation = "atlassian.idea.resin3.location";
            moduleXml = "resin3_env-module";
        } else
        {
            getLog().info("Creating Resin Environment...");
            regex = "\\$\\{atlassian.idea.resin.location\\}";
            imlFile = "resin_env.iml";
            resinLocation = "atlassian.idea.resin.location";
            moduleXml = "resin_env-module";
        }

        property = invertSlashes(atlassianProperties.getProperty(resinLocation));
        dir.mkdirs();
        File moduleFile = new File(dir, imlFile);
        String resourceName = moduleXml + ".xml";

        Document document = readXmlDocument(resourceName);

        Element module = document.getRootElement();

        addJARS(module, resinLocation);
        //replace variables specified within the xml file
        replaceURLElement(module, regex, property);
        replaceJavaVersion(module, moduleFile);
        replaceArtifactId(module);

        writeXmlDocument(moduleFile, document);


    }

    private String moduleName(File imlFile)
    {
        String name = imlFile.getName();
        /* chop the suffix */
        String moduleName = name.substring(0, name.length() - 4);
        return moduleName;
    }

    protected void replaceJavaVersion(Element module, File moduleFile)
    {
        String moduleName = moduleName(moduleFile);
        /* Check if there is a jdk for that specific module */
        String jdkName;
        if (atlassianProperties.containsKey("atlassian.idea." + moduleName + ".jdk.name"))
        {
            jdkName = atlassianProperties.getProperty("atlassian.idea." + moduleName + ".jdk.name");
            getLog().info("JDK Name Variable Found - " + moduleName + " used JDK Name \"" + jdkName + "\" instead of Default.");

        } else
        {
            jdkName = atlassianProperties.getProperty("atlassian.idea.jdk.name");
            if (!(atlassianProperties.containsKey("atlassian.idea.jdk.name") && (jdkName.length() > 0)))
            {
                jdkName = System.getProperties().getProperty("java.version");
                jdkName = jdkName.substring(0, 3);
            }
        }

        Element component = findComponent(module, "NewModuleRootManager");
        findElement(component, "orderEntry", "type", "jdk").addAttribute("jdkName", jdkName);
    }

    protected void replaceURLElement(Element module, String regex, String property)
    {
        Element component = findElement(module, "component", "NewModuleRootManager");
        Element resinLibraries = findElement(component, "orderEntry", "type", "module-library").element("library").element("CLASSES");
        for (Iterator children = resinLibraries.elementIterator("root"); children.hasNext();)
        {
            Element child = (Element) children.next();
            String replaceStr = child.attributeValue("url").replaceFirst(regex, property);
            child.addAttribute("url", replaceStr);
        }


    }

    protected void modifyIpr(String modulePath) throws MojoExecutionException
    {
        File projectFile = new File(executedProject.getBasedir(), executedProject.getArtifactId() + ".ipr");
        Document document = readXmlDocument(projectFile);

        Element module = document.getRootElement();
        Element component = findComponent(module, "ProjectModuleManager");
        Element modules = findElement(component, "modules");

        /* Add Atlassian specific modules */
        Element m = findModule(modules, modulePath);


        writeXmlDocument(projectFile, document);


    }

    protected void createOptions(Element conf, String mainClassName, String vmParam, String programParam, String workingDir, String moduleName)
    {
        findElement(conf, "option", "MAIN_CLASS_NAME")
                .addAttribute("value", mainClassName);
        findElement(conf, "option", "VM_PARAMETERS")
                .addAttribute("value", vmParam);
        findElement(conf, "option", "PROGRAM_PARAMETERS")
                .addAttribute("value", programParam);
        findElement(conf, "option", "WORKING_DIRECTORY")
                .addAttribute("value", workingDir);
        findElement(conf, "module", moduleName);
    }

    protected void replaceArtifactId(Element module) throws MojoExecutionException
    {

        if (atlassianProperties.getProperty("appserver.module.dependency") != null)
        {
            Element component = findElement(module, "component", "NewModuleRootManager");
            removeElement(component, "orderEntry", "type", "module");
            String dependency = atlassianProperties.getProperty("appserver.module.dependency");
            addModuleDependency(component, dependency);
        } else
        {
            throw new MojoExecutionException("appserver.module.dependency variable not set\n"
                    + "Please set appserver.module.dependency variable to a module that contains "
                    + "the needed classes for webapp to run");
        }
    }

    protected void addWebAppModules(Element module) throws MojoExecutionException
    {

        if (atlassianProperties.getProperty("appserver.module.dependency") != null)
        {
            Element component = findElement(module, "component", "NewModuleRootManager");
            String dependency = atlassianProperties.getProperty("appserver.module.dependency");
            if (dependency.equals(CONFLUENCE_APPSERVER_MODULE) && executedProject.getArtifactId().equals(CONFLUENCE_APPSERVER_MODULE))
            {
                addModuleDependency(component, "confluencerpc");
                addModuleDependency(component, "functestrpc");
                addModuleDependency(component, "confluence");
            }
        } /*else
        {
            throw new MojoExecutionException("appserver.module.dependency variable not set\n"
                    + "Please set appserver.module.dependency variable to a module that contains "
                    + "the needed classes for webapp to run");
        }*/
    }

    private void addModuleDependency(Element element, String module)
    {
        //check if it exists
        for (Iterator children = element.elementIterator("orderEntry"); children.hasNext();)
        {
            Element child = (Element) children.next();
            if ("module".equals(child.attributeValue("type")) && module.equals(child.attributeValue("module-name")))
            {
                return;
            }
        }

        //if it doesnt exist add it.
        createElement(element, "orderEntry")
                .addAttribute("type", "module")
                .addAttribute("module-name", module);
    }

    protected void modifyEnvModule(String location, File resourceFile, File imlFile) throws MojoExecutionException
    {
        String regex = "\\$\\{" + location + "\\}";
        String property = invertSlashes(atlassianProperties.getProperty(location));

        Document document = readXmlDocument(resourceFile.getName());
        Element module = document.getRootElement();
        addJARS(module, location);
        replaceJavaVersion(module, imlFile);
        replaceURLElement(module, regex, property);
        replaceArtifactId(module);
        writeXmlDocument(imlFile, document);


    }

    private void addJARS(Element root, String propertyName)
    {
        Element component = findComponent(root, "NewModuleRootManager");
        Element orderEntry = findElement(component, "orderEntry", "type", "module-library");

        if (propertyName.equals("atlassian.idea.tomcat.location"))
        {
            Element tomcatLibrary = findElement(orderEntry, "library", "name", "Tomcat Libraries");
            Element classes = findElement(tomcatLibrary, "CLASSES");
            String location = atlassianProperties.getProperty(propertyName);

            setupTomcatLibraries(location, classes);
        }
        if (propertyName.equals("atlassian.idea.resin.location"))
        {
            Element resinLibrary = findElement(orderEntry, "library", "name", "Resin Libraries");
            Element classes = findElement(resinLibrary, "CLASSES");
            String location = atlassianProperties.getProperty(propertyName);

            setupResinLibaries(propertyName, location, classes);
        }
        if (propertyName.equals("atlassian.idea.resin3.location"))
        {
            Element resinLibrary = findElement(orderEntry, "library", "name", "Resin 3 Libraries");
            Element classes = findElement(resinLibrary, "CLASSES");
            String location = atlassianProperties.getProperty(propertyName);

            setupResinLibaries(propertyName, location, classes);
        }
    }

    private void setupResinLibaries(String propertyName, String location, Element classes)
    {
        File lib = new File(location, File.separator + "lib");
        String[] jars = lib.list();
        for (int i = 0; i < jars.length; i++)
        {
            String jar = jars[i];
            findElement(classes, "root", "url", "jar://${" + propertyName + "}/lib/" + jar + "!/");
        }
    }

    private void setupTomcatLibraries
            (String
                    location, Element
                    classes)
    {
        File lib = new File(location, File.separator + "common" + File.separator + "lib");
        String[] jars = lib.list();
        for (int i = 0; i < jars.length; i++)
        {
            String jar = jars[i];
            findElement(classes, "root", "url", "jar://${atlassian.idea.tomcat.location}/common/lib/" + jar + "!/");
        }

        lib = new File(location, File.separator + "bin");
        jars = lib.list();
        for (int i = 0; i < jars.length; i++)
        {
            String jar = jars[i];
            if (jar.substring(jar.length() - 3).equals("jar"))
            {
                findElement(classes, "root", "url", "jar://${atlassian.idea.tomcat.location}/bin/" + jar + "!/");
            }
        }

        lib = new File(location, File.separator + "server" + File.separator + "lib");
        jars = lib.list();
        for (int i = 0; i < jars.length; i++)
        {
            String jar = jars[i];
            if (jar.substring(jar.length() - 3).equals("jar"))
            {
                findElement(classes, "root", "url", "jar://${atlassian.idea.tomcat.location}/server/lib/" + jar + "!/");
            }
        }

    }

    protected void callMavenIdea()
            throws MojoExecutionException
    {

        // Call the mvn idea:idea once at the start
        if (executedProject.isExecutionRoot())
        {

            // Execute idea:idea command
            Commandline cl = new Commandline("mvn");
            cl.createArgument().setValue("idea:idea");
            cl.createArgument().setValue("-Pidea");
            // set jdkName to atlassian.idea.jdk.name if it is not null
            if (executedProject.getProperties().containsKey("atlassian.idea.jdk.name"))
            {
                String jdkName = executedProject.getProperties().getProperty("atlassian.idea.jdk.name");
                if (jdkName != null)
                {
                    cl.createArgument().setValue("-DjdkName=" + jdkName);
                }
            }
            // pass the systemEnvironment eg. -DjdkName=<version>
            try
            {
                cl.addSystemEnvironment();
            } catch (Exception e)
            {
                getLog().error("System Failed to load System Environment for $ mvn idea:idea");
                e.printStackTrace();  //To change body of catch statement use File | Settings | File Templates.
            }

            // Get the input stream and read from it
            StreamConsumer stdOut = new StreamConsumer()
            {
                public void consumeLine(String line)
                {
                    System.out.println("    " + line);
                }
            };
            StreamConsumer stdErr = new StreamConsumer()
            {
                public void consumeLine(String line)
                {
                    System.err.println("    " + line);
                    ;
                }
            };

            try
            {
                getLog().info("Executing: [" + cl.toString() + "]");

                int result = CommandLineUtils.executeCommandLine(cl, stdOut, stdErr);

                if (result != 0)
                {
                    throw new MojoExecutionException("Maven execution failed, exit code: \'" + result + "\'" + stdOut.toString() + stdErr.toString());
                }
            }
            catch (CommandLineException e)
            {
                throw new MojoExecutionException("Can't run goal ");// + goals, stdOut.toString(), stdErr.toString(), e);
            }

            ideaHasBeenRun = true;
        }
    }
}
