package com.atlassian.maven.plugins;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.dom4j.Document;
import org.dom4j.Element;

import java.io.File;

/**
 * @author Atlassian
 * @goal tomcat
 * @execute phase="generate-sources"
 */
public class Tomcat
        extends AbstractIdeaMojo
{

    public void execute() throws MojoExecutionException, MojoFailureException
    {
        try
        {
            doDependencyResolution(executedProject, localRepo);
        }
        catch (Exception e)
        {
            throw new MojoExecutionException("Unable to build project dependencies.", e);
        }

        build();
    }

    public void build()
            throws MojoExecutionException
    {
        initAtlassianVariables();

        if (executedProject.isExecutionRoot())
        {

            String tomcatLocation = atlassianProperties.getProperty("atlassian.idea.tomcat.location");
            if (tomcatLocation != null)
            {
                File tomcat = new File(tomcatLocation + "/bin/bootstrap.jar");

                if (tomcat.exists())
                {
                    modifyIpr("$PROJECT_DIR$/modules/tomcat_env/tomcat_env.iml");
                    modifyIws();
                    generateModulesForApps();
                    createTomcatXml();
                } else
                {
                    getLog().error("Unable to find tomcat jar in " + tomcat.getAbsolutePath());
                }
            }
        }
    }


    public void modifyIws() throws MojoExecutionException
    {
        File workspaceFile = new File(executedProject.getBasedir(), executedProject.getArtifactId() + ".iws");
        Document document = readXmlDocument(workspaceFile);

        Element module = document.getRootElement();
        Element runManager = findComponent(module, "RunManager");
        Element conf = findElement(runManager, "configuration", "Tomcat")
                .addAttribute("type", "Application")
                .addAttribute("default", "false")
                .addAttribute("selected", "false")
                .addAttribute("factoryName", "Application");
        String mainClassName = "org.apache.catalina.startup.Bootstrap";
        String tomcatLocation = atlassianProperties.getProperty("atlassian.idea.tomcat.location");
        String vmParameters = "-Dcatalina.home=\"" + tomcatLocation + "\" -Dcatalina.base=\"tomcat\" -Djava.util.logging.manager=org.apache.juli.ClassLoaderLogManager " + appParams;
        String programParameters = "start";
        String workingDirectory = "file://$PROJECT_DIR$";
        String moduleName = "tomcat_env";
        createOptions(conf, mainClassName, vmParameters, programParameters, workingDirectory, moduleName);

        writeXmlDocument(workspaceFile, document);


    }


    void generateModulesForApps() throws MojoExecutionException
    {
        getLog().info("Creating Tomcat Environment...");
        File modulePath = new File(executedProject.getBasedir(), "/modules/tomcat_env");
        File resourceFile = new File(modulePath, "tomcat_env-module.xml");
        File imlFile = new File(modulePath, "tomcat_env.iml");
        modulePath.mkdirs();
        modifyEnvModule("atlassian.idea.tomcat.location", resourceFile, imlFile);
    }


    private void createTomcatXml()
    {
        File baseDir = executedProject.getBasedir();
        File tomcatDir = new File(baseDir, "/tomcat/conf/");
        tomcatDir.mkdirs();

        String resourceName = "tomcat-server.xml";
        try
        {
            File tomcatServerXml = new File(tomcatDir, "/server.xml");

            Document document = readXmlDocument(resourceName);

            Element root = document.getRootElement();
            configDefaultCompiler(root);

            root.element("Service").element("Connector").addAttribute("port", appPort);


            String contextPath = atlassianProperties.getProperty("atlassian.idea.application.webapp.contextpath");
            contextPath = contextPath == null ? "" : "/" + contextPath;
            String mavenWarSrc = atlassianProperties.getProperty("maven.war.src");
            Element context = root.element("Service").element("Engine").element("Host").element("Context");
            context.addAttribute("path", contextPath);
            if (mavenWarSrc != null)
            {
                context.addAttribute("docBase",  executedProject.getBasedir() + "/" + mavenWarSrc);
            } else
            {
                throw new MojoExecutionException("maven.war.src variable not set\n"
                    + "Please set maven.war.src variable your webapp directory (relative to the project root)");
            }

            resourceRef(context);
            writeXmlDocument(tomcatServerXml, document);
            /*Copy tomcat-web.xml file*/
            document = readXmlDocument("tomcat-web.xml");
            File tomcatWebXml = new File(tomcatDir, "/web.xml");
            writeXmlDocument(tomcatWebXml, document);


        } catch (MojoExecutionException e)
        {
            e.printStackTrace();  //To change body of catch statement use File | Settings | File Templates.
        }
    }

    private void resourceRef(Element root)
    {
        if (atlassianProperties.containsKey("atlassian.idea.application.db.jndiname"))
        {
            String jndiName = atlassianProperties.getProperty("atlassian.idea.application.db.jndiname");
            String driverName = atlassianProperties.getProperty("atlassian.idea.application.db.drivername");
            String url = atlassianProperties.getProperty("atlassian.idea.application.db.url");
            String username = atlassianProperties.getProperty("atlassian.idea.application.db.username");
            String password = atlassianProperties.getProperty("atlassian.idea.application.db.password");

            Element resource = findElement(root, "resource-ref", "type", "javax.sql.DataSource")
                    .addAttribute("name", jndiName)
                    .addAttribute("username", username)
                    .addAttribute("password", password)
                    .addAttribute("driverClassName", driverName)
                    .addAttribute("url", url);

        }
    }

    private void configDefaultCompiler(Element module)
    {
        Element option = findElement(module, "java")
                .addAttribute("compiler-args", "");
        if (atlassianProperties.containsKey("atlassian.idea.jikespath"))
        {
            String jikesPath = atlassianProperties.getProperty("atlassian.idea.jikespath");
            File jikes = new File(jikesPath);

            if (jikes.exists())
            {
                option.addAttribute("compiler", jikesPath);
                return;
            }


        }
        option.addAttribute("compiler", "jikes");

    }
}
