package com.atlassian.maven.plugins;

/*
 * Copyright 2005-2006 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import org.apache.maven.artifact.factory.ArtifactFactory;
import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.artifact.resolver.ArtifactResolver;
import org.apache.maven.artifact.metadata.ArtifactMetadataSource;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;
import org.dom4j.Element;
import org.dom4j.Document;

import java.io.File;
import java.util.Set;
import java.util.Properties;

/**
 * @author Edwin Punzalan
 * @goal project
 * @execute phase="generate-sources"
 */
public class IdeaProjectMojo
        extends AbstractIdeaMojo
{
    /**
     * Specify the name of the registered IDEA JDK to use
     * for the project.
     *
     * @parameter expression="${jdkName}"
     */
    private String jdkName;

    /**
     * Specify the version of the JDK to use for the project for the purpose of enabled assertions and 5.0 language features.
     * The default value is the specification version of the executing JVM.
     *
     * @parameter expression="${jdkLevel}"
     * @todo would be good to use the compilation source if possible
     */
    private String jdkLevel;

    /**
     * Specify the resource pattern in wildcard format, for example "?*.xml;?*.properties".
     * Currently supports 4.x and 5.x.
     * The default value is any file without a java extension ("!?*.java").
     * Because IDEA doesn't distinguish between source and resources directories, this is needed.
     * Please note that the default value includes package.html files as it's not possible to exclude those.
     *
     * @parameter  expression="${wildcardResourcePatterns}" default-value="?*.properties;?*.xml;?*.xsl;?*.html;?*.dtd;?*.xsd;?*.tld;?*.gif;?*.png;?*.jpeg;?*.jpg;?*.vm;?*.css;?*.js"
     */
    private String wildcardResourcePatterns;

    /**
     * Specify the version of idea to use.  This is needed to identify the default formatting of
     * project-jdk-name used by idea.  Currently supports 4.x and 5.x.
     * <p/>
     * This will only be used when parameter jdkName is not set.
     *
     * @parameter expression="${ideaVersion}"
     * default-value="5.x"
     */
    private String ideaVersion;

    private Set macros;

    private Properties atlassianProperties;

    public void initParam(MavenProject project, ArtifactFactory artifactFactory, ArtifactRepository localRepo,
                          ArtifactResolver artifactResolver, ArtifactMetadataSource artifactMetadataSource, Log log,
                          boolean overwrite, String jdkName, String jdkLevel, String wildcardResourcePatterns,
                          String ideaVersion, Set macros)
    {
        super.initParam(project, artifactFactory, localRepo, artifactResolver, artifactMetadataSource, log,
                overwrite);

        this.jdkName = jdkName;

        this.jdkLevel = jdkLevel;

        this.wildcardResourcePatterns = wildcardResourcePatterns;

        this.ideaVersion = ideaVersion;

        this.macros = macros;
    }

    /**
     * Create IDEA (.ipr) project files.
     *
     * @throws org.apache.maven.plugin.MojoExecutionException
     *
     */
    public void execute()
            throws MojoExecutionException
    {
        try
        {
            doDependencyResolution(executedProject, localRepo);
        }
        catch (Exception e)
        {
            throw new MojoExecutionException("Unable to build project dependencies.", e);
        }

        rewriteProject();
    }

    public void rewriteProject()
            throws MojoExecutionException
    {
        File projectFile = new File(executedProject.getBasedir(), executedProject.getArtifactId() + ".ipr");
        atlassianProperties = executedProject.getProperties();
        Document document = readXmlDocument(projectFile);

        Element module = document.getRootElement();


        Element component = findComponent(module, "ProjectModuleManager");
        Element modules = findElement(component, "modules");

        /* Add Atlassian specific modules */
        addAtlassianModules(modules);

        /* Default Compiler Config */
        configDefaultCompiler(module);


        writeXmlDocument(projectFile, document);


    }

    private void configDefaultCompiler(Element module)
    {
        Element component = findComponent(module, "CompilerConfiguration");
        Element option = findElement(component, "option", "DEFAULT_COMPILER");

        if (atlassianProperties.containsKey("atlassian.idea.jikespath"))
        {
            System.out.println("atlassianProperties = " + atlassianProperties.getProperty("atlassian.idea.jikespath"));
            File jikes = new File(atlassianProperties.getProperty("atlassian.idea.jikespath"));

            if (jikes.exists())
            {
                option.addAttribute("value", "Jikes");
            }

        } else
        {
            option.addAttribute("value", "Javac");
        }
    }


    private void addAtlassianModules(Element modules)
    {

        findModule(modules, "$PROJECT_DIR$/modules/test_env/test_env.iml");


        File funcTests = new File(executedProject.getBasedir(), "subprojects/func_tests");
        if (funcTests.exists())
        {
            findModule(modules, "$PROJECT_DIR$/modules/func_test_env/func_test_env.iml");
        }

    }
}
